/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.properties;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Base64;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.nifi.properties.ProtectedNiFiProperties;
import org.apache.nifi.properties.SensitivePropertyProtectionException;
import org.apache.nifi.properties.SensitivePropertyProviderFactory;
import org.apache.nifi.properties.StandardSensitivePropertyProviderFactory;
import org.apache.nifi.util.NiFiBootstrapUtils;
import org.apache.nifi.util.NiFiProperties;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NiFiPropertiesLoader {
    private static final Logger logger = LoggerFactory.getLogger(NiFiPropertiesLoader.class);
    private static final Base64.Encoder KEY_ENCODER = Base64.getEncoder().withoutPadding();
    private static final int SENSITIVE_PROPERTIES_KEY_LENGTH = 24;
    private static final String EMPTY_SENSITIVE_PROPERTIES_KEY = String.format("%s=", "nifi.sensitive.props.key");
    private static final String MIGRATION_INSTRUCTIONS = "See Admin Guide section [Updating the Sensitive Properties Key]";
    private static final String PROPERTIES_KEY_MESSAGE = String.format("Sensitive Properties Key [%s] not found: %s", "nifi.sensitive.props.key", "See Admin Guide section [Updating the Sensitive Properties Key]");
    private final String defaultPropertiesFilePath = NiFiBootstrapUtils.getDefaultApplicationPropertiesFilePath();
    private NiFiProperties instance;
    private String keyHex;
    private SensitivePropertyProviderFactory sensitivePropertyProviderFactory;

    public static NiFiPropertiesLoader withKey(String keyHex) {
        NiFiPropertiesLoader loader = new NiFiPropertiesLoader();
        loader.setKeyHex(keyHex);
        return loader;
    }

    public void setKeyHex(String keyHex) {
        if (this.keyHex != null && !this.keyHex.trim().isEmpty()) {
            throw new RuntimeException("Cannot overwrite an existing key");
        }
        this.keyHex = keyHex;
    }

    public static NiFiProperties loadDefaultWithKeyFromBootstrap() throws IOException {
        try {
            return new NiFiPropertiesLoader().loadDefault();
        }
        catch (Exception e) {
            logger.warn("Encountered an error naively loading the nifi.properties file because one or more properties are protected: {}", (Object)e.getLocalizedMessage());
            throw e;
        }
    }

    private NiFiProperties loadDefault() {
        return this.load(this.defaultPropertiesFilePath);
    }

    private SensitivePropertyProviderFactory getSensitivePropertyProviderFactory() {
        if (this.sensitivePropertyProviderFactory == null) {
            this.sensitivePropertyProviderFactory = StandardSensitivePropertyProviderFactory.withKey((String)this.keyHex);
        }
        return this.sensitivePropertyProviderFactory;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    ProtectedNiFiProperties readProtectedPropertiesFromDisk(File file) {
        if (file == null || !file.exists() || !file.canRead()) {
            String path = file == null ? "missing file" : file.getAbsolutePath();
            logger.error("Cannot read from '{}' -- file is missing or not readable", (Object)path);
            throw new IllegalArgumentException("NiFi properties file missing or unreadable");
        }
        Properties rawProperties = new Properties();
        try (BufferedInputStream inputStream = new BufferedInputStream(new FileInputStream(file));){
            rawProperties.load(inputStream);
            logger.info("Loaded {} properties from {}", (Object)rawProperties.size(), (Object)file.getAbsolutePath());
            Set<String> keys = rawProperties.stringPropertyNames();
            for (String key : keys) {
                String property = rawProperties.getProperty(key);
                rawProperties.setProperty(key, property.trim());
            }
            Object object = new ProtectedNiFiProperties(rawProperties);
            return object;
        }
        catch (Exception ex) {
            logger.error("Cannot load properties file due to {}", (Object)ex.getLocalizedMessage());
            throw new RuntimeException("Cannot load properties file due to " + ex.getLocalizedMessage(), ex);
        }
    }

    public NiFiProperties load(File file) {
        ProtectedNiFiProperties protectedNiFiProperties = this.readProtectedPropertiesFromDisk(file);
        if (protectedNiFiProperties.hasProtectedKeys()) {
            Security.addProvider((Provider)new BouncyCastleProvider());
            this.getSensitivePropertyProviderFactory().getSupportedSensitivePropertyProviders().forEach(protectedNiFiProperties::addSensitivePropertyProvider);
        }
        return protectedNiFiProperties.getUnprotectedProperties();
    }

    public NiFiProperties load(String path) {
        if (path != null && !path.trim().isEmpty()) {
            return this.load(new File(path));
        }
        return this.loadDefault();
    }

    public NiFiProperties get() {
        if (this.instance == null) {
            this.instance = this.getDefaultProperties();
        }
        return this.instance;
    }

    private NiFiProperties getDefaultProperties() {
        NiFiProperties defaultProperties = this.loadDefault();
        if (NiFiPropertiesLoader.isKeyGenerationRequired(defaultProperties)) {
            if (defaultProperties.isClustered()) {
                logger.error("Clustered Configuration Found: Shared Sensitive Properties Key [{}] required for cluster nodes", (Object)"nifi.sensitive.props.key");
                throw new SensitivePropertyProtectionException(PROPERTIES_KEY_MESSAGE);
            }
            File flowConfiguration = defaultProperties.getFlowConfigurationFile();
            if (flowConfiguration.exists()) {
                logger.error("Flow Configuration [{}] Found: Migration Required for blank Sensitive Properties Key [{}]", (Object)flowConfiguration, (Object)"nifi.sensitive.props.key");
                throw new SensitivePropertyProtectionException(PROPERTIES_KEY_MESSAGE);
            }
            this.setSensitivePropertiesKey();
            defaultProperties = this.loadDefault();
        }
        return defaultProperties;
    }

    private void setSensitivePropertiesKey() {
        logger.warn("Generating Random Sensitive Properties Key [{}]", (Object)"nifi.sensitive.props.key");
        SecureRandom secureRandom = new SecureRandom();
        byte[] sensitivePropertiesKeyBinary = new byte[24];
        secureRandom.nextBytes(sensitivePropertiesKeyBinary);
        String sensitivePropertiesKey = KEY_ENCODER.encodeToString(sensitivePropertiesKeyBinary);
        try {
            File niFiPropertiesFile = new File(this.defaultPropertiesFilePath);
            Path niFiPropertiesPath = Paths.get(niFiPropertiesFile.toURI());
            List<String> lines = Files.readAllLines(niFiPropertiesPath);
            List updatedLines = lines.stream().map(line -> {
                if (line.equals(EMPTY_SENSITIVE_PROPERTIES_KEY)) {
                    return line + sensitivePropertiesKey;
                }
                return line;
            }).collect(Collectors.toList());
            Files.write(niFiPropertiesPath, updatedLines, new OpenOption[0]);
            logger.info("NiFi Properties [{}] updated with Sensitive Properties Key", (Object)niFiPropertiesPath);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to set Sensitive Properties Key", e);
        }
    }

    private static boolean isKeyGenerationRequired(NiFiProperties properties) {
        String configuredSensitivePropertiesKey = properties.getProperty("nifi.sensitive.props.key");
        return configuredSensitivePropertiesKey == null || configuredSensitivePropertiesKey.length() == 0;
    }
}

