/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.util.crypto;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.security.util.EncryptionMethod;
import org.apache.nifi.security.util.crypto.AESKeyedCipherProvider;
import org.apache.nifi.security.util.crypto.CipherUtility;
import org.apache.nifi.security.util.crypto.KeyedCipherProvider;
import org.apache.nifi.security.util.crypto.RandomIVPBECipherProvider;
import org.mindrot.jbcrypt.BCrypt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BcryptCipherProvider
extends RandomIVPBECipherProvider {
    private static final Logger logger = LoggerFactory.getLogger(BcryptCipherProvider.class);
    private final int workFactor;
    private static final int DEFAULT_WORK_FACTOR = 12;
    private static final int DEFAULT_SALT_LENGTH = 16;
    private static final Pattern BCRYPT_SALT_FORMAT = Pattern.compile("^\\$\\d\\w\\$\\d{2}\\$[\\w\\/\\.]{22}");

    public BcryptCipherProvider() {
        this(12);
    }

    public BcryptCipherProvider(int workFactor) {
        this.workFactor = workFactor;
        if (workFactor < 12) {
            logger.warn("The provided work factor {} is below the recommended minimum {}", (Object)workFactor, (Object)12);
        }
    }

    @Override
    public Cipher getCipher(EncryptionMethod encryptionMethod, String password, byte[] salt, byte[] iv, int keyLength, boolean encryptMode) throws Exception {
        try {
            return this.getInitializedCipher(encryptionMethod, password, salt, iv, keyLength, encryptMode);
        }
        catch (IllegalArgumentException e) {
            throw e;
        }
        catch (Exception e) {
            throw new ProcessException("Error initializing the cipher", (Throwable)e);
        }
    }

    @Override
    Logger getLogger() {
        return logger;
    }

    @Override
    public Cipher getCipher(EncryptionMethod encryptionMethod, String password, byte[] salt, int keyLength, boolean encryptMode) throws Exception {
        return this.getCipher(encryptionMethod, password, salt, new byte[0], keyLength, encryptMode);
    }

    protected Cipher getInitializedCipher(EncryptionMethod encryptionMethod, String password, byte[] salt, byte[] iv, int keyLength, boolean encryptMode) throws Exception {
        if (encryptionMethod == null) {
            throw new IllegalArgumentException("The encryption method must be specified");
        }
        if (!encryptionMethod.isCompatibleWithStrongKDFs()) {
            throw new IllegalArgumentException(encryptionMethod.name() + " is not compatible with Bcrypt");
        }
        if (StringUtils.isEmpty((CharSequence)password)) {
            throw new IllegalArgumentException("Encryption with an empty password is not supported");
        }
        String algorithm = encryptionMethod.getAlgorithm();
        String provider = encryptionMethod.getProvider();
        String cipherName = CipherUtility.parseCipherFromAlgorithm(algorithm);
        if (!CipherUtility.isValidKeyLength(keyLength, cipherName)) {
            throw new IllegalArgumentException(String.valueOf(keyLength) + " is not a valid key length for " + cipherName);
        }
        String bcryptSalt = this.formatSaltForBcrypt(salt);
        String hash = BCrypt.hashpw((String)password, (String)bcryptSalt);
        MessageDigest digest = MessageDigest.getInstance("SHA-512", provider);
        byte[] dk = digest.digest(hash.getBytes(StandardCharsets.UTF_8));
        dk = Arrays.copyOf(dk, keyLength / 8);
        SecretKeySpec tempKey = new SecretKeySpec(dk, algorithm);
        AESKeyedCipherProvider keyedCipherProvider = new AESKeyedCipherProvider();
        return ((KeyedCipherProvider)keyedCipherProvider).getCipher(encryptionMethod, tempKey, iv, encryptMode);
    }

    private String formatSaltForBcrypt(byte[] salt) {
        if (salt == null || salt.length == 0) {
            throw new IllegalArgumentException("The salt cannot be empty. To generate a salt, use BcryptCipherProvider#generateSalt()");
        }
        String rawSalt = new String(salt, StandardCharsets.UTF_8);
        Matcher matcher = BCRYPT_SALT_FORMAT.matcher(rawSalt);
        if (matcher.find()) {
            return rawSalt;
        }
        throw new IllegalArgumentException("The salt must be of the format $2a$10$gUVbkVzp79H8YaCOsCVZNu. To generate a salt, use BcryptCipherProvider#generateSalt()");
    }

    @Override
    public byte[] generateSalt() {
        return BCrypt.gensalt((int)this.workFactor).getBytes(StandardCharsets.UTF_8);
    }

    @Override
    public int getDefaultSaltLength() {
        return 16;
    }

    protected int getWorkFactor() {
        return this.workFactor;
    }
}

