/*
 * Created on Oct 23, 2010
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 * 
 * Copyright @2010-2011 the original author or authors.
 */
package org.assertj.core.api;

import java.util.Comparator;

import org.assertj.core.internal.Characters;
import org.assertj.core.internal.ComparatorBasedComparisonStrategy;
import org.assertj.core.util.VisibleForTesting;

/**
 * Base class for all implementations of assertions for {@link Character}s.
 * 
 * @param <S> the "self" type of this assertion class. Please read &quot;<a href="http://bit.ly/anMa4g"
 *          target="_blank">Emulating 'self types' using Java Generics to simplify fluent API implementation</a>&quot;
 *          for more details.
 * 
 * @author Yvonne Wang
 * @author David DIDIER
 * @author Ansgar Konermann
 * @author Alex Ruiz
 * @author Joel Costigliola
 * @author Mikhail Mazursky
 */
public abstract class AbstractCharacterAssert<S extends AbstractCharacterAssert<S>> extends
    AbstractComparableAssert<S, Character> {

  @VisibleForTesting
  Characters characters = Characters.instance();

  protected AbstractCharacterAssert(Character actual, Class<?> selfType) {
    super(actual, selfType);
  }

  /**
   * Verifies that the actual value is equal to the given one.
   * 
   * @param expected the given value to compare the actual value to.
   * @return {@code this} assertion object.
   * @throws AssertionError if the actual value is {@code null}.
   * @throws AssertionError if the actual value is not equal to the given one.
   */
  public S isEqualTo(char expected) {
    characters.assertEqual(info, actual, expected);
    return myself;
  }

  /**
   * Verifies that the actual value is not equal to the given one.
   * 
   * @param other the given value to compare the actual value to.
   * @return {@code this} assertion object.
   * @throws AssertionError if the actual value is {@code null}.
   * @throws AssertionError if the actual value is equal to the given one.
   */
  public S isNotEqualTo(char other) {
    characters.assertNotEqual(info, actual, other);
    return myself;
  }

  /**
   * Verifies that the actual value is less than the given one.
   * <p>
   * Example:
   * 
   * <pre>
   * // assertion will pass
   * assertThat('A').isLessThan('a');
   * assertThat('a').isLessThan('b');
   *
   * // assertion will fail
   * assertThat('a').isLessThan('A');
   * assertThat('b').isLessThan('a');
   * </pre>
   * 
   * </p>
   * 
   * @param other the given value to compare the actual value to.
   * @return {@code this} assertion object.
   * @throws AssertionError if the actual value is {@code null}.
   * @throws AssertionError if the actual value is equal to or greater than the given one.
   */
  public S isLessThan(char other) {
    characters.assertLessThan(info, actual, other);
    return myself;
  }

  /**
   * Verifies that the actual value is less than or equal to the given one.
   * <p>
   * Example:
   * 
   * <pre>
   * // assertion will pass
   * assertThat('A').isLessThanOrEqualTo('a');
   * assertThat('A').isLessThanOrEqualTo('A');
   *
   * // assertion will fail
   * assertThat('b').isLessThanOrEqualTo('a');
   * </pre>
   * 
   * </p>
   * 
   * @param other the given value to compare the actual value to.
   * @return {@code this} assertion object.
   * @throws AssertionError if the actual value is {@code null}.
   * @throws AssertionError if the actual value is greater than the given one.
   */
  public S isLessThanOrEqualTo(char other) {
    characters.assertLessThanOrEqualTo(info, actual, other);
    return myself;
  }

  /**
   * Verifies that the actual value is greater than the given one.
   * <p>
   * Example:
   * 
   * <pre>
   * // assertion will pass
   * assertThat('a').isGreaterThan('A');
   * assertThat('b').isGreaterThan('a');
   *
   * // assertion will fail
   * assertThat('A').isGreaterThan('a');
   * assertThat('a').isGreaterThan('b');
   * </pre>
   * 
   * </p>
   * 
   * @param other the given value to compare the actual value to.
   * @return {@code this} assertion object.
   * @throws AssertionError if the actual value is {@code null}.
   * @throws AssertionError if the actual value is equal to or less than the given one.
   */
  public S isGreaterThan(char other) {
    characters.assertGreaterThan(info, actual, other);
    return myself;
  }

  /**
   * Verifies that the actual value is greater than or equal to the given one.
   * <p>
   * Example:
   * 
   * <pre>
   * // assertion will pass
   * assertThat('A').isGreaterThanOrEqualTo('A');
   * assertThat('b').isGreaterThanOrEqualTo('a');
   * 
   * // assertion will fail
   * assertThat('a').isGreaterThanOrEqualTo('b');
   * </pre>
   * 
   * </p>
   * 
   * @param other the given value to compare the actual value to.
   * @return {@code this} assertion object.
   * @throws AssertionError if the actual value is {@code null}.
   * @throws AssertionError if the actual value is less than the given one.
   */
  public S isGreaterThanOrEqualTo(char other) {
    characters.assertGreaterThanOrEqualTo(info, actual, other);
    return myself;
  }

  /**
   * Verifies that the actual value is a lowercase character.
   * <p>
   * Example:
   *
   * <pre>
   * // assertion will pass
   * assertThat('a').isLowerCase();
   *
   * // assertion will fail
   * assertThat('A').isLowerCase();
   * </pre>
   *
   * </p>
   * 
   * @return {@code this} assertion object.
   * @throws AssertionError if the actual value is {@code null}.
   * @throws AssertionError if the actual value is not a lowercase character.
   */
  public S isLowerCase() {
    characters.assertLowerCase(info, actual);
    return myself;
  }

  /**
   * Verifies that the actual value is a uppercase character.
   * <p>
   * Example:
   * 
   * <pre>
   * // assertion will pass
   * assertThat('A').isLowerCase();
   * 
   * // assertion will fail
   * assertThat('a').isLowerCase();
   * </pre>
   * 
   * </p>
   * 
   * @return {@code this} assertion object.
   * @throws AssertionError if the actual value is {@code null}.
   * @throws AssertionError if the actual value is not a uppercase character.
   */
  public S isUpperCase() {
    characters.assertUpperCase(info, actual);
    return myself;
  }

  @Override
  public S usingComparator(Comparator<? super Character> customComparator) {
    super.usingComparator(customComparator);
    this.characters = new Characters(new ComparatorBasedComparisonStrategy(customComparator));
    return myself;
  }

  @Override
  public S usingDefaultComparator() {
    super.usingDefaultComparator();
    this.characters = Characters.instance();
    return myself;
  }
}
