/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.security;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.concurrent.ExecutionException;
import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import org.apache.cassandra.config.TransparentDataEncryptionOptions;
import org.apache.cassandra.security.KeyProvider;
import org.cassandraunit.shaded.com.google.common.annotations.VisibleForTesting;
import org.cassandraunit.shaded.com.google.common.cache.CacheBuilder;
import org.cassandraunit.shaded.com.google.common.cache.CacheLoader;
import org.cassandraunit.shaded.com.google.common.cache.LoadingCache;
import org.cassandraunit.shaded.com.google.common.cache.RemovalListener;
import org.cassandraunit.shaded.com.google.common.cache.RemovalNotification;
import org.cassandraunit.shaded.io.netty.util.concurrent.FastThreadLocal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CipherFactory {
    private final Logger logger = LoggerFactory.getLogger(CipherFactory.class);
    private static final FastThreadLocal<CachedCipher> cipherThreadLocal = new FastThreadLocal();
    private final SecureRandom secureRandom;
    private final LoadingCache<String, Key> cache;
    private final int ivLength;
    private final KeyProvider keyProvider;

    public CipherFactory(TransparentDataEncryptionOptions options) {
        this.logger.info("initializing CipherFactory");
        this.ivLength = options.iv_length;
        try {
            this.secureRandom = SecureRandom.getInstance("SHA1PRNG");
            Class<?> keyProviderClass = Class.forName(options.key_provider.class_name);
            Constructor<?> ctor = keyProviderClass.getConstructor(TransparentDataEncryptionOptions.class);
            this.keyProvider = (KeyProvider)ctor.newInstance(options);
        }
        catch (Exception e) {
            throw new RuntimeException("couldn't load cipher factory", e);
        }
        this.cache = CacheBuilder.newBuilder().maximumSize(64L).concurrencyLevel(Runtime.getRuntime().availableProcessors()).removalListener(new RemovalListener<String, Key>(){

            @Override
            public void onRemoval(RemovalNotification<String, Key> notice) {
                CipherFactory.this.logger.info("key {} removed from cipher key cache", notice.getKey());
            }
        }).build(new CacheLoader<String, Key>(){

            @Override
            public Key load(String alias) throws Exception {
                CipherFactory.this.logger.info("loading secret key for alias {}", (Object)alias);
                return CipherFactory.this.keyProvider.getSecretKey(alias);
            }
        });
    }

    public Cipher getEncryptor(String transformation, String keyAlias) throws IOException {
        byte[] iv = new byte[this.ivLength];
        this.secureRandom.nextBytes(iv);
        return this.buildCipher(transformation, keyAlias, iv, 1);
    }

    public Cipher getDecryptor(String transformation, String keyAlias, byte[] iv) throws IOException {
        assert (iv != null && iv.length > 0) : "trying to decrypt, but the initialization vector is empty";
        return this.buildCipher(transformation, keyAlias, iv, 2);
    }

    @VisibleForTesting
    Cipher buildCipher(String transformation, String keyAlias, byte[] iv, int cipherMode) throws IOException {
        try {
            CachedCipher cachedCipher = cipherThreadLocal.get();
            if (cachedCipher != null) {
                Cipher cipher = cachedCipher.cipher;
                if (cachedCipher.mode == cipherMode && cipher.getAlgorithm().equals(transformation) && cachedCipher.keyAlias.equals(keyAlias) && Arrays.equals(cipher.getIV(), iv)) {
                    return cipher;
                }
            }
            Key key = this.retrieveKey(keyAlias);
            Cipher cipher = Cipher.getInstance(transformation);
            cipher.init(cipherMode, key, new IvParameterSpec(iv));
            cipherThreadLocal.set(new CachedCipher(cipherMode, keyAlias, cipher));
            return cipher;
        }
        catch (InvalidAlgorithmParameterException | InvalidKeyException | NoSuchAlgorithmException | NoSuchPaddingException e) {
            this.logger.error("could not build cipher", (Throwable)e);
            throw new IOException("cannot load cipher", e);
        }
    }

    private Key retrieveKey(String keyAlias) throws IOException {
        try {
            return this.cache.get(keyAlias);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof IOException) {
                throw (IOException)e.getCause();
            }
            throw new IOException("failed to load key from cache: " + keyAlias, e);
        }
    }

    private static class CachedCipher {
        public final int mode;
        public final String keyAlias;
        public final Cipher cipher;

        private CachedCipher(int mode, String keyAlias, Cipher cipher) {
            this.mode = mode;
            this.keyAlias = keyAlias;
            this.cipher = cipher;
        }
    }
}

