/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.asic.common;

import eu.europa.esig.dss.asic.common.ASiCParameters;
import eu.europa.esig.dss.enumerations.ASiCContainerType;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.InMemoryDocument;
import eu.europa.esig.dss.model.MimeType;
import eu.europa.esig.dss.utils.Utils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ASiCUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ASiCUtils.class);
    public static final String MIME_TYPE = "mimetype";
    public static final String MIME_TYPE_COMMENT = "mimetype=";
    public static final String META_INF_FOLDER = "META-INF/";
    public static final String PACKAGE_ZIP = "package.zip";
    private static final long ZIP_ENTRY_THRESHOLD = 1000000L;
    private static final long ZIP_ENTRY_RATIO = 50L;
    private static final int MAX_MALFORMED_FILES = 100;

    private ASiCUtils() {
    }

    public static boolean isSignature(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains("signature") && !entryName.contains("Manifest");
    }

    public static String getMimeTypeString(ASiCParameters asicParameters) {
        String asicParameterMimeType = asicParameters.getMimeType();
        String mimeTypeString = Utils.isStringBlank((String)asicParameterMimeType) ? (ASiCUtils.isASiCE(asicParameters) ? MimeType.ASICE.getMimeTypeString() : MimeType.ASICS.getMimeTypeString()) : asicParameterMimeType;
        return mimeTypeString;
    }

    public static boolean isASiCMimeType(MimeType asicMimeType) {
        return MimeType.ASICS.equals((Object)asicMimeType) || MimeType.ASICE.equals((Object)asicMimeType) || MimeType.ODT.equals((Object)asicMimeType) || MimeType.ODS.equals((Object)asicMimeType);
    }

    public static ASiCContainerType getASiCContainerType(MimeType asicMimeType) {
        if (MimeType.ASICS.equals((Object)asicMimeType)) {
            return ASiCContainerType.ASiC_S;
        }
        if (MimeType.ASICE.equals((Object)asicMimeType) || MimeType.ODT.equals((Object)asicMimeType) || MimeType.ODS.equals((Object)asicMimeType)) {
            return ASiCContainerType.ASiC_E;
        }
        throw new IllegalArgumentException("Not allowed mimetype " + asicMimeType);
    }

    public static boolean isASiCE(ASiCParameters asicParameters) {
        return ASiCContainerType.ASiC_E.equals((Object)asicParameters.getContainerType());
    }

    public static boolean isASiCS(ASiCParameters asicParameters) {
        return ASiCContainerType.ASiC_S.equals((Object)asicParameters.getContainerType());
    }

    public static MimeType getMimeType(ASiCParameters asicParameters) {
        return ASiCUtils.isASiCE(asicParameters) ? MimeType.ASICE : MimeType.ASICS;
    }

    public static boolean isArchiveContainsCorrectSignatureFileWithExtension(DSSDocument toSignDocument, String extension) {
        List<String> filenames = ASiCUtils.getFileNames(toSignDocument);
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename) || !filename.endsWith(extension)) continue;
            return true;
        }
        return false;
    }

    public static boolean isArchive(List<DSSDocument> docs) {
        if (Utils.collectionSize(docs) == 1) {
            return ASiCUtils.isASiCContainer(docs.get(0));
        }
        return false;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean isASiCContainer(DSSDocument dssDocument) {
        if (dssDocument == null) {
            return false;
        }
        byte[] preamble = new byte[2];
        try (InputStream is = dssDocument.openStream();){
            int r = is.read(preamble, 0, 2);
            if (r != 2) {
                boolean bl = false;
                return bl;
            }
        }
        catch (IOException e) {
            throw new DSSException("Unable to read the 2 first bytes", (Throwable)e);
        }
        if (preamble[0] != 80) return false;
        if (preamble[1] != 75) return false;
        return true;
    }

    public static boolean isXAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(".xml");
    }

    public static boolean isCAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(".p7s");
    }

    public static boolean isOpenDocument(DSSDocument mimeTypeDocument) {
        MimeType mimeType = ASiCUtils.getMimeType(mimeTypeDocument);
        return MimeType.ODS == mimeType || MimeType.ODT == mimeType;
    }

    public static MimeType getMimeType(DSSDocument mimeTypeDocument) {
        MimeType mimeType;
        block9: {
            if (mimeTypeDocument == null) {
                return null;
            }
            InputStream is = mimeTypeDocument.openStream();
            try {
                byte[] byteArray = Utils.toByteArray((InputStream)is);
                String mimeTypeString = new String(byteArray, "UTF-8");
                mimeType = MimeType.fromMimeTypeString((String)mimeTypeString);
                if (is == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new DSSException((Throwable)e);
                }
            }
            is.close();
        }
        return mimeType;
    }

    public static ASiCContainerType getContainerType(DSSDocument archive, DSSDocument mimetype, String zipComment, List<DSSDocument> signedDocuments) {
        ASiCContainerType containerType = ASiCUtils.getContainerTypeFromMimeType(archive.getMimeType());
        if (containerType == null && (containerType = ASiCUtils.getContainerTypeFromMimeTypeDocument(mimetype)) == null) {
            containerType = ASiCUtils.getContainerTypeFromZipComment(zipComment);
        }
        if (containerType == null) {
            LOG.warn("Unable to define the ASiC Container type with its properties");
            if (Utils.collectionSize(signedDocuments) == 1) {
                containerType = ASiCContainerType.ASiC_S;
            } else if (Utils.collectionSize(signedDocuments) > 1) {
                containerType = ASiCContainerType.ASiC_E;
            } else {
                throw new DSSException("The provided file does not contain signed documents. The signature validation is not possible");
            }
        }
        return containerType;
    }

    private static ASiCContainerType getContainerTypeFromZipComment(String zipComment) {
        int indexOf;
        if (Utils.isStringNotBlank((String)zipComment) && (indexOf = zipComment.indexOf(MIME_TYPE_COMMENT)) > -1) {
            String asicCommentMimeTypeString = zipComment.substring(MIME_TYPE_COMMENT.length() + indexOf);
            MimeType mimeTypeFromZipComment = MimeType.fromMimeTypeString((String)asicCommentMimeTypeString);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromZipComment);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeTypeDocument(DSSDocument mimetype) {
        if (mimetype != null) {
            MimeType mimeTypeFromEmbeddedFile = ASiCUtils.getMimeType(mimetype);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromEmbeddedFile);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeType(MimeType mimeType) {
        if (ASiCUtils.isASiCMimeType(mimeType)) {
            return ASiCUtils.getASiCContainerType(mimeType);
        }
        return null;
    }

    public static String getPadNumber(int num) {
        String numStr = String.valueOf(num);
        String zeroPad = "000";
        return zeroPad.substring(numStr.length()) + numStr;
    }

    public static boolean isAsic(List<DSSDocument> documents) {
        if (ASiCUtils.isArchive(documents)) {
            DSSDocument archive = documents.get(0);
            boolean cades = ASiCUtils.isArchiveContainsCorrectSignatureFileWithExtension(archive, "p7s");
            boolean xades = ASiCUtils.isArchiveContainsCorrectSignatureFileWithExtension(archive, "xml");
            return cades || xades;
        }
        return false;
    }

    public static boolean isArchiveManifest(String fileName) {
        return fileName.contains("ASiCArchiveManifest") && fileName.endsWith(".xml");
    }

    public static void secureCopy(InputStream is, OutputStream os, long containerSize) throws IOException {
        int nRead;
        byte[] data = new byte[2048];
        int byteCounter = 0;
        long allowedSize = containerSize * 50L;
        while ((nRead = is.read(data)) != -1) {
            if ((long)(byteCounter += nRead) > 1000000L && (long)byteCounter > allowedSize) {
                throw new DSSException("Zip Bomb detected in the ZIP container. Validation is interrupted.");
            }
            os.write(data, 0, nRead);
        }
    }

    public static List<String> getFileNames(DSSDocument archive) {
        ArrayList<String> filenames = new ArrayList<String>();
        try (InputStream is = archive.openStream();
             ZipInputStream zis = new ZipInputStream(is);){
            ZipEntry entry;
            while ((entry = ASiCUtils.getNextValidEntry(zis)) != null) {
                filenames.add(entry.getName());
            }
        }
        catch (IOException e) {
            throw new DSSException("Unable to retrieve filenames", (Throwable)e);
        }
        return filenames;
    }

    public static ZipEntry getNextValidEntry(ZipInputStream zis) {
        for (int counter = 0; counter < 100; ++counter) {
            try {
                return zis.getNextEntry();
            }
            catch (Exception e) {
                LOG.warn("ZIP container contains a malformed, corrupted or not accessible entry! The entry is skipped. Reason: [{}]", (Object)e.getMessage());
                ASiCUtils.closeEntry(zis);
                continue;
            }
        }
        throw new DSSException("Unable to retrieve a valid ZipEntry (100 tries)");
    }

    public static DSSDocument getCurrentDocument(String filepath, ZipInputStream zis, long containerSize) throws IOException {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            ASiCUtils.secureCopy(zis, baos, containerSize);
            baos.flush();
            InMemoryDocument inMemoryDocument = new InMemoryDocument(baos.toByteArray(), filepath);
            return inMemoryDocument;
        }
    }

    public static void closeEntry(ZipInputStream zis) {
        try {
            zis.closeEntry();
        }
        catch (IOException e) {
            throw new DSSException("Unable to close entry", (Throwable)e);
        }
    }
}

