/*******************************************************************************
 * Copyright (c) 2021 Eclipse RDF4J contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Distribution License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *******************************************************************************/
package org.eclipse.rdf4j.sail.lmdb.config;

import org.eclipse.rdf4j.sail.Sail;
import org.eclipse.rdf4j.sail.config.SailConfigException;
import org.eclipse.rdf4j.sail.config.SailFactory;
import org.eclipse.rdf4j.sail.config.SailImplConfig;
import org.eclipse.rdf4j.sail.lmdb.LmdbStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link SailFactory} that creates {@link LmdbStore}s based on RDF configuration data.
 */
public class LmdbStoreFactory implements SailFactory {

	private static final Logger logger = LoggerFactory.getLogger(LmdbStoreFactory.class);

	/**
	 * The type of repositories that are created by this factory.
	 *
	 * @see SailFactory#getSailType()
	 */
	public static final String SAIL_TYPE = "rdf4j:LmdbStore";

	/**
	 * Returns the Sail's type: <tt>rdf4j:LmdbStore</tt>.
	 */
	@Override
	public String getSailType() {
		return SAIL_TYPE;
	}

	@Override
	public SailImplConfig getConfig() {
		return new LmdbStoreConfig();
	}

	@Override
	public Sail getSail(SailImplConfig config) throws SailConfigException {
		if (!SAIL_TYPE.equals(config.getType())) {
			throw new SailConfigException("Invalid Sail type: " + config.getType());
		}

		if (config instanceof LmdbStoreConfig) {
			return new LmdbStore(((LmdbStoreConfig) config));
		} else {
			logger.warn("Config is instance of {} is not LmdbStoreConfig.", config.getClass().getName());
			return new LmdbStore();
		}
	}
}
