package org.infinispan.marshall;

import net.jcip.annotations.ThreadSafe;
import org.infinispan.io.ByteBuffer;

import java.io.IOException;

/**
 * A marshaller is a class that is able to marshall and unmarshall objects efficiently.
 * <p/>
 * This interface is used to marshall {@link org.infinispan.commands.ReplicableCommand}s, their parameters and their
 * response values, as well as any other arbitraty Object <--> byte[] conversions, such as those used in client/server
 * communications.
 * <p/>
 * The interface is also used by the {@link org.infinispan.loaders.CacheStore} framework to efficiently serialize data
 * to be persisted, as well as the {@link org.infinispan.statetransfer.StateTransferManager} when serializing the cache
 * for transferring state en-masse.
 * <p/>
 * A single instance of any implementation is shared by multiple threads, so implementations <i>need</i> to be threadsafe,
 * and preferably immutable.
 *
 * @author Manik Surtani
 * @version 4.1
 */
@ThreadSafe
public interface Marshaller {

   /**
    * Marshalls an object to a byte array.  The estimatedSize parameter is a hint that can be passed in to allow for
    * efficient sizing of the byte array before attempting to marshall the object.  The more accurate this estimate is,
    * the less likely byte[]s will need to be resized to hold the byte stream generated by marshalling the object.
    *
    * @param obj           object to convert to a byte array.  Must not be null.
    * @param estimatedSize an estimate of how large the resulting byte array may be
    * @return a byte array
    * @throws IOException
    */
   byte[] objectToByteBuffer(Object obj, int estimatedSize) throws IOException;

   /**
    * Marshalls an object to a byte array.
    *
    * @param obj object to convert to a byte array.  Must not be null.
    * @return a byte array
    * @throws IOException
    */
   byte[] objectToByteBuffer(Object obj) throws IOException;

   /**
    * Unmarshalls an object from a byte array.
    *
    * @param buf byte array containing the binary representation of an object.  Must not be null.
    * @return an object
    * @throws IOException
    * @throws ClassNotFoundException
    */
   Object objectFromByteBuffer(byte[] buf) throws IOException, ClassNotFoundException;

   /**
    * Unmarshalls an object from a specific portion of a byte array.
    *
    * @param buf    byte array containing the binary representation of an object.  Must not be null.
    * @param offset point in buffer to start reading
    * @param length number of bytes to consider
    * @return an object
    * @throws IOException
    * @throws ClassNotFoundException
    */
   Object objectFromByteBuffer(byte[] buf, int offset, int length) throws IOException, ClassNotFoundException;

   /**
    * A method that returns an instance of {@link org.infinispan.io.ByteBuffer}, which allows direct access to the byte
    * array with minimal array copying
    *
    * @param o object to marshall
    * @return a ByteBuffer
    * @throws Exception
    */
   ByteBuffer objectToBuffer(Object o) throws IOException;

   /**
    * A method that checks whether the given object is marshallable as per the rules of this marshaller.
    * 
    * @param o object to verify whether it's marshallable or not
    * @return true if the object is marshallable, otherwise false
    */
   boolean isMarshallable(Object o);
}

