/*
 * Decompiled with CFR 0.152.
 */
package org.iternine.jeppetto.testsupport.db;

import com.mongodb.CommandResult;
import com.mongodb.DB;
import com.mongodb.Mongo;
import com.mongodb.MongoException;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.dbunit.database.IDatabaseConnection;
import org.iternine.jeppetto.testsupport.db.Database;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class MongoDatabase
extends Database {
    private static final Logger logger = LoggerFactory.getLogger(MongoDatabase.class);
    private static File mongod;
    private String mongoDbName;
    private Process mongoProcess;
    private final int mongoDbPort;

    public static MongoDatabase forPlatform(int port) {
        String platform = System.getProperty("os.name", "unknown").toLowerCase();
        if ("linux".equals(platform)) {
            return new LinuxOrMacMongoDatabase(platform, port).initDatabase();
        }
        if ("mac os x".equals(platform)) {
            return new LinuxOrMacMongoDatabase("mac", port).initDatabase();
        }
        throw new RuntimeException("Unknown platform for mongodb test context: " + platform);
    }

    public MongoDatabase(int port) {
        super(null);
        this.mongoDbPort = port;
    }

    public void setMongoDbName(String mongoDbName) {
        this.mongoDbName = mongoDbName;
    }

    @Override
    public void close() {
        if (this.mongoDbName == null) {
            return;
        }
        try {
            Mongo mongo = new Mongo("127.0.0.1", this.mongoDbPort);
            DB db = mongo.getDB(this.mongoDbName);
            db.resetError();
            db.dropDatabase();
            CommandResult err = db.getLastError();
            if (err != null && err.get("err") != null) {
                logger.error("Could not drop database {}: {}", (Object)this.mongoDbName, (Object)err);
            }
            mongo.dropDatabase(this.mongoDbName);
            if (mongo.getDatabaseNames().contains(this.mongoDbName)) {
                logger.error("Database {} will not go away!", (Object)this.mongoDbName);
            }
        }
        catch (UnknownHostException e) {
        }
        catch (MongoException e) {
            logger.warn("Could not drop database {}: {}", (Object)this.mongoDbName, (Object)e.getMessage());
        }
    }

    @Override
    protected void onNewIDatabaseConnection(IDatabaseConnection connection) {
        throw new UnsupportedOperationException("MongoDatabase does not support new IDatabaseConnections.");
    }

    protected abstract String getPlatform();

    protected abstract void makeExecutable(File var1);

    protected abstract ProcessBuilder createMongoProcess(File var1, File var2, int var3) throws IOException;

    protected MongoDatabase initDatabase() {
        if (this.alreadyRunning(this.mongoDbPort)) {
            logger.debug("Mongo already running, using existing server.");
            return this;
        }
        return this.downloadExtractAndStartMongo();
    }

    private int getMongoDbWebPort() {
        return this.mongoDbPort + 1000;
    }

    private boolean alreadyRunning(int port) {
        try {
            Socket socket = new Socket("localhost", port);
            socket.close();
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    private MongoDatabase downloadExtractAndStartMongo() {
        try {
            File dbpath = this.createDataDir();
            ProcessBuilder processBuilder = this.createMongoProcess(this.findMongod(), dbpath, this.mongoDbPort).redirectErrorStream(true);
            this.mongoProcess = processBuilder.start();
            Thread readerDaemon = new Thread(new Runnable(){

                @Override
                public void run() {
                    try {
                        String line;
                        BufferedReader reader = new BufferedReader(new InputStreamReader(MongoDatabase.this.mongoProcess.getInputStream()));
                        while ((line = reader.readLine()) != null) {
                            logger.debug(line);
                        }
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                    logger.debug("Mongod out consumer daemon exit.");
                }
            });
            readerDaemon.setDaemon(true);
            readerDaemon.start();
            if (!this.waitForMongo(this.getMongoDbWebPort())) {
                try {
                    throw new RuntimeException("Mongod process aborted with code " + this.mongoProcess.exitValue());
                }
                catch (IllegalThreadStateException e) {
                    throw new RuntimeException("Mongod process still running but it doesn't seem to be running normally.");
                }
            }
            logger.debug("Mongod seems to be running: http://localhost:{}", (Object)this.getMongoDbWebPort());
        }
        catch (Exception e) {
            this.close();
            throw new RuntimeException(e);
        }
        return this;
    }

    private File findMongod() throws IOException {
        if (mongod != null) {
            return mongod;
        }
        File mongoArchive = this.findMongoArchive();
        if (!mongoArchive.exists()) {
            throw new RuntimeException("Mongo archive not found at: " + mongoArchive.getAbsolutePath());
        }
        logger.debug(String.format("Extracting mongod from %s%n", mongoArchive.getAbsolutePath()));
        ZipFile zip = new ZipFile(mongoArchive);
        ZipEntry mongodEntry = zip.getEntry("bin/mongod");
        InputStream input = zip.getInputStream(mongodEntry);
        mongod = new File(System.getProperty("java.io.tmpdir", "/tmp"), "mongod-bin/mongod");
        if (!mongod.getParentFile().mkdirs()) {
            logger.warn("Could not create parent dir for mongod, either a permission issue or dir was left behind by previous process.");
        }
        this.copy(input, mongod);
        zip.close();
        logger.debug(String.format("Copied mongod binary to %s%n", mongod.getAbsolutePath()));
        return mongod;
    }

    private File findMongoArchive() {
        String mavenLocalRepo = System.getProperty("maven.repo.local");
        String actualRepo = mavenLocalRepo == null || mavenLocalRepo.isEmpty() || mavenLocalRepo.contains("$") ? System.getProperty("user.home") + "/.m2/repository" : mavenLocalRepo;
        return new File(String.format("%3$s/org/mongodb/mongod-binary/%1$s/mongod-binary-%1$s-%2$s.zip", this.getMongoVersion(), this.getPlatform(), actualRepo));
    }

    private File createDataDir() {
        File tmpDir = new File(System.getProperty("java.io.tmpdir", "/tmp"));
        File datadir = new File(tmpDir, "mongodb-data");
        if (!datadir.mkdirs()) {
            logger.warn("Could not create data dir {}. Either a permissions issue, or the data dir was left behind by a previous process.", (Object)datadir.getAbsolutePath());
        }
        return datadir;
    }

    private String getMongoVersion() {
        Properties mongoProps = new Properties();
        try {
            mongoProps.load(this.getClass().getClassLoader().getResourceAsStream("mongo.properties"));
            return mongoProps.getProperty("mongo.version");
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private boolean waitForMongo(int port) throws InterruptedException {
        int secondsToWait = 7;
        for (int i = 0; i < secondsToWait + 1; ++i) {
            if (this.alreadyRunning(port)) {
                return true;
            }
            if (i >= secondsToWait) continue;
            Thread.sleep(1000L);
        }
        return false;
    }

    private void copy(InputStream input, File output) throws IOException {
        int len;
        BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(output));
        byte[] buffer = new byte[10240];
        while ((len = input.read(buffer)) >= 0) {
            out.write(buffer, 0, len);
        }
        input.close();
        out.close();
        this.makeExecutable(output);
    }

    private static class LinuxOrMacMongoDatabase
    extends MongoDatabase {
        private String platform;

        public LinuxOrMacMongoDatabase(String platform, int port) {
            super(port);
            this.platform = platform;
        }

        @Override
        protected String getPlatform() {
            return this.platform;
        }

        @Override
        protected void makeExecutable(File file) {
            ProcessBuilder pb = new ProcessBuilder("chmod", "+x", file.getAbsolutePath());
            try {
                Process p = pb.start();
                int exit = p.waitFor();
                if (exit != 0) {
                    throw new RuntimeException("chmod of " + file.getAbsolutePath() + " returned " + exit);
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        protected ProcessBuilder createMongoProcess(File mongod, File dbpath, int port) throws IOException {
            return new ProcessBuilder(mongod.getAbsolutePath(), "--dbpath", dbpath.getAbsolutePath(), "--quiet", "--bind_ip", "localhost", "--smallfiles", "--noprealloc", "--port", Integer.toString(port));
        }
    }
}

