/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.ha.framework.interfaces;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Default implementation of FamilyClusterInfo
 *
 * @see org.jboss.ha.framework.interfaces.FamilyClusterInfo
 * @see org.jboss.ha.framework.interfaces.ClusteringTargetsRepository
 *
 * @author  <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 * @version $Revision: 74878 $
 */
public class FamilyClusterInfoImpl implements FamilyClusterInfo
{
   
   // Constants -----------------------------------------------------
   
   // Attributes ----------------------------------------------------
   
   private String familyName = null;
   private ArrayList targets = null;
   private long currentViewId = 0;
   private boolean isViewMembersInSyncWithViewId = false;
   
   private int cursor = FamilyClusterInfo.UNINITIALIZED_CURSOR;
   private Object arbitraryObject = null;
      
   // Static --------------------------------------------------------
    
   // Constructors --------------------------------------------------
   
   protected FamilyClusterInfoImpl (String familyName, List targets, long viewId)
   {
      this.familyName = familyName;
      this.targets = cloneList(targets);
      this.currentViewId = viewId;
      
      this.isViewMembersInSyncWithViewId = true;
   }

   // Public --------------------------------------------------------
   
   // FamilyClusterInfo implementation ----------------------------------------------
   
   public String getFamilyName () { return this.familyName; }
   
   /**
    * Returns an unmodifiable view of the target list.
    * 
    * @see Collections#unmodifiableList(List)
    */
   public synchronized List getTargets () 
   { 
      return Collections.unmodifiableList(this.targets); 
   }
   public long getCurrentViewId () { return this.currentViewId; }
   public int getCursor () { return this.cursor; }
   public int setCursor (int cursor) { return (this.cursor = cursor);}
   public Object getObject () { return this.arbitraryObject; }
   public Object setObject (Object whatever) { this.arbitraryObject = whatever; return this.arbitraryObject; }
   
   public List removeDeadTarget(Object target)
   {
      synchronized (this)
      {
         ArrayList tmp = (ArrayList) targets.clone();
         tmp.remove (target);
         this.targets = tmp;
         this.isViewMembersInSyncWithViewId = false;
         return Collections.unmodifiableList(this.targets);
      }      
   }
   
   public List updateClusterInfo (List targets, long viewId)
   {
      synchronized (this)
      {
         this.targets = cloneList(targets);
         this.currentViewId = viewId;
         this.isViewMembersInSyncWithViewId = true;
         return Collections.unmodifiableList(this.targets);
      }
   }
      
   public boolean currentMembershipInSyncWithViewId ()
   {
      return this.isViewMembersInSyncWithViewId;
   }
   
   public void resetView ()
   {
      synchronized (this)
      {
         this.currentViewId = -1;
         this.isViewMembersInSyncWithViewId = false;
      }
   }
      
   // Object overrides ---------------------------------------------------
   
   public int hashCode()
   {
      return this.familyName.hashCode ();
   }
   
   public boolean equals (Object o)
   {
      if (o instanceof FamilyClusterInfoImpl)
      {
         FamilyClusterInfoImpl fr = (FamilyClusterInfoImpl)o;
         return fr.familyName == this.familyName;
      }
      else
         return false;         
   }

   public String toString()
   {
      StringBuffer tmp = new StringBuffer(super.toString());
      tmp.append("{familyName=");
      tmp.append(familyName);
      tmp.append(",targets=");
      tmp.append(targets);
      tmp.append(",currentViewId=");
      tmp.append(currentViewId);
      tmp.append(",isViewMembersInSyncWithViewId=");
      tmp.append(isViewMembersInSyncWithViewId);
      tmp.append(",cursor=");
      tmp.append(cursor);
      tmp.append(",arbitraryObject=");
      tmp.append(arbitraryObject);
      tmp.append("}");
      return tmp.toString();
   }
   // Package protected ---------------------------------------------
   
   // Protected -----------------------------------------------------
   
   // Private -------------------------------------------------------
   
   // Inner classes -------------------------------------------------

   private static ArrayList cloneList(List toClone)
   {
      if (toClone instanceof ArrayList)
      {
      	 synchronized (toClone)
      	 {
         	return (ArrayList) ((ArrayList) toClone).clone();
         }
      }
      
      ArrayList clone = new ArrayList(toClone.size());
      synchronized (toClone)
      {
      	clone.addAll(toClone);
      }
      return clone;      
   }
}
