/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode;

/**
 * Used internally to determine the correct classloader for a class.
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public interface ClassLoaderFinder
{
   /**
    * Gets the correct classloader to use to find a class
    * 
    * @param initiating the initiating classloader
    * @param name the name of the class
    * @return the classloader to use to load the class, or null if the class does not exist
    */
   ClassLoader getLoaderForClass(ClassLoader initiating, String name);
   
   /**
    * Singleton default implementation of ClassLoaderFinder suitable for use
    * in simple/flat classloading environments
    */
   public class Default implements ClassLoaderFinder
   {
      public static Default INSTANCE = new Default();
      
      private Default()
      {
      }
      
      public ClassLoader getLoaderForClass(ClassLoader initiating, String name)
      {
         return initiating;
      }
   }
   
   /**
    * Singleton default implementation of ClassLoaderFinder suitable for use
    * in complex classloading environments. It loads the class
    */
   public class LoadClass implements ClassLoaderFinder
   {
      public static LoadClass INSTANCE = new LoadClass();
      
      private LoadClass()
      {
         
      }
      
      public ClassLoader getLoaderForClass(ClassLoader initiating, String name)
      {
         try
         {
            Class<?> clazz = SecurityActions.loadClass(initiating, name);;
            ClassLoader loader = SecurityActions.getClassLoader(clazz);
            if (loader == null)
               loader = SecurityActions.getSystemClassLoader();;
            return loader;
         }
         catch (ClassNotFoundException e)
         {
            if (name.endsWith(".package-info"))
               return null;
            // AutoGenerated
            throw new RuntimeException("Could not find '" + name + "' in classloader " + initiating);
         }
      }
   }
   
}
