/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.accessor.generated;

import java.lang.reflect.Method;

import javassist.Modifier;
import javassist.util.proxy.RuntimeSupport;

import org.jboss.reflect.plugins.bytecode.BytecodeFieldInfo;
import org.jboss.reflect.plugins.bytecode.accessor.FieldAccessor;

/**
 * <p>Class to create implementations of the {{@link FieldAccessor} interface.</p>
 * 
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class GeneratedFieldAccessorFactory extends GeneratedMemberAccessorFactory
{
   /** The names of the interfaces we are implementing */ 
   protected static final String[] interfaceNames;
   
   /** The methods from the interface that are being implemented */ 
   protected static final Method[] methods;

   /** The descriptors for the methods */
   private static final String[] methodDescriptors = new String[2];

   static
   {
      interfaceNames = new String[] {FieldAccessor.class.getName()};
      methods = new Method[2];
      try
      {
         methods[0] = SecurityActions.getDeclaredMethod(FieldAccessor.class, "get", Object.class);
         methods[1] = SecurityActions.getDeclaredMethod(FieldAccessor.class, "set", Object.class, Object.class);
      }
      catch (NoSuchMethodException e)
      {
         throw new RuntimeException(e);
      }
      
      methodDescriptors[0] = RuntimeSupport.makeDescriptor(methods[0]);
      methodDescriptors[1] = RuntimeSupport.makeDescriptor(methods[1]);
   }

   /** The field we are targeting */
   final BytecodeFieldInfo field;
   
   /** The name of the class being generated */
   final String className = FieldAccessor.class.getName() + counter.incrementAndGet();
   
   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param field the field we are generating a {@link FieldAccessor} for
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   GeneratedFieldAccessorFactory(Class<?> superClass, BytecodeFieldInfo field, boolean debug)
   {
      super(superClass, debug);
      this.field = field;
   }

   @Override
   String getGeneratedClassName()
   {
      return className;
   }

   @Override
   String[] getInterfaceNames()
   {
      return interfaceNames;
   }
   
   @Override
   String getAccessedMember()
   {
      return field.getDeclaringClass().getName() + "." + field.getName() + field.getDescriptor();
   }

   @Override
   String initAccessedMember()
   {
      return null;
   }

   @Override
   boolean implementMethod(int index, ClassFileWriterContext<?> cfwc)
   {
      if (index >= methods.length)
         return false;

      Method method = methods[index]; 

      cfwc.beginMethod(Modifier.PUBLIC, method.getName(), methodDescriptors[index], THROWABLE_EXCEPTIONS);

      boolean isStatic = Modifier.isStatic(field.getModifiers());

      int maxLocals = 0;
      if (index == 0)
      {
         //We need 2 local variable slots.
         //One for 'this' and one for the argument array
         //These are all object references and so take one slot each 
         maxLocals = 2;
         makeGetMethod(cfwc, isStatic);
      }
      else
      {
         //We need 3 local variable slots.
         //One for 'this', one for the target reference and one for the argument array.
         //These are all object references and so take one slot each 
         maxLocals = 3;
         makeSetMethod(cfwc, isStatic);
      }

      cfwc.endMethod(maxLocals);
      
      return true;
   }

   
   private void makeGetMethod(ClassFileWriterContext<?> cfwc, boolean isStatic)
   {
      if (isStatic)
      {
         cfwc.addGetStatic(ClassFileWriterContext.jvmClassName(field.getDeclaringClass().getName()), field.getName(), field.getDescriptor());
      }
      else
      {
         String targetClassName = ClassFileWriterContext.jvmClassName(field.getDeclaringClass().getName());
         //push and cast the target object
         cfwc.addAload(1);
         cfwc.addCheckcast(targetClassName);
         cfwc.addGetField(targetClassName, field.getName(), field.getDescriptor());
      }
      boxReturnValue(cfwc, field.getDescriptor());
      cfwc.addAReturn();
   }
   
   private void makeSetMethod(ClassFileWriterContext<?> cfwc, boolean isStatic)
   {
      if (!isStatic)
      {
         //push and cast the target object
         cfwc.addAload(1);
         cfwc.addCheckcast(ClassFileWriterContext.jvmClassName(field.getDeclaringClass().getName()));
      }

      //push and cast the value
      cfwc.addAload(2);
      castAndUnbox(cfwc, field.getDescriptor());
      
      if (isStatic)
      {
         cfwc.addPutStatic(ClassFileWriterContext.jvmClassName(field.getDeclaringClass().getName()), field.getName(), field.getDescriptor());
      }
      else
      {
         cfwc.addPutField(ClassFileWriterContext.jvmClassName(field.getDeclaringClass().getName()), field.getName(), field.getDescriptor());
      }

      cfwc.addAReturn();
   }
}