/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.accessor.generated;

import java.lang.reflect.Method;

import org.jboss.reflect.plugins.bytecode.BytecodeMethodInfo;
import org.jboss.reflect.plugins.bytecode.accessor.MethodAccessor;

import javassist.Modifier;
import javassist.util.proxy.RuntimeSupport;

/**
 * <p>Class to create implementations of the {@link MethodAccessor} interface.</p>
 * 
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class GeneratedMethodAccessorFactory extends GeneratedBehaviorAccessorFactory
{
   /** The names of the interfaces we are implementing */ 
   private static final String[] interfaceNames;
   
   /** The methods from the interface that are being implemented */ 
   private static final Method[] methods;
   
   /** The descriptors for the methods */
   private static final String[] methodDescriptors = new String[1];
   
   static
   {
      interfaceNames = new String[] {MethodAccessor.class.getName()};
      methods = new Method[1];
      try
      {
         methods[0] = SecurityActions.getDeclaredMethod(MethodAccessor.class, "invoke", Object.class, Object[].class);
      }
      catch (NoSuchMethodException e)
      {
         throw new RuntimeException(e);
      }
      methodDescriptors[0] = RuntimeSupport.makeDescriptor(methods[0]);
   }

   /** The name of the class being generated */
   private final String className = MethodAccessor.class.getName() + counter.incrementAndGet();
   
   
   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param method the constructor we are generating a {@link MethodAccessor} for
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   GeneratedMethodAccessorFactory(Class<?> superClass, BytecodeMethodInfo method, boolean debug)
   {
      super(superClass, method, debug);
   }

   String getGeneratedClassName()
   {
      return className;
   }
   

   String[] getInterfaceNames()
   {
      return interfaceNames;
   }

   /**
    * The return type of the method we are targeting
    * 
    * @return the target type
    */
   private String getReturnType()
   {
      String desc = getBehavior().getDescriptor();
      int i = desc.indexOf(')');
      return desc.substring(i +1);
   }

   @Override
   boolean implementMethod(int index, ClassFileWriterContext<?> cfwc)
   {
      if (index >= methods.length)
         return false;

      Method method = methods[index];
      
      cfwc.beginMethod(Modifier.PUBLIC, method.getName(), methodDescriptors[index], THROWABLE_EXCEPTIONS);
 
      boolean isStatic = Modifier.isStatic(behavior.getModifiers());
      
      String targetClassName = ClassFileWriterContext.jvmClassName(getBehavior().getDeclaringClass().getName());
      if (!isStatic)
      {
         //push and cast the target object
         cfwc.addAload(1);
         cfwc.addCheckcast(targetClassName);
      }
      
      addParameters(cfwc, getBehavior().getSignatureKey(), 2);
      
      if (isStatic)
         cfwc.addInvokeStatic(targetClassName, getBehavior().getName(), getBehavior().getDescriptor());
      else if (getBehavior().getDeclaringClass().isInterface())
      {
         cfwc.addInvokeInterface(targetClassName, getBehavior().getName(), getBehavior().getDescriptor(), countParameterStackSize(0, getBehavior().getSignatureKey()));
      }
      else
         cfwc.addInvokeVirtual(targetClassName, getBehavior().getName(), getBehavior().getDescriptor());
      
      if (getReturnType().equals("V"))
      {
         //return null
         cfwc.addAConstNull();
         cfwc.addAReturn();
      }
      else
      {
         boxReturnValue(cfwc, getReturnType());
         cfwc.addAReturn();
      }

      //We need 3 local variable slots.
      //One for 'this', one for the target reference and one for the argument array.
      //These are all object references and so take one slot each 

      //TODO the max stack
      cfwc.endMethod(3);

      return true;
   }
}