/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.accessor.reflect;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

import org.jboss.reflect.plugins.bytecode.BytecodeConstructorInfo;
import org.jboss.reflect.plugins.bytecode.BytecodeFieldInfo;
import org.jboss.reflect.plugins.bytecode.BytecodeMethodInfo;
import org.jboss.reflect.plugins.bytecode.accessor.ConstructorAccessor;
import org.jboss.reflect.plugins.bytecode.accessor.FieldAccessor;
import org.jboss.reflect.plugins.bytecode.accessor.MethodAccessor;
import org.jboss.reflect.spi.TypeInfo;

/**
 * <p>Class to create implementations of the {@link MethodAccessor}, {@link ConstructorAccessor} 
 * and {@link FieldAccessor} interfaces.</p>
 * <p>This implementation simply creates implementations that use reflection</p>
 * 
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class ReflectAccessorFactory
{
   private static final Class<?>[] NO_PARAMS = new Class[0];
   
   /**
    * Creates a new MethodAccessor implementation for a given method
    * 
    * @param methodInfo the method for which we want to create a MethodAccessor implementation
    * @return the implementing instance
    * @throws RuntimeException if an error ocurred
    */
   public static MethodAccessor createMethodAccessor(BytecodeMethodInfo methodInfo)
   {
      if (methodInfo == null)
         throw new IllegalArgumentException("Null method");
      
      try
      {
         //Ok to load the class here since this is only called by MethodInfo.invoke() in which
         //case the class must be loaded
         Class<?> clazz = methodInfo.getDeclaringClass().getType();
         Class<?>[] params = getParams(methodInfo.getParameterTypes());
         Method method = clazz.getDeclaredMethod(methodInfo.getName(), params); 
         
         return new ReflectMethodAccessor(method);
      }
      catch(RuntimeException e)
      {
         throw e;
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   /**
    * Creates a new ConstructorAccessor implementation for a given constructor
    * 
    * @param constructorInfo the constructor for which we want to create a ConstructorAccessor implementation
    * @return the implementing instance
    * @throws RuntimeException if an error ocurred
    */
   public static ConstructorAccessor createConstructorAccessor(BytecodeConstructorInfo constructorInfo)
   {
      if (constructorInfo == null)
         throw new IllegalArgumentException("Null constructor");

      try
      {
         //Ok to load the class here since this is only called by ConstructorInfo.newInstance() in which
         //case the class must be loaded
         Class<?> clazz = constructorInfo.getDeclaringClass().getType();
         Class<?>[] params = getParams(constructorInfo.getParameterTypes());
         Constructor<?> ctor = clazz.getDeclaredConstructor(params); 
         return new ReflectConstructorAccessor(ctor);
      }
      catch(RuntimeException e)
      {
         throw e;
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   /**
    * Creates a new FieldAccessor implementation for a given field
    * 
    * @param fieldInfo the field for which we want to create a JavassistField implementation
    * @return the implementing instance
    * @throws RuntimeException if an error ocurred
    */
   public static FieldAccessor createFieldAccessor(BytecodeFieldInfo fieldInfo)
   {
      if (fieldInfo == null)
         throw new IllegalArgumentException("Null field");

      try
      {
         //Ok to load the class here since this is only called by FieldInfo.get()/set() in which
         //case the class must be loaded
         Class<?> clazz = fieldInfo.getDeclaringClass().getType();
         Field field = clazz.getDeclaredField(fieldInfo.getName()); 
         return new ReflectFieldAccessor(field);
      }
      catch(RuntimeException e)
      {
         throw e;
      }
      catch(Exception e)
      {
         throw new RuntimeException(e);
      }
   }
   
   private static Class<?>[] getParams(TypeInfo[] typeParams)
   {
      if (typeParams == null || typeParams.length == 0)
         return NO_PARAMS;
      Class<?>[] params = new Class[typeParams.length];
      for (int i = 0 ; i < params.length ; i++)
         params[i] = typeParams[i].getType();
      return params;
   }

}
