/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.javassist;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.HashSet;
import java.util.Set;

import org.jboss.reflect.plugins.javassist.bytecode.JavassistMemberFactory;
import org.jboss.reflect.plugins.javassist.defaultmember.DefaultMemberFactory;
import org.jboss.reflect.plugins.javassist.metrics.MetricsJavassistConstructor;
import org.jboss.reflect.plugins.javassist.metrics.MetricsJavassistField;
import org.jboss.reflect.plugins.javassist.metrics.MetricsJavassistMethod;

/**
 * <p>JavassistReflectionFactory.
 * </p><p>
 * Creates instances of {@link JavassistMethod}, {@link JavassistConstructor} and
 * {@link JavassistField} that invoke methods, constructors, and accesses fields without
 * the use of reflection. Although a lot faster, there is an overhead associated with generating
 * these classes, so they should only be used for frequently used members </p>
 * <p>
 * To output the generated classes to disk, so the bytecode can be inspected, specify
 * <code>-Dorg.jboss.reflect.plugins.javassist.JavassistReflectionFactory.debug=true</code>.
 * </p>
 * <p>
 * To enable metrics, which can be useful in determining which members accessors should be generated, specify
 * <code>-Dorg.jboss.reflect.plugins.javassist.JavassistReflectionFactory.enableMetrics=true</code>.
 * </p>
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 105060 $
 */
public class JavassistReflectionFactory
{
   public static final JavassistReflectionFactory INSTANCE;
   
   /** Whether to output generated classes to disk */
   private static final boolean debug;
   
   /** Whether to enable metrics for the classes or not */
   private static final boolean enableMetrics;
   
   /** A file containing the names of the members that should be generated */
   private static final Set<String> forceGenerate;
   
   protected static final Class<?> MAGIC_ACCESSOR_IMPL;
   static
   {
      debug = initBooleanProperty(JavassistReflectionFactory.class.getName() + ".debug");
      enableMetrics = initBooleanProperty(JavassistReflectionFactory.class.getName() + ".enableMetrics");
      forceGenerate = initForceGenerate(JavassistReflectionFactory.class.getName() + ".file");
      INSTANCE = new JavassistReflectionFactory();
      
      try
      {
         MAGIC_ACCESSOR_IMPL = SecurityActions.classForName("sun.reflect.MagicAccessorImpl");
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException(e);
      }
   }
   
   private static boolean initBooleanProperty(final String name)
   {
      return AccessController.doPrivileged(new PrivilegedAction<Boolean>()
      {
         public Boolean run()
         {
            return Boolean.getBoolean(name);
         }
      });
      
   }

   public void addForceGenerate(String s)
   {
      forceGenerate.add(s);
   }
   
   public void removeForceGenerate(String s)
   {
      forceGenerate.remove(s);
   }
   
   private static Set<String> initForceGenerate(final String name)
   {
      try
      {
         return AccessController.doPrivileged(new PrivilegedExceptionAction<Set<String>>()
         {
            public Set<String> run() throws Exception
            {
               String generateFile = System.getProperty(JavassistReflectionFactory.class.getName() + ".file", null);
               Set<String> forceGenerate = new HashSet<String>();
               if (generateFile != null)
               {
                  File file = new File(generateFile);
                  if (!file.exists())
                     throw new IllegalArgumentException("No file called '" + generateFile + "' specified via -D" + JavassistReflectionFactory.class.getName() + ".file");
                  
                  BufferedReader in = null;
                  try
                  {
                     in = new BufferedReader(new FileReader(file));
                     String s = in.readLine();
                     while (s != null)
                     {
                        s = s.trim();
                        if (s.length() > 0)
                           forceGenerate.add(s);
                        s = in.readLine();
                     }
                  }
                  finally
                  {
                     try
                     {
                        in.close();
                     }
                     catch(Exception ignore)
                     {
                     }
                  }
               }
               return forceGenerate;
            }
         });
      }
      catch(PrivilegedActionException e)
      {
         if (e.getCause() instanceof RuntimeException)
            throw (RuntimeException)e.getCause();
         throw new RuntimeException(e.getCause());
      }
   }

   /**
    * Create a new JavassistReflectionFactory.
    * 
    */
   private JavassistReflectionFactory()
   {
   }

   /**
    * Creates a javassist method.
    * 
    * This method generates an implementation of {@code JavassistMethod}, loads
    * the generated class and instantiates it.
    * 
    * @param ctMethod the method
    * @return the method
    * @throws Throwable for any error
    */
   public JavassistMethod createMethod(JavassistMethodInfo info) throws Throwable
   {
      String sig = JavassistUtil.getSignature(info.getCtMethod());
      JavassistMethod method = generateAccessor(sig) ?
            JavassistMemberFactory.createJavassistMethod(MAGIC_ACCESSOR_IMPL, info, debug) :
               DefaultMemberFactory.createJavassistMethod(info);
      
      if (enableMetrics)
         method = new MetricsJavassistMethod(method, sig);
      
      return method;
   }
   
   /**
    * Create a javassist constructor
    * 
    * @param ctConstructor the constructor
    * @return the constructor
    * @throws Throwable for any error
    */
   public JavassistConstructor createConstructor(JavassistConstructorInfo info) throws Throwable
   {
      String sig = JavassistUtil.getSignature(info.getCtConstructor());
      JavassistConstructor ctor = generateAccessor(sig) ?
            JavassistMemberFactory.createJavassistConstructor(MAGIC_ACCESSOR_IMPL, info, debug) :
               DefaultMemberFactory.createJavassistConstructor(info);
            
      if (enableMetrics)
         ctor = new MetricsJavassistConstructor(ctor, sig);
      
      return ctor;
   }
   
   /**
    * Create a javassist field
    * 
    * @param ctField the field
    * @return the field
    * @throws Throwable for any error
    */
   public JavassistField createField(JavassistFieldInfo info) throws Throwable
   {
      String sig = JavassistUtil.getSignature(info.getCtField());
      JavassistField field = generateAccessor(sig) ?
            JavassistMemberFactory.createJavassistField(MAGIC_ACCESSOR_IMPL, info, debug) :
               DefaultMemberFactory.createJavassistField(info);
            
      if (enableMetrics)
         field = new MetricsJavassistField(field, sig);
      
      return field;
   }
   
   private boolean generateAccessor(String signature)
   {
      if (forceGenerate.isEmpty())
         return true;
      return forceGenerate.contains(signature);
   }
}
