/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.annotations.plugins;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

import org.jboss.metadata.spi.signature.*;

/**
 * Serialization helper for Signature
 * TODO - move to MDR
 *
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
public class SignatureSerializer implements Externalizable
{
   /** The serial version UID */
   private static final long serialVersionUID = 1L;

   private transient Signature signature;

   public SignatureSerializer()
   {
      // for de-serialization
   }

   public SignatureSerializer(Signature signature)
   {
      this.signature = signature;
   }

   public Signature getSignature()
   {
      return signature;
   }

   public void writeExternal(ObjectOutput out) throws IOException
   {
      String declaringClass = null;
      String name = null;
      String[] parameters = null;
      Integer parameterIndex = -1;

      if (signature instanceof ConstructorSignature)
      {
         ConstructorSignature constructorSignature = (ConstructorSignature) signature;
         parameters = constructorSignature.getParameters();
      }
      else if (signature instanceof MethodSignature)
      {
         MethodSignature methodSignature = (MethodSignature) signature;
         name = methodSignature.getName();
         parameters = methodSignature.getParameters();
      }
      else if (signature instanceof DeclaredMethodSignature)
      {
         DeclaredMethodSignature methodSignature = (DeclaredMethodSignature) signature;
         declaringClass = methodSignature.getDeclaringClass();
         name = methodSignature.getName();
         parameters = methodSignature.getParameters();
      }
      else if (signature instanceof MethodParametersSignature)
      {
         MethodParametersSignature methodSignature = (MethodParametersSignature) signature;
         name = methodSignature.getName();
         parameters = methodSignature.getParameters();
         parameterIndex = methodSignature.getParam();
      }
      else if (signature instanceof ConstructorParametersSignature)
      {
         ConstructorParametersSignature constructorSignature = (ConstructorParametersSignature) signature;
         parameters = constructorSignature.getParameters();
         parameterIndex = constructorSignature.getParam();
      }
      else if (signature instanceof FieldSignature)
      {
         FieldSignature fieldSignature = (FieldSignature) signature;
         name = fieldSignature.getName();
      }
      else
         throw new IllegalArgumentException("Unknown signature type: " + signature);

      // declaring class
      boolean bdc = declaringClass != null;
      out.writeBoolean(bdc);
      if (bdc)
         out.writeUTF(declaringClass);
      // name
      out.writeUTF(name);
      // parameters
      boolean bp = parameters != null;
      out.writeBoolean(bp);
      if (bp)
         out.writeObject(parameters);
      // index
      out.writeObject(parameterIndex);
   }

   public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
   {
      // declaring class
      String declaringClass = null;
      if (in.readBoolean())
         declaringClass = in.readUTF();
      // name
      String name = in.readUTF();
      // parameters
      String[] parameters = null;
      if (in.readBoolean())
         parameters = (String[]) in.readObject();
      // index
      Integer parameterIndex = (Integer) in.readObject();

      if (name == null)
      {
         if (parameterIndex >= 0)
            signature = new ConstructorParametersSignature(parameterIndex, parameters);
         else
            signature = new ConstructorSignature(parameters);
      }
      else
      {
         if (declaringClass != null)
         {
            signature = new DeclaredMethodSignature(declaringClass, name, parameters);
         }
         else if (parameters == null)
         {
            signature = new FieldSignature(name);
         }
         else if (parameterIndex >= 0)
         {
            signature = new MethodParametersSignature(name, parameters, parameterIndex);
         }
         else
         {
            signature = new MethodSignature(name, parameters);
         }
      }
   }
}