/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.web.plugins;

import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.net.URL;
import java.util.HashSet;
import java.util.Set;

import org.jboss.scanning.annotations.spi.AnnotationIndex;
import org.jboss.scanning.annotations.spi.Element;
import org.jboss.scanning.hierarchy.spi.HierarchyIndex;
import org.jboss.scanning.web.spi.ResourcesIndex;
import org.jboss.vfs.VirtualFile;

/**
 * Default resource index.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class DefaultResourcesIndex implements ResourcesIndex
{
   /** The annotation index */
   private AnnotationIndex annotations;
   /** The hierarchy index */
   private HierarchyIndex hierarchy;

   public DefaultResourcesIndex(AnnotationIndex annotations, HierarchyIndex hierarchy)
   {
      if (annotations == null)
         throw new IllegalArgumentException("Null annotations index");
      if (hierarchy == null)
         throw new IllegalArgumentException("Null hierarchy index");

      this.annotations = annotations;
      this.hierarchy = hierarchy;
   }

   public <A extends Annotation> Set<Class<?>> getAnnotatedClasses(VirtualFile cpEntry, Class<A> annotationToLookFor)
   {
      if (annotationToLookFor == null)
         throw new IllegalArgumentException("Null annotation class");

      URL url = cpEntry != null ? toURL(cpEntry) : null;
      Set<Element<A, AnnotatedElement>> elements = annotations.getAnnotatedClasses(url, annotationToLookFor, null);
      Set<Class<?>> results = new HashSet<Class<?>>();
      for (Element<A, AnnotatedElement> elt : elements)
         results.add(elt.getOwner());
      return results;
   }

   @SuppressWarnings("deprecation")
   public <T> Set<Class<? extends T>> getInheritedClasses(VirtualFile cpEntry, Class<T> superTypeToLookFor)
   {
      if (superTypeToLookFor == null)
         throw new IllegalArgumentException("Null super type");

      URL url = cpEntry != null ? toURL(cpEntry) : null;
      return hierarchy.getSubClassesOf(url, superTypeToLookFor);
   }

   protected static URL toURL(VirtualFile file)
   {
      try
      {
         return file.toURL();

      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }
}
