/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Objects;
import org.neo4j.cli.AbstractCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.commandline.Util;
import org.neo4j.commandline.dbms.CannotWriteException;
import org.neo4j.commandline.dbms.LockChecker;
import org.neo4j.commandline.dbms.StoreVersionLoader;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.ConfigUtils;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.archive.IncorrectFormat;
import org.neo4j.dbms.archive.Loader;
import org.neo4j.function.ThrowingSupplier;
import org.neo4j.io.fs.FileUtils;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.kernel.database.NormalizedDatabaseName;
import org.neo4j.kernel.internal.locker.FileLockException;
import picocli.CommandLine;

@CommandLine.Command(name="load", header={"Load a database from an archive created with the dump command."}, description={"Load a database from an archive. <archive-path> must be an archive created with the dump command. <database> is the name of the database to create. Existing databases can be replaced by specifying --force. It is not possible to replace a database that is mounted in a running Neo4j server. If --info is specified, then the database is not loaded, but information (i.e. file count, byte count, and format of load file) about the archive is printed instead."})
public class LoadCommand
extends AbstractCommand {
    private static final String STANDARD_INPUT = "-";
    @CommandLine.Option(names={"--from"}, required=true, paramLabel="<path>", description={"Path to archive created with the dump command or '-' to read from standard input."})
    private String from;
    @CommandLine.Option(names={"--database"}, description={"Name of the database to load."}, defaultValue="neo4j", converter={Converters.DatabaseNameConverter.class})
    protected NormalizedDatabaseName database;
    @CommandLine.Option(names={"--force"}, arity="0", description={"If an existing database should be replaced."})
    private boolean force;
    @CommandLine.Option(names={"--info"}, description={"Print meta-data information about the archive file, instead of loading the contained database."})
    private boolean info;
    private final Loader loader;

    public LoadCommand(ExecutionContext ctx, Loader loader) {
        super(ctx);
        this.loader = Objects.requireNonNull(loader);
    }

    public void execute() {
        if (this.info) {
            this.inspectDump();
        } else {
            try {
                this.loadDump();
            }
            catch (IOException e) {
                Util.wrapIOException((IOException)e);
            }
        }
    }

    private void inspectDump() {
        try {
            Loader.DumpMetaData metaData = this.loader.getMetaData(this.getArchiveInputStreamSupplier());
            this.ctx.out().println("Format: " + metaData.format);
            this.ctx.out().println("Files: " + metaData.fileCount);
            this.ctx.out().println("Bytes: " + metaData.byteCount);
        }
        catch (IOException e) {
            Util.wrapIOException((IOException)e);
        }
    }

    private ThrowingSupplier<InputStream, IOException> getArchiveInputStreamSupplier() throws IOException {
        Path path = this.getArchivePath();
        if (path != null) {
            return () -> Files.newInputStream(path, new OpenOption[0]);
        }
        return () -> ((ExecutionContext)this.ctx).in();
    }

    private Path getArchivePath() {
        if (STANDARD_INPUT.equals(this.from)) {
            return null;
        }
        return Path.of(this.from, new String[0]);
    }

    protected void loadDump() throws IOException {
        Config config = this.buildConfig();
        DatabaseLayout databaseLayout = Neo4jLayout.of((Config)config).databaseLayout(this.database.name());
        this.ctx.fs().mkdirs(databaseLayout.databaseDirectory());
        this.ctx.fs().mkdirs(databaseLayout.getNeo4jLayout().transactionLogsRootDirectory());
        try (Closeable ignore = LockChecker.checkDatabaseLock((DatabaseLayout)databaseLayout);){
            LoadCommand.deleteIfNecessary(databaseLayout, this.force);
            this.load(this.getArchiveInputStreamSupplier(), databaseLayout, this.getArchivePath());
        }
        catch (FileLockException e) {
            throw new CommandFailedException("The database is in use. Stop database '" + this.database.name() + "' and try again.", (Throwable)e);
        }
        catch (IOException e) {
            Util.wrapIOException((IOException)e);
        }
        catch (CannotWriteException e) {
            throw new CommandFailedException("You do not have permission to load the database.", (Throwable)e);
        }
        StoreVersionLoader.Result result = this.loader.getStoreVersion(this.ctx.fs(), config, databaseLayout);
        if (!result.isLatest) {
            this.ctx.err().printf("The loaded database is not on the latest format (current:%s, latest:%s). Set %s=true to enable migration.%n", result.currentFormatName, result.latestFormatName, GraphDatabaseSettings.allow_upgrade.name());
        }
    }

    protected Config buildConfig() {
        Config cfg = Config.newBuilder().fromFileNoThrow(this.ctx.confDir().resolve("neo4j.conf")).commandExpansion(this.allowCommandExpansion).set(GraphDatabaseSettings.neo4j_home, (Object)this.ctx.homeDir()).build();
        ConfigUtils.disableAllConnectors((Config)cfg);
        return cfg;
    }

    private static void deleteIfNecessary(DatabaseLayout databaseLayout, boolean force) {
        try {
            if (force) {
                FileUtils.deleteDirectory((Path)databaseLayout.databaseDirectory(), path -> !path.equals(databaseLayout.databaseLockFile()));
                FileUtils.deleteDirectory((Path)databaseLayout.getTransactionLogsDirectory());
            }
        }
        catch (IOException e) {
            Util.wrapIOException((IOException)e);
        }
    }

    private void load(ThrowingSupplier<InputStream, IOException> streamSupplier, DatabaseLayout databaseLayout, Path archive) {
        String inputName = archive == null ? "reading from stdin" : archive.toString();
        try {
            this.loader.load(databaseLayout, streamSupplier, inputName);
        }
        catch (NoSuchFileException e) {
            if (archive != null && Paths.get(e.getMessage(), new String[0]).toAbsolutePath().equals(archive.toAbsolutePath())) {
                throw new CommandFailedException("Archive does not exist: " + inputName, (Throwable)e);
            }
            Util.wrapIOException((IOException)e);
        }
        catch (FileAlreadyExistsException e) {
            throw new CommandFailedException("Database already exists: " + databaseLayout.getDatabaseName(), (Throwable)e);
        }
        catch (AccessDeniedException e) {
            throw new CommandFailedException("You do not have permission to load the database.", (Throwable)e);
        }
        catch (IOException e) {
            Util.wrapIOException((IOException)e);
        }
        catch (IncorrectFormat incorrectFormat) {
            throw new CommandFailedException("Not a valid Neo4j archive: " + inputName, (Throwable)incorrectFormat);
        }
    }
}

