/*
 * Decompiled with CFR 0.152.
 */
package sun.security.ssl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Hashtable;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.SecretKey;
import javax.crypto.ShortBufferException;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import sun.misc.HexDumpEncoder;
import sun.security.ssl.Authenticator;
import sun.security.ssl.CipherSuite;
import sun.security.ssl.Debug;
import sun.security.ssl.JsseJce;
import sun.security.ssl.MAC;
import sun.security.ssl.ProtocolVersion;

final class CipherBox {
    static final CipherBox NULL = new CipherBox();
    private static final Debug debug = Debug.getInstance("ssl");
    private final ProtocolVersion protocolVersion;
    private final Cipher cipher;
    private SecureRandom random;
    private final byte[] fixedIv;
    private final Key key;
    private final int mode;
    private final int tagSize;
    private final int recordIvSize;
    private final CipherSuite.CipherType cipherType;
    private static Hashtable<Integer, IvParameterSpec> masks;

    private CipherBox() {
        this.protocolVersion = ProtocolVersion.DEFAULT;
        this.cipher = null;
        this.cipherType = CipherSuite.CipherType.STREAM_CIPHER;
        this.fixedIv = new byte[0];
        this.key = null;
        this.mode = 1;
        this.random = null;
        this.tagSize = 0;
        this.recordIvSize = 0;
    }

    private CipherBox(ProtocolVersion protocolVersion, CipherSuite.BulkCipher bulkCipher, SecretKey key, IvParameterSpec iv, SecureRandom random, boolean encrypt) throws NoSuchAlgorithmException {
        try {
            this.protocolVersion = protocolVersion;
            this.cipher = JsseJce.getCipher(bulkCipher.transformation);
            int n = this.mode = encrypt ? 1 : 2;
            if (random == null) {
                random = JsseJce.getSecureRandom();
            }
            this.random = random;
            this.cipherType = bulkCipher.cipherType;
            if (iv == null && bulkCipher.ivSize != 0 && this.mode == 2 && protocolVersion.v >= ProtocolVersion.TLS11.v) {
                iv = CipherBox.getFixedMask(bulkCipher.ivSize);
            }
            if (this.cipherType == CipherSuite.CipherType.AEAD_CIPHER) {
                this.tagSize = bulkCipher.tagSize;
                this.key = key;
                this.fixedIv = iv.getIV();
                if (this.fixedIv == null || this.fixedIv.length != bulkCipher.fixedIvSize) {
                    throw new RuntimeException("Improper fixed IV for AEAD");
                }
                this.recordIvSize = bulkCipher.ivSize - bulkCipher.fixedIvSize;
            } else {
                this.tagSize = 0;
                this.fixedIv = new byte[0];
                this.recordIvSize = 0;
                this.key = null;
                this.cipher.init(this.mode, (Key)key, iv, random);
            }
        }
        catch (NoSuchAlgorithmException e) {
            throw e;
        }
        catch (Exception e) {
            throw new NoSuchAlgorithmException("Could not create cipher " + bulkCipher, e);
        }
        catch (ExceptionInInitializerError e) {
            throw new NoSuchAlgorithmException("Could not create cipher " + bulkCipher, e);
        }
    }

    static CipherBox newCipherBox(ProtocolVersion version, CipherSuite.BulkCipher cipher, SecretKey key, IvParameterSpec iv, SecureRandom random, boolean encrypt) throws NoSuchAlgorithmException {
        if (!cipher.allowed) {
            throw new NoSuchAlgorithmException("Unsupported cipher " + cipher);
        }
        if (cipher == CipherSuite.B_NULL) {
            return NULL;
        }
        return new CipherBox(version, cipher, key, iv, random, encrypt);
    }

    private static IvParameterSpec getFixedMask(int ivSize) {
        IvParameterSpec iv;
        if (masks == null) {
            masks = new Hashtable(5);
        }
        if ((iv = masks.get(ivSize)) == null) {
            iv = new IvParameterSpec(new byte[ivSize]);
            masks.put(ivSize, iv);
        }
        return iv;
    }

    int encrypt(byte[] buf, int offset, int len) {
        if (this.cipher == null) {
            return len;
        }
        try {
            int blockSize = this.cipher.getBlockSize();
            if (this.cipherType == CipherSuite.CipherType.BLOCK_CIPHER) {
                len = CipherBox.addPadding(buf, offset, len, blockSize);
            }
            if (debug != null && Debug.isOn("plaintext")) {
                try {
                    HexDumpEncoder hd = new HexDumpEncoder();
                    System.out.println("Padded plaintext before ENCRYPTION:  len = " + len);
                    hd.encodeBuffer((InputStream)new ByteArrayInputStream(buf, offset, len), (OutputStream)System.out);
                }
                catch (IOException hd) {
                    // empty catch block
                }
            }
            if (this.cipherType == CipherSuite.CipherType.AEAD_CIPHER) {
                try {
                    return this.cipher.doFinal(buf, offset, len, buf, offset);
                }
                catch (BadPaddingException | IllegalBlockSizeException ibe) {
                    throw new RuntimeException("Cipher error in AEAD mode in JCE provider " + this.cipher.getProvider().getName(), ibe);
                }
            }
            int newLen = this.cipher.update(buf, offset, len, buf, offset);
            if (newLen != len) {
                throw new RuntimeException("Cipher buffering error in JCE provider " + this.cipher.getProvider().getName());
            }
            return newLen;
        }
        catch (ShortBufferException e) {
            throw new ArrayIndexOutOfBoundsException(e.toString());
        }
    }

    int encrypt(ByteBuffer bb, int outLimit) {
        int newLen;
        int len = bb.remaining();
        if (this.cipher == null) {
            bb.position(bb.limit());
            return len;
        }
        int pos = bb.position();
        int blockSize = this.cipher.getBlockSize();
        if (this.cipherType == CipherSuite.CipherType.BLOCK_CIPHER) {
            len = CipherBox.addPadding(bb, blockSize);
            bb.position(pos);
        }
        if (debug != null && Debug.isOn("plaintext")) {
            try {
                HexDumpEncoder hd = new HexDumpEncoder();
                System.out.println("Padded plaintext before ENCRYPTION:  len = " + len);
                hd.encodeBuffer(bb.duplicate(), (OutputStream)System.out);
            }
            catch (IOException hd) {
                // empty catch block
            }
        }
        ByteBuffer dup = bb.duplicate();
        if (this.cipherType == CipherSuite.CipherType.AEAD_CIPHER) {
            try {
                int newLen2;
                int outputSize = this.cipher.getOutputSize(dup.remaining());
                if (outputSize > bb.remaining()) {
                    if (outLimit < pos + outputSize) {
                        throw new ShortBufferException("need more space in output buffer");
                    }
                    bb.limit(pos + outputSize);
                }
                if ((newLen2 = this.cipher.doFinal(dup, bb)) != outputSize) {
                    throw new RuntimeException("Cipher buffering error in JCE provider " + this.cipher.getProvider().getName());
                }
                return newLen2;
            }
            catch (BadPaddingException | IllegalBlockSizeException | ShortBufferException ibse) {
                throw new RuntimeException("Cipher error in AEAD mode in JCE provider " + this.cipher.getProvider().getName(), ibse);
            }
        }
        try {
            newLen = this.cipher.update(dup, bb);
        }
        catch (ShortBufferException sbe) {
            throw new RuntimeException("Cipher buffering error in JCE provider " + this.cipher.getProvider().getName());
        }
        if (bb.position() != dup.position()) {
            throw new RuntimeException("bytebuffer padding error");
        }
        if (newLen != len) {
            throw new RuntimeException("Cipher buffering error in JCE provider " + this.cipher.getProvider().getName());
        }
        return newLen;
    }

    int decrypt(byte[] buf, int offset, int len, int tagLen) throws BadPaddingException {
        if (this.cipher == null) {
            return len;
        }
        try {
            int newLen;
            if (this.cipherType == CipherSuite.CipherType.AEAD_CIPHER) {
                try {
                    newLen = this.cipher.doFinal(buf, offset, len, buf, offset);
                }
                catch (IllegalBlockSizeException ibse) {
                    throw new RuntimeException("Cipher error in AEAD mode in JCE provider " + this.cipher.getProvider().getName(), ibse);
                }
            } else {
                newLen = this.cipher.update(buf, offset, len, buf, offset);
                if (newLen != len) {
                    throw new RuntimeException("Cipher buffering error in JCE provider " + this.cipher.getProvider().getName());
                }
            }
            if (debug != null && Debug.isOn("plaintext")) {
                try {
                    HexDumpEncoder hd = new HexDumpEncoder();
                    System.out.println("Padded plaintext after DECRYPTION:  len = " + newLen);
                    hd.encodeBuffer((InputStream)new ByteArrayInputStream(buf, offset, newLen), (OutputStream)System.out);
                }
                catch (IOException hd) {
                    // empty catch block
                }
            }
            if (this.cipherType == CipherSuite.CipherType.BLOCK_CIPHER) {
                int blockSize = this.cipher.getBlockSize();
                newLen = CipherBox.removePadding(buf, offset, newLen, tagLen, blockSize, this.protocolVersion);
                if (this.protocolVersion.v >= ProtocolVersion.TLS11.v && newLen < blockSize) {
                    throw new BadPaddingException("invalid explicit IV");
                }
            }
            return newLen;
        }
        catch (ShortBufferException e) {
            throw new ArrayIndexOutOfBoundsException(e.toString());
        }
    }

    int decrypt(ByteBuffer bb, int tagLen) throws BadPaddingException {
        int len = bb.remaining();
        if (this.cipher == null) {
            bb.position(bb.limit());
            return len;
        }
        try {
            int newLen;
            int pos = bb.position();
            ByteBuffer dup = bb.duplicate();
            if (this.cipherType == CipherSuite.CipherType.AEAD_CIPHER) {
                try {
                    newLen = this.cipher.doFinal(dup, bb);
                }
                catch (IllegalBlockSizeException ibse) {
                    throw new RuntimeException("Cipher error in AEAD mode \"" + ibse.getMessage() + " \"in JCE provider " + this.cipher.getProvider().getName());
                }
            } else {
                newLen = this.cipher.update(dup, bb);
                if (newLen != len) {
                    throw new RuntimeException("Cipher buffering error in JCE provider " + this.cipher.getProvider().getName());
                }
            }
            bb.limit(pos + newLen);
            if (debug != null && Debug.isOn("plaintext")) {
                try {
                    HexDumpEncoder hd = new HexDumpEncoder();
                    System.out.println("Padded plaintext after DECRYPTION:  len = " + newLen);
                    hd.encodeBuffer((ByteBuffer)bb.duplicate().position(pos), (OutputStream)System.out);
                }
                catch (IOException hd) {
                    // empty catch block
                }
            }
            if (this.cipherType == CipherSuite.CipherType.BLOCK_CIPHER) {
                int blockSize = this.cipher.getBlockSize();
                bb.position(pos);
                newLen = CipherBox.removePadding(bb, tagLen, blockSize, this.protocolVersion);
                if (this.protocolVersion.v >= ProtocolVersion.TLS11.v) {
                    if (newLen < blockSize) {
                        throw new BadPaddingException("invalid explicit IV");
                    }
                    bb.position(bb.limit());
                }
            }
            return newLen;
        }
        catch (ShortBufferException e) {
            throw new ArrayIndexOutOfBoundsException(e.toString());
        }
    }

    private static int addPadding(byte[] buf, int offset, int len, int blockSize) {
        int newlen = len + 1;
        if (newlen % blockSize != 0) {
            newlen += blockSize - 1;
            newlen -= newlen % blockSize;
        }
        int pad = newlen - len;
        if (buf.length < newlen + offset) {
            throw new IllegalArgumentException("no space to pad buffer");
        }
        offset += len;
        for (int i = 0; i < pad; ++i) {
            buf[offset++] = (byte)(pad - 1);
        }
        return newlen;
    }

    private static int addPadding(ByteBuffer bb, int blockSize) {
        int len = bb.remaining();
        int offset = bb.position();
        int newlen = len + 1;
        if (newlen % blockSize != 0) {
            newlen += blockSize - 1;
            newlen -= newlen % blockSize;
        }
        int pad = newlen - len;
        bb.limit(newlen + offset);
        offset += len;
        for (int i = 0; i < pad; ++i) {
            bb.put(offset++, (byte)(pad - 1));
        }
        bb.position(offset);
        bb.limit(offset);
        return newlen;
    }

    private static int[] checkPadding(byte[] buf, int offset, int len, byte pad) {
        if (len <= 0) {
            throw new RuntimeException("padding len must be positive");
        }
        int[] results = new int[]{0, 0};
        int i = 0;
        while (i <= 256) {
            for (int j = 0; j < len && i <= 256; ++j, ++i) {
                if (buf[offset + j] != pad) {
                    results[0] = results[0] + 1;
                    continue;
                }
                results[1] = results[1] + 1;
            }
        }
        return results;
    }

    private static int[] checkPadding(ByteBuffer bb, byte pad) {
        if (!bb.hasRemaining()) {
            throw new RuntimeException("hasRemaining() must be positive");
        }
        int[] results = new int[]{0, 0};
        bb.mark();
        int i = 0;
        while (i <= 256) {
            while (bb.hasRemaining() && i <= 256) {
                if (bb.get() != pad) {
                    results[0] = results[0] + 1;
                } else {
                    results[1] = results[1] + 1;
                }
                ++i;
            }
            bb.reset();
        }
        return results;
    }

    private static int removePadding(byte[] buf, int offset, int len, int tagLen, int blockSize, ProtocolVersion protocolVersion) throws BadPaddingException {
        int padOffset = offset + len - 1;
        int padLen = buf[padOffset] & 0xFF;
        int newLen = len - (padLen + 1);
        if (newLen - tagLen < 0) {
            CipherBox.checkPadding(buf, offset, len, (byte)(padLen & 0xFF));
            throw new BadPaddingException("Invalid Padding length: " + padLen);
        }
        int[] results = CipherBox.checkPadding(buf, offset + newLen, padLen + 1, (byte)(padLen & 0xFF));
        if (protocolVersion.v >= ProtocolVersion.TLS10.v) {
            if (results[0] != 0) {
                throw new BadPaddingException("Invalid TLS padding data");
            }
        } else if (padLen > blockSize) {
            throw new BadPaddingException("Invalid SSLv3 padding");
        }
        return newLen;
    }

    private static int removePadding(ByteBuffer bb, int tagLen, int blockSize, ProtocolVersion protocolVersion) throws BadPaddingException {
        int offset;
        int padOffset;
        int padLen;
        int len = bb.remaining();
        int newLen = len - ((padLen = bb.get(padOffset = (offset = bb.position()) + len - 1) & 0xFF) + 1);
        if (newLen - tagLen < 0) {
            CipherBox.checkPadding(bb.duplicate(), (byte)(padLen & 0xFF));
            throw new BadPaddingException("Invalid Padding length: " + padLen);
        }
        int[] results = CipherBox.checkPadding((ByteBuffer)bb.duplicate().position(offset + newLen), (byte)(padLen & 0xFF));
        if (protocolVersion.v >= ProtocolVersion.TLS10.v) {
            if (results[0] != 0) {
                throw new BadPaddingException("Invalid TLS padding data");
            }
        } else if (padLen > blockSize) {
            throw new BadPaddingException("Invalid SSLv3 padding");
        }
        bb.position(offset + newLen);
        bb.limit(offset + newLen);
        return newLen;
    }

    void dispose() {
        try {
            if (this.cipher != null) {
                this.cipher.doFinal();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    boolean isCBCMode() {
        return this.cipherType == CipherSuite.CipherType.BLOCK_CIPHER;
    }

    boolean isAEADMode() {
        return this.cipherType == CipherSuite.CipherType.AEAD_CIPHER;
    }

    boolean isNullCipher() {
        return this.cipher == null;
    }

    int getExplicitNonceSize() {
        switch (this.cipherType) {
            case BLOCK_CIPHER: {
                if (this.protocolVersion.v < ProtocolVersion.TLS11.v) break;
                return this.cipher.getBlockSize();
            }
            case AEAD_CIPHER: {
                return this.recordIvSize;
            }
        }
        return 0;
    }

    int applyExplicitNonce(Authenticator authenticator, byte contentType, ByteBuffer bb) throws BadPaddingException {
        switch (this.cipherType) {
            case BLOCK_CIPHER: {
                int tagLen;
                int n = tagLen = authenticator instanceof MAC ? ((MAC)authenticator).MAClen() : 0;
                if (tagLen != 0 && !this.sanityCheck(tagLen, bb.remaining())) {
                    throw new BadPaddingException("ciphertext sanity check failed");
                }
                if (this.protocolVersion.v < ProtocolVersion.TLS11.v) break;
                return this.cipher.getBlockSize();
            }
            case AEAD_CIPHER: {
                if (bb.remaining() < this.recordIvSize + this.tagSize) {
                    throw new BadPaddingException("invalid AEAD cipher fragment");
                }
                byte[] iv = Arrays.copyOf(this.fixedIv, this.fixedIv.length + this.recordIvSize);
                bb.get(iv, this.fixedIv.length, this.recordIvSize);
                bb.position(bb.position() - this.recordIvSize);
                GCMParameterSpec spec = new GCMParameterSpec(this.tagSize * 8, iv);
                try {
                    this.cipher.init(this.mode, this.key, spec, this.random);
                }
                catch (InvalidAlgorithmParameterException | InvalidKeyException ikae) {
                    throw new RuntimeException("invalid key or spec in GCM mode", ikae);
                }
                byte[] aad = authenticator.acquireAuthenticationBytes(contentType, bb.remaining() - this.recordIvSize - this.tagSize);
                this.cipher.updateAAD(aad);
                return this.recordIvSize;
            }
        }
        return 0;
    }

    int applyExplicitNonce(Authenticator authenticator, byte contentType, byte[] buf, int offset, int cipheredLength) throws BadPaddingException {
        ByteBuffer bb = ByteBuffer.wrap(buf, offset, cipheredLength);
        return this.applyExplicitNonce(authenticator, contentType, bb);
    }

    byte[] createExplicitNonce(Authenticator authenticator, byte contentType, int fragmentLength) {
        byte[] nonce = new byte[]{};
        switch (this.cipherType) {
            case BLOCK_CIPHER: {
                if (this.protocolVersion.v < ProtocolVersion.TLS11.v) break;
                nonce = new byte[this.cipher.getBlockSize()];
                this.random.nextBytes(nonce);
                break;
            }
            case AEAD_CIPHER: {
                nonce = authenticator.sequenceNumber();
                byte[] iv = Arrays.copyOf(this.fixedIv, this.fixedIv.length + nonce.length);
                System.arraycopy(nonce, 0, iv, this.fixedIv.length, nonce.length);
                GCMParameterSpec spec = new GCMParameterSpec(this.tagSize * 8, iv);
                try {
                    this.cipher.init(this.mode, this.key, spec, this.random);
                }
                catch (InvalidAlgorithmParameterException | InvalidKeyException ikae) {
                    throw new RuntimeException("invalid key or spec in GCM mode", ikae);
                }
                byte[] aad = authenticator.acquireAuthenticationBytes(contentType, fragmentLength);
                this.cipher.updateAAD(aad);
            }
        }
        return nonce;
    }

    private boolean sanityCheck(int tagLen, int fragmentLen) {
        if (!this.isCBCMode()) {
            return fragmentLen >= tagLen;
        }
        int blockSize = this.cipher.getBlockSize();
        if (fragmentLen % blockSize == 0) {
            int minimal = tagLen + 1;
            int n = minimal = minimal >= blockSize ? minimal : blockSize;
            if (this.protocolVersion.v >= ProtocolVersion.TLS11.v) {
                minimal += blockSize;
            }
            return fragmentLen >= minimal;
        }
        return false;
    }
}

