/*
 * ArrayInstance.java January 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.xml.core;

import java.lang.reflect.Array;

import org.simpleframework.xml.strategy.Value;

/**
 * The <code>ArrayInstance</code> object is used for creating arrays
 * from a specified <code>Value</code> object. This allows primitive
 * and composite arrays to be acquired either by reference or by value 
 * from the given value object. This must be  given the length of the 
 * array so that it can be allocated correctly.
 * 
 * @author Niall Gallagher
 * 
 * @see org.simpleframework.xml.core.Instance
 */
class ArrayInstance implements Instance {
 
   /**
    * This is the value object that contains the criteria.
    */
   private final Value value;
   
   /**
    * This is the array component type for the created array.
    */
   private final Class type;
   
   /**
    * This is the length of the array to be instantiated.
    */
   private final int length;
   
   /**
    * Constructor for the <code>ArrayInstance</code> object. This
    * is used to create an object that can create an array of the
    * given length and specified component type.
    * 
    * @param value this is the value object describing the instance
    */
   public ArrayInstance(Value value) {
      this.length = value.getLength();
      this.type = value.getType();
      this.value = value;
   }
   
   /**
    * This method is used to acquire an instance of the type that
    * is defined by this object. If for some reason the type can
    * not be instantiated an exception is thrown from this.
    * 
    * @return an instance of the type this object represents
    */
   public Object getInstance() throws Exception {      
      if(value.isReference()) {
         return value.getValue();
      }
      Object array = Array.newInstance(type, length);
      
      if(value != null) {
         value.setValue(array);
      }
      return array;
   }
   
   /**
    * This method is used acquire the value from the type and if
    * possible replace the value for the type. If the value can
    * not be replaced then an exception should be thrown. This 
    * is used to allow primitives to be inserted into a graph.
    * 
    * @param array this is the array to insert as the value
    * 
    * @return an instance of the type this object represents
    */
   public Object setInstance(Object array) {
      if(value != null) {
         value.setValue(array);
      }
      return array;
   }
   
   /**
    * This is the type of the object instance that will be created
    * by the <code>getInstance</code> method. This allows the 
    * deserialization process to perform checks against the field.
    * 
    * @return the type of the object that will be instantiated
    */
   public Class getType() {
      return type;
   }

   /**
    * This is used to determine if the type is a reference type.
    * A reference type is a type that does not require any XML
    * deserialization based on its annotations. Values that are
    * references could be substitutes objects of existing ones. 
    * 
    * @return this returns true if the object is a reference
    */
   public boolean isReference() {
      return value.isReference();
   }
}
