/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.scanner.scan.filesystem;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.nio.file.FileSystemLoopException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.api.batch.ScannerSide;
import org.sonar.api.batch.fs.InputComponent;
import org.sonar.api.batch.fs.InputFile;
import org.sonar.api.batch.fs.InputFileFilter;
import org.sonar.api.batch.fs.internal.DefaultInputDir;
import org.sonar.api.batch.fs.internal.DefaultInputFile;
import org.sonar.api.batch.fs.internal.DefaultInputModule;
import org.sonar.api.scan.filesystem.PathResolver;
import org.sonar.api.utils.MessageException;
import org.sonar.scanner.scan.DefaultComponentTree;
import org.sonar.scanner.scan.filesystem.BatchIdGenerator;
import org.sonar.scanner.scan.filesystem.DefaultModuleFileSystem;
import org.sonar.scanner.scan.filesystem.ExclusionFilters;
import org.sonar.scanner.scan.filesystem.InputComponentStore;
import org.sonar.scanner.scan.filesystem.InputFileBuilder;
import org.sonar.scanner.scan.filesystem.LanguageDetection;
import org.sonar.scanner.scan.filesystem.ModuleFileSystemInitializer;
import org.sonar.scanner.util.ProgressReport;

@ScannerSide
public class FileIndexer {
    private static final Logger LOG = LoggerFactory.getLogger(FileIndexer.class);
    private final InputFileFilter[] filters;
    private final ExclusionFilters exclusionFilters;
    private final InputFileBuilder inputFileBuilder;
    private final DefaultComponentTree componentTree;
    private final DefaultInputModule module;
    private final BatchIdGenerator batchIdGenerator;
    private final InputComponentStore componentStore;
    private final ModuleFileSystemInitializer moduleFileSystemInitializer;
    private ExecutorService executorService;
    private final List<Future<Void>> tasks;
    private final DefaultModuleFileSystem defaultModuleFileSystem;
    private final LanguageDetection langDetection;
    private ProgressReport progressReport;

    public FileIndexer(BatchIdGenerator batchIdGenerator, InputComponentStore componentStore, DefaultInputModule module, ExclusionFilters exclusionFilters, DefaultComponentTree componentTree, InputFileBuilder inputFileBuilder, ModuleFileSystemInitializer initializer, DefaultModuleFileSystem defaultModuleFileSystem, LanguageDetection languageDetection, InputFileFilter[] filters) {
        this.batchIdGenerator = batchIdGenerator;
        this.componentStore = componentStore;
        this.module = module;
        this.componentTree = componentTree;
        this.inputFileBuilder = inputFileBuilder;
        this.moduleFileSystemInitializer = initializer;
        this.defaultModuleFileSystem = defaultModuleFileSystem;
        this.langDetection = languageDetection;
        this.filters = filters;
        this.exclusionFilters = exclusionFilters;
        this.tasks = new ArrayList<Future<Void>>();
    }

    public FileIndexer(BatchIdGenerator batchIdGenerator, InputComponentStore componentStore, DefaultInputModule module, ExclusionFilters exclusionFilters, DefaultComponentTree componentTree, InputFileBuilder inputFileBuilder, ModuleFileSystemInitializer initializer, DefaultModuleFileSystem defaultModuleFileSystem, LanguageDetection languageDetection) {
        this(batchIdGenerator, componentStore, module, exclusionFilters, componentTree, inputFileBuilder, initializer, defaultModuleFileSystem, languageDetection, new InputFileFilter[0]);
    }

    public void index() {
        int threads = Math.max(1, Runtime.getRuntime().availableProcessors() - 1);
        this.executorService = Executors.newFixedThreadPool(threads, new ThreadFactoryBuilder().setNameFormat("FileIndexer-%d").build());
        this.progressReport = new ProgressReport("Report about progress of file indexation", TimeUnit.SECONDS.toMillis(10L));
        this.progressReport.start("Index files");
        this.exclusionFilters.prepare();
        Progress progress = new Progress();
        this.indexFiles(this.moduleFileSystemInitializer.sources(), InputFile.Type.MAIN, progress);
        this.indexFiles(this.moduleFileSystemInitializer.tests(), InputFile.Type.TEST, progress);
        this.waitForTasksToComplete();
        this.progressReport.stop(progress.count() + " " + FileIndexer.pluralizeFiles(progress.count()) + " indexed");
        if (this.exclusionFilters.hasPattern()) {
            LOG.info("{} {} ignored because of inclusion/exclusion patterns", (Object)progress.excludedByPatternsCount(), (Object)FileIndexer.pluralizeFiles(progress.excludedByPatternsCount()));
        }
    }

    private void waitForTasksToComplete() {
        this.executorService.shutdown();
        for (Future<Void> task : this.tasks) {
            try {
                task.get();
            }
            catch (ExecutionException e) {
                throw e.getCause() instanceof RuntimeException ? (RuntimeException)e.getCause() : new IllegalStateException(e.getCause());
            }
            catch (InterruptedException e) {
                throw new IllegalStateException(e);
            }
        }
    }

    private static String pluralizeFiles(int count) {
        return count == 1 ? "file" : "files";
    }

    private void indexFiles(List<Path> sources, InputFile.Type type, Progress progress) {
        try {
            for (Path dirOrFile : sources) {
                if (dirOrFile.toFile().isDirectory()) {
                    this.indexDirectory(dirOrFile, type, progress);
                    continue;
                }
                this.tasks.add(this.executorService.submit(() -> this.indexFile(dirOrFile, type, progress)));
            }
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to index files", e);
        }
    }

    private void indexDirectory(Path dirToIndex, InputFile.Type type, Progress progress) throws IOException {
        Files.walkFileTree(dirToIndex.normalize(), Collections.singleton(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, new IndexFileVisitor(type, progress));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Void indexFile(Path sourceFile, InputFile.Type type, Progress progress) throws IOException {
        Path realAbsoluteFile = sourceFile.toRealPath(LinkOption.NOFOLLOW_LINKS).toAbsolutePath().normalize();
        if (!realAbsoluteFile.startsWith(this.module.getBaseDir())) {
            LOG.warn("File '{}' is ignored. It is not located in module basedir '{}'.", (Object)realAbsoluteFile.toAbsolutePath(), (Object)this.module.getBaseDir());
            return null;
        }
        Path relativePath = this.module.getBaseDir().relativize(realAbsoluteFile);
        if (!this.exclusionFilters.accept(realAbsoluteFile, relativePath, type)) {
            progress.increaseExcludedByPatternsCount();
            return null;
        }
        String language = this.langDetection.language(realAbsoluteFile, relativePath);
        if (language == null && this.langDetection.getForcedLanguage() != null) {
            LOG.warn("File '{}' is ignored because it doesn't belong to the forced language '{}'", (Object)realAbsoluteFile.toAbsolutePath(), (Object)this.langDetection.getForcedLanguage());
            return null;
        }
        DefaultInputFile inputFile = this.inputFileBuilder.create(type, realAbsoluteFile, language);
        if (!this.accept((InputFile)inputFile)) {
            progress.increaseExcludedByPatternsCount();
            return null;
        }
        String parentRelativePath = this.getParentRelativePath(realAbsoluteFile);
        FileIndexer fileIndexer = this;
        synchronized (fileIndexer) {
            progress.markAsIndexed(inputFile);
            this.indexFileAndParentDir((InputFile)inputFile, parentRelativePath);
        }
        LOG.debug("'{}' indexed {}with language '{}'", new Object[]{relativePath, type == InputFile.Type.TEST ? "as test " : "", inputFile.language()});
        this.inputFileBuilder.checkMetadata(inputFile);
        return null;
    }

    private String getParentRelativePath(Path filePath) {
        Path parentDir = filePath.getParent();
        return (String)PathResolver.relativize((Path)this.module.getBaseDir(), (Path)parentDir).orElseThrow(() -> new IllegalStateException("Failed to compute relative path of file: " + parentDir));
    }

    private void indexFileAndParentDir(InputFile inputFile, String parentRelativePath) {
        DefaultInputDir inputDir = (DefaultInputDir)this.componentStore.getDir(this.module.key(), parentRelativePath);
        if (inputDir == null) {
            inputDir = new DefaultInputDir(this.module.key(), parentRelativePath, this.batchIdGenerator.get().intValue());
            inputDir.setModuleBaseDir(this.module.getBaseDir());
            this.componentTree.index((InputComponent)inputDir, (InputComponent)this.module);
            this.defaultModuleFileSystem.add(inputDir);
        }
        this.componentTree.index((InputComponent)inputFile, (InputComponent)inputDir);
        this.defaultModuleFileSystem.add(inputFile);
    }

    private boolean accept(InputFile indexedFile) {
        for (InputFileFilter filter : this.filters) {
            if (filter.accept(indexedFile)) continue;
            LOG.debug("'{}' excluded by {}", (Object)indexedFile, (Object)filter.getClass().getName());
            return false;
        }
        return true;
    }

    private class Progress {
        private AtomicInteger indexedCount = new AtomicInteger(0);
        private AtomicInteger excludedByPatternsCount = new AtomicInteger(0);

        private Progress() {
        }

        void markAsIndexed(DefaultInputFile inputFile) {
            if (FileIndexer.this.componentStore.getFile(inputFile.getProjectRelativePath()) != null) {
                throw MessageException.of((String)("File " + inputFile + " can't be indexed twice. Please check that inclusion/exclusion patterns produce disjoint sets for main and test files"));
            }
            int count = this.indexedCount.incrementAndGet();
            FileIndexer.this.progressReport.message(count + " " + FileIndexer.pluralizeFiles(count) + " indexed...  (last one was " + inputFile.getProjectRelativePath() + ")");
        }

        void increaseExcludedByPatternsCount() {
            this.excludedByPatternsCount.incrementAndGet();
        }

        public int excludedByPatternsCount() {
            return this.excludedByPatternsCount.get();
        }

        int count() {
            return this.indexedCount.get();
        }
    }

    private class IndexFileVisitor
    implements FileVisitor<Path> {
        private final Progress status;
        private final InputFile.Type type;

        IndexFileVisitor(InputFile.Type type, Progress status) {
            this.status = status;
            this.type = type;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            Path fileName = dir.getFileName();
            if (fileName != null && fileName.toString().length() > 1 && fileName.toString().charAt(0) == '.') {
                return FileVisitResult.SKIP_SUBTREE;
            }
            if (Files.isHidden(dir)) {
                return FileVisitResult.SKIP_SUBTREE;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (!Files.isHidden(file)) {
                FileIndexer.this.tasks.add(FileIndexer.this.executorService.submit(() -> FileIndexer.this.indexFile(file, this.type, this.status)));
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            if (exc instanceof FileSystemLoopException) {
                LOG.warn("Not indexing due to symlink loop: {}", (Object)file.toFile());
                return FileVisitResult.CONTINUE;
            }
            throw exc;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            return FileVisitResult.CONTINUE;
        }
    }
}

