/*
 *  Copyright (c) 2005-2008, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 *  WSO2 Inc. licenses this file to you under the Apache License,
 *  Version 2.0 (the "License"); you may not use this file except
 *  in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package org.wso2.carbon.utils;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.impl.Constants;
import org.apache.xerces.util.SecurityManager;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.wso2.carbon.utils.dbcreator.DatabaseCreator;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.File;

public final class DBUtils {
    private static Log log = LogFactory.getLog(DBUtils.class);
    private static final int ENTITY_EXPANSION_LIMIT = 0;

    private DBUtils() {
        //disable external instantiation
    }
    
    /**
     * Converts upper case column name to lower case to match PostgreSQL.
     *
     * @param dbProductName - eg: "MySQL,Oracle,PostgreSQL". Obtained using Connection.getMetaData().getDatabaseProductName();
     * @param columnName    - table column name to be converted
     * @return
     */
    public static String getConvertedAutoGeneratedColumnName(String dbProductName,
                                                             String columnName) {
        String autoGeneratedColumnName = columnName;

        if ("PostgreSQL".equals(dbProductName)) {
            autoGeneratedColumnName = columnName.toLowerCase();
            if (log.isDebugEnabled()) {
                log.debug("Database product name is PostgreSQL. Converting column name "
                          + columnName + " to lowercase ("
                          + autoGeneratedColumnName + ").");
            }
        }
        return autoGeneratedColumnName;
    }

    /**
     * Determines whether the given type of Database can return generated keys.
     *
     * @param dbProductName - eg: "MySQL,Oracle,PostgreSQL". Obtained using Connection.getMetaData().getDatabaseProductName();
     * @return
     */
    public static boolean canReturnGeneratedKeys(String dbProductName) {
        return !dbProductName.equals("OpenEdge RDBMS");
    }

    /**
     * get the registry.xml database type
     *
     * @return database name from the registry.xml
     * @throws Exception
     */
    public static String getRegistryDBConfig() throws Exception {
        return extractDBConfigFromXMLConfigFile(CarbonUtils.getRegistryXMLPath());
    }

    /**
     * get the user-mgt.xml database type
     *
     * @return database name from the user-mgt.xml
     * @throws Exception
     */
    public static String getUserManagerDBConfig() throws Exception {
    	return extractDBConfigFromXMLConfigFile(CarbonUtils.getUserMgtXMLPath());
    }

	private static String extractDBConfigFromXMLConfigFile(String path)
			throws Exception {
		String dbConfigInfo = null;
        if (path != null) {
            File file = new File(path);
            Document doc = null;
            try {
                DocumentBuilderFactory docBuilderFactory = getSecuredDocumentBuilder();
                doc = docBuilderFactory.newDocumentBuilder().parse(file);
            } catch (Exception e) {
            	log.error("Failed pasring config file " + path + ". ", e);
            	throw e;
            }
            NodeList dbConfig = doc.getElementsByTagName("dbConfig");
            Element inner = (Element) dbConfig.item(0);
            NodeList urlFinder = inner.getElementsByTagName("url");
            String urlValue = urlFinder.item(0).getTextContent();
            try {
            	dbConfigInfo = DatabaseCreator.getDatabaseType(urlValue);
            } catch (Exception e) {
            	log.error("Failed to extract database type for config file " + path + ". ", e);
            	throw e;
            }
        } else {
            log.error("Cannot find config file; Invalide path: null");
        }
        
        return dbConfigInfo;
	}

    private static DocumentBuilderFactory getSecuredDocumentBuilder() {

        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        dbf.setNamespaceAware(true);
        dbf.setXIncludeAware(false);
        dbf.setExpandEntityReferences(false);
        try {
            dbf.setFeature(Constants.SAX_FEATURE_PREFIX + Constants.EXTERNAL_GENERAL_ENTITIES_FEATURE, false);
            dbf.setFeature(Constants.SAX_FEATURE_PREFIX + Constants.EXTERNAL_PARAMETER_ENTITIES_FEATURE, false);
            dbf.setFeature(Constants.XERCES_FEATURE_PREFIX + Constants.LOAD_EXTERNAL_DTD_FEATURE, false);
            dbf.setFeature(XMLConstants.FEATURE_SECURE_PROCESSING, true);
        } catch (ParserConfigurationException e) {
            log.error(
                    "Failed to load XML Processor Feature " + Constants.EXTERNAL_GENERAL_ENTITIES_FEATURE + " or " +
                            Constants.EXTERNAL_PARAMETER_ENTITIES_FEATURE + " or " + Constants.LOAD_EXTERNAL_DTD_FEATURE);
        }

        SecurityManager securityManager = new SecurityManager();
        securityManager.setEntityExpansionLimit(ENTITY_EXPANSION_LIMIT);
        dbf.setAttribute(Constants.XERCES_PROPERTY_PREFIX + Constants.SECURITY_MANAGER_PROPERTY, securityManager);

        return dbf;
    }

}
