/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure containing summary information about resource drifts for a stack instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackInstanceResourceDriftsSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<StackInstanceResourceDriftsSummary.Builder, StackInstanceResourceDriftsSummary> {
    private static final SdkField<String> STACK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackId").getter(getter(StackInstanceResourceDriftsSummary::stackId)).setter(setter(Builder::stackId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackId").build()).build();

    private static final SdkField<String> LOGICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogicalResourceId").getter(getter(StackInstanceResourceDriftsSummary::logicalResourceId))
            .setter(setter(Builder::logicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalResourceId").build()).build();

    private static final SdkField<String> PHYSICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhysicalResourceId").getter(getter(StackInstanceResourceDriftsSummary::physicalResourceId))
            .setter(setter(Builder::physicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhysicalResourceId").build())
            .build();

    private static final SdkField<List<PhysicalResourceIdContextKeyValuePair>> PHYSICAL_RESOURCE_ID_CONTEXT_FIELD = SdkField
            .<List<PhysicalResourceIdContextKeyValuePair>> builder(MarshallingType.LIST)
            .memberName("PhysicalResourceIdContext")
            .getter(getter(StackInstanceResourceDriftsSummary::physicalResourceIdContext))
            .setter(setter(Builder::physicalResourceIdContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhysicalResourceIdContext").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PhysicalResourceIdContextKeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(PhysicalResourceIdContextKeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(StackInstanceResourceDriftsSummary::resourceType))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<List<PropertyDifference>> PROPERTY_DIFFERENCES_FIELD = SdkField
            .<List<PropertyDifference>> builder(MarshallingType.LIST)
            .memberName("PropertyDifferences")
            .getter(getter(StackInstanceResourceDriftsSummary::propertyDifferences))
            .setter(setter(Builder::propertyDifferences))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PropertyDifferences").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PropertyDifference> builder(MarshallingType.SDK_POJO)
                                            .constructor(PropertyDifference::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STACK_RESOURCE_DRIFT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackResourceDriftStatus")
            .getter(getter(StackInstanceResourceDriftsSummary::stackResourceDriftStatusAsString))
            .setter(setter(Builder::stackResourceDriftStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackResourceDriftStatus").build())
            .build();

    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(StackInstanceResourceDriftsSummary::timestamp))
            .setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_ID_FIELD,
            LOGICAL_RESOURCE_ID_FIELD, PHYSICAL_RESOURCE_ID_FIELD, PHYSICAL_RESOURCE_ID_CONTEXT_FIELD, RESOURCE_TYPE_FIELD,
            PROPERTY_DIFFERENCES_FIELD, STACK_RESOURCE_DRIFT_STATUS_FIELD, TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stackId;

    private final String logicalResourceId;

    private final String physicalResourceId;

    private final List<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext;

    private final String resourceType;

    private final List<PropertyDifference> propertyDifferences;

    private final String stackResourceDriftStatus;

    private final Instant timestamp;

    private StackInstanceResourceDriftsSummary(BuilderImpl builder) {
        this.stackId = builder.stackId;
        this.logicalResourceId = builder.logicalResourceId;
        this.physicalResourceId = builder.physicalResourceId;
        this.physicalResourceIdContext = builder.physicalResourceIdContext;
        this.resourceType = builder.resourceType;
        this.propertyDifferences = builder.propertyDifferences;
        this.stackResourceDriftStatus = builder.stackResourceDriftStatus;
        this.timestamp = builder.timestamp;
    }

    /**
     * <p>
     * The ID of the stack instance.
     * </p>
     * 
     * @return The ID of the stack instance.
     */
    public final String stackId() {
        return stackId;
    }

    /**
     * <p>
     * The logical name of the resource specified in the template.
     * </p>
     * 
     * @return The logical name of the resource specified in the template.
     */
    public final String logicalResourceId() {
        return logicalResourceId;
    }

    /**
     * <p>
     * The name or unique identifier that corresponds to a physical instance ID of a resource supported by
     * CloudFormation.
     * </p>
     * 
     * @return The name or unique identifier that corresponds to a physical instance ID of a resource supported by
     *         CloudFormation.
     */
    public final String physicalResourceId() {
        return physicalResourceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the PhysicalResourceIdContext property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasPhysicalResourceIdContext() {
        return physicalResourceIdContext != null && !(physicalResourceIdContext instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Context information that enables CloudFormation to uniquely identify a resource. CloudFormation uses context
     * key-value pairs in cases where a resource's logical and physical IDs aren't enough to uniquely identify that
     * resource. Each context key-value pair specifies a unique resource that contains the targeted resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPhysicalResourceIdContext} method.
     * </p>
     * 
     * @return Context information that enables CloudFormation to uniquely identify a resource. CloudFormation uses
     *         context key-value pairs in cases where a resource's logical and physical IDs aren't enough to uniquely
     *         identify that resource. Each context key-value pair specifies a unique resource that contains the
     *         targeted resource.
     */
    public final List<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext() {
        return physicalResourceIdContext;
    }

    /**
     * <p>
     * Type of resource. For more information, go to <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
     * Web Services Resource Types Reference</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * 
     * @return Type of resource. For more information, go to <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
     *         >Amazon Web Services Resource Types Reference</a> in the <i>CloudFormation User Guide</i>.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the PropertyDifferences property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPropertyDifferences() {
        return propertyDifferences != null && !(propertyDifferences instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Status of the actual configuration of the resource compared to its expected configuration. These will be present
     * only for resources whose <code>StackInstanceResourceDriftStatus</code> is <code>MODIFIED</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPropertyDifferences} method.
     * </p>
     * 
     * @return Status of the actual configuration of the resource compared to its expected configuration. These will be
     *         present only for resources whose <code>StackInstanceResourceDriftStatus</code> is <code>MODIFIED</code>.
     */
    public final List<PropertyDifference> propertyDifferences() {
        return propertyDifferences;
    }

    /**
     * <p>
     * The drift status of the resource in a stack instance.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DELETED</code>: The resource differs from its expected template configuration in that the resource has been
     * deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFIED</code>: One or more resource properties differ from their expected template values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The resource's actual configuration matches its expected template configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: CloudFormation doesn't currently return this value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #stackResourceDriftStatus} will return {@link StackResourceDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #stackResourceDriftStatusAsString}.
     * </p>
     * 
     * @return The drift status of the resource in a stack instance.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DELETED</code>: The resource differs from its expected template configuration in that the resource
     *         has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFIED</code>: One or more resource properties differ from their expected template values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The resource's actual configuration matches its expected template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: CloudFormation doesn't currently return this value.
     *         </p>
     *         </li>
     * @see StackResourceDriftStatus
     */
    public final StackResourceDriftStatus stackResourceDriftStatus() {
        return StackResourceDriftStatus.fromValue(stackResourceDriftStatus);
    }

    /**
     * <p>
     * The drift status of the resource in a stack instance.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DELETED</code>: The resource differs from its expected template configuration in that the resource has been
     * deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFIED</code>: One or more resource properties differ from their expected template values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The resource's actual configuration matches its expected template configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: CloudFormation doesn't currently return this value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #stackResourceDriftStatus} will return {@link StackResourceDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #stackResourceDriftStatusAsString}.
     * </p>
     * 
     * @return The drift status of the resource in a stack instance.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DELETED</code>: The resource differs from its expected template configuration in that the resource
     *         has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFIED</code>: One or more resource properties differ from their expected template values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The resource's actual configuration matches its expected template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: CloudFormation doesn't currently return this value.
     *         </p>
     *         </li>
     * @see StackResourceDriftStatus
     */
    public final String stackResourceDriftStatusAsString() {
        return stackResourceDriftStatus;
    }

    /**
     * <p>
     * Time at which the stack instance drift detection operation was initiated.
     * </p>
     * 
     * @return Time at which the stack instance drift detection operation was initiated.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackId());
        hashCode = 31 * hashCode + Objects.hashCode(logicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(physicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasPhysicalResourceIdContext() ? physicalResourceIdContext() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasPropertyDifferences() ? propertyDifferences() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stackResourceDriftStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackInstanceResourceDriftsSummary)) {
            return false;
        }
        StackInstanceResourceDriftsSummary other = (StackInstanceResourceDriftsSummary) obj;
        return Objects.equals(stackId(), other.stackId()) && Objects.equals(logicalResourceId(), other.logicalResourceId())
                && Objects.equals(physicalResourceId(), other.physicalResourceId())
                && hasPhysicalResourceIdContext() == other.hasPhysicalResourceIdContext()
                && Objects.equals(physicalResourceIdContext(), other.physicalResourceIdContext())
                && Objects.equals(resourceType(), other.resourceType())
                && hasPropertyDifferences() == other.hasPropertyDifferences()
                && Objects.equals(propertyDifferences(), other.propertyDifferences())
                && Objects.equals(stackResourceDriftStatusAsString(), other.stackResourceDriftStatusAsString())
                && Objects.equals(timestamp(), other.timestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackInstanceResourceDriftsSummary").add("StackId", stackId())
                .add("LogicalResourceId", logicalResourceId()).add("PhysicalResourceId", physicalResourceId())
                .add("PhysicalResourceIdContext", hasPhysicalResourceIdContext() ? physicalResourceIdContext() : null)
                .add("ResourceType", resourceType())
                .add("PropertyDifferences", hasPropertyDifferences() ? propertyDifferences() : null)
                .add("StackResourceDriftStatus", stackResourceDriftStatusAsString()).add("Timestamp", timestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackId":
            return Optional.ofNullable(clazz.cast(stackId()));
        case "LogicalResourceId":
            return Optional.ofNullable(clazz.cast(logicalResourceId()));
        case "PhysicalResourceId":
            return Optional.ofNullable(clazz.cast(physicalResourceId()));
        case "PhysicalResourceIdContext":
            return Optional.ofNullable(clazz.cast(physicalResourceIdContext()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "PropertyDifferences":
            return Optional.ofNullable(clazz.cast(propertyDifferences()));
        case "StackResourceDriftStatus":
            return Optional.ofNullable(clazz.cast(stackResourceDriftStatusAsString()));
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackInstanceResourceDriftsSummary, T> g) {
        return obj -> g.apply((StackInstanceResourceDriftsSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackInstanceResourceDriftsSummary> {
        /**
         * <p>
         * The ID of the stack instance.
         * </p>
         * 
         * @param stackId
         *        The ID of the stack instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackId(String stackId);

        /**
         * <p>
         * The logical name of the resource specified in the template.
         * </p>
         * 
         * @param logicalResourceId
         *        The logical name of the resource specified in the template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalResourceId(String logicalResourceId);

        /**
         * <p>
         * The name or unique identifier that corresponds to a physical instance ID of a resource supported by
         * CloudFormation.
         * </p>
         * 
         * @param physicalResourceId
         *        The name or unique identifier that corresponds to a physical instance ID of a resource supported by
         *        CloudFormation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalResourceId(String physicalResourceId);

        /**
         * <p>
         * Context information that enables CloudFormation to uniquely identify a resource. CloudFormation uses context
         * key-value pairs in cases where a resource's logical and physical IDs aren't enough to uniquely identify that
         * resource. Each context key-value pair specifies a unique resource that contains the targeted resource.
         * </p>
         * 
         * @param physicalResourceIdContext
         *        Context information that enables CloudFormation to uniquely identify a resource. CloudFormation uses
         *        context key-value pairs in cases where a resource's logical and physical IDs aren't enough to uniquely
         *        identify that resource. Each context key-value pair specifies a unique resource that contains the
         *        targeted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalResourceIdContext(Collection<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext);

        /**
         * <p>
         * Context information that enables CloudFormation to uniquely identify a resource. CloudFormation uses context
         * key-value pairs in cases where a resource's logical and physical IDs aren't enough to uniquely identify that
         * resource. Each context key-value pair specifies a unique resource that contains the targeted resource.
         * </p>
         * 
         * @param physicalResourceIdContext
         *        Context information that enables CloudFormation to uniquely identify a resource. CloudFormation uses
         *        context key-value pairs in cases where a resource's logical and physical IDs aren't enough to uniquely
         *        identify that resource. Each context key-value pair specifies a unique resource that contains the
         *        targeted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalResourceIdContext(PhysicalResourceIdContextKeyValuePair... physicalResourceIdContext);

        /**
         * <p>
         * Context information that enables CloudFormation to uniquely identify a resource. CloudFormation uses context
         * key-value pairs in cases where a resource's logical and physical IDs aren't enough to uniquely identify that
         * resource. Each context key-value pair specifies a unique resource that contains the targeted resource.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudformation.model.PhysicalResourceIdContextKeyValuePair.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.cloudformation.model.PhysicalResourceIdContextKeyValuePair#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudformation.model.PhysicalResourceIdContextKeyValuePair.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #physicalResourceIdContext(List<PhysicalResourceIdContextKeyValuePair>)}.
         * 
         * @param physicalResourceIdContext
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudformation.model.PhysicalResourceIdContextKeyValuePair.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #physicalResourceIdContext(java.util.Collection<PhysicalResourceIdContextKeyValuePair>)
         */
        Builder physicalResourceIdContext(Consumer<PhysicalResourceIdContextKeyValuePair.Builder>... physicalResourceIdContext);

        /**
         * <p>
         * Type of resource. For more information, go to <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         * >Amazon Web Services Resource Types Reference</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * 
         * @param resourceType
         *        Type of resource. For more information, go to <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >Amazon Web Services Resource Types Reference</a> in the <i>CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * Status of the actual configuration of the resource compared to its expected configuration. These will be
         * present only for resources whose <code>StackInstanceResourceDriftStatus</code> is <code>MODIFIED</code>.
         * </p>
         * 
         * @param propertyDifferences
         *        Status of the actual configuration of the resource compared to its expected configuration. These will
         *        be present only for resources whose <code>StackInstanceResourceDriftStatus</code> is
         *        <code>MODIFIED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propertyDifferences(Collection<PropertyDifference> propertyDifferences);

        /**
         * <p>
         * Status of the actual configuration of the resource compared to its expected configuration. These will be
         * present only for resources whose <code>StackInstanceResourceDriftStatus</code> is <code>MODIFIED</code>.
         * </p>
         * 
         * @param propertyDifferences
         *        Status of the actual configuration of the resource compared to its expected configuration. These will
         *        be present only for resources whose <code>StackInstanceResourceDriftStatus</code> is
         *        <code>MODIFIED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propertyDifferences(PropertyDifference... propertyDifferences);

        /**
         * <p>
         * Status of the actual configuration of the resource compared to its expected configuration. These will be
         * present only for resources whose <code>StackInstanceResourceDriftStatus</code> is <code>MODIFIED</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudformation.model.PropertyDifference.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.cloudformation.model.PropertyDifference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudformation.model.PropertyDifference.Builder#build()} is called
         * immediately and its result is passed to {@link #propertyDifferences(List<PropertyDifference>)}.
         * 
         * @param propertyDifferences
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudformation.model.PropertyDifference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #propertyDifferences(java.util.Collection<PropertyDifference>)
         */
        Builder propertyDifferences(Consumer<PropertyDifference.Builder>... propertyDifferences);

        /**
         * <p>
         * The drift status of the resource in a stack instance.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DELETED</code>: The resource differs from its expected template configuration in that the resource has
         * been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFIED</code>: One or more resource properties differ from their expected template values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The resource's actual configuration matches its expected template configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: CloudFormation doesn't currently return this value.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackResourceDriftStatus
         *        The drift status of the resource in a stack instance.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DELETED</code>: The resource differs from its expected template configuration in that the
         *        resource has been deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFIED</code>: One or more resource properties differ from their expected template values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The resource's actual configuration matches its expected template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: CloudFormation doesn't currently return this value.
         *        </p>
         *        </li>
         * @see StackResourceDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackResourceDriftStatus
         */
        Builder stackResourceDriftStatus(String stackResourceDriftStatus);

        /**
         * <p>
         * The drift status of the resource in a stack instance.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DELETED</code>: The resource differs from its expected template configuration in that the resource has
         * been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFIED</code>: One or more resource properties differ from their expected template values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The resource's actual configuration matches its expected template configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: CloudFormation doesn't currently return this value.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackResourceDriftStatus
         *        The drift status of the resource in a stack instance.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DELETED</code>: The resource differs from its expected template configuration in that the
         *        resource has been deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFIED</code>: One or more resource properties differ from their expected template values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The resource's actual configuration matches its expected template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: CloudFormation doesn't currently return this value.
         *        </p>
         *        </li>
         * @see StackResourceDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackResourceDriftStatus
         */
        Builder stackResourceDriftStatus(StackResourceDriftStatus stackResourceDriftStatus);

        /**
         * <p>
         * Time at which the stack instance drift detection operation was initiated.
         * </p>
         * 
         * @param timestamp
         *        Time at which the stack instance drift detection operation was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);
    }

    static final class BuilderImpl implements Builder {
        private String stackId;

        private String logicalResourceId;

        private String physicalResourceId;

        private List<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext = DefaultSdkAutoConstructList.getInstance();

        private String resourceType;

        private List<PropertyDifference> propertyDifferences = DefaultSdkAutoConstructList.getInstance();

        private String stackResourceDriftStatus;

        private Instant timestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(StackInstanceResourceDriftsSummary model) {
            stackId(model.stackId);
            logicalResourceId(model.logicalResourceId);
            physicalResourceId(model.physicalResourceId);
            physicalResourceIdContext(model.physicalResourceIdContext);
            resourceType(model.resourceType);
            propertyDifferences(model.propertyDifferences);
            stackResourceDriftStatus(model.stackResourceDriftStatus);
            timestamp(model.timestamp);
        }

        public final String getStackId() {
            return stackId;
        }

        public final void setStackId(String stackId) {
            this.stackId = stackId;
        }

        @Override
        public final Builder stackId(String stackId) {
            this.stackId = stackId;
            return this;
        }

        public final String getLogicalResourceId() {
            return logicalResourceId;
        }

        public final void setLogicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
        }

        @Override
        public final Builder logicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
            return this;
        }

        public final String getPhysicalResourceId() {
            return physicalResourceId;
        }

        public final void setPhysicalResourceId(String physicalResourceId) {
            this.physicalResourceId = physicalResourceId;
        }

        @Override
        public final Builder physicalResourceId(String physicalResourceId) {
            this.physicalResourceId = physicalResourceId;
            return this;
        }

        public final List<PhysicalResourceIdContextKeyValuePair.Builder> getPhysicalResourceIdContext() {
            List<PhysicalResourceIdContextKeyValuePair.Builder> result = PhysicalResourceIdContextCopier
                    .copyToBuilder(this.physicalResourceIdContext);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPhysicalResourceIdContext(
                Collection<PhysicalResourceIdContextKeyValuePair.BuilderImpl> physicalResourceIdContext) {
            this.physicalResourceIdContext = PhysicalResourceIdContextCopier.copyFromBuilder(physicalResourceIdContext);
        }

        @Override
        public final Builder physicalResourceIdContext(Collection<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext) {
            this.physicalResourceIdContext = PhysicalResourceIdContextCopier.copy(physicalResourceIdContext);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder physicalResourceIdContext(PhysicalResourceIdContextKeyValuePair... physicalResourceIdContext) {
            physicalResourceIdContext(Arrays.asList(physicalResourceIdContext));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder physicalResourceIdContext(
                Consumer<PhysicalResourceIdContextKeyValuePair.Builder>... physicalResourceIdContext) {
            physicalResourceIdContext(Stream.of(physicalResourceIdContext)
                    .map(c -> PhysicalResourceIdContextKeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final List<PropertyDifference.Builder> getPropertyDifferences() {
            List<PropertyDifference.Builder> result = PropertyDifferencesCopier.copyToBuilder(this.propertyDifferences);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPropertyDifferences(Collection<PropertyDifference.BuilderImpl> propertyDifferences) {
            this.propertyDifferences = PropertyDifferencesCopier.copyFromBuilder(propertyDifferences);
        }

        @Override
        public final Builder propertyDifferences(Collection<PropertyDifference> propertyDifferences) {
            this.propertyDifferences = PropertyDifferencesCopier.copy(propertyDifferences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder propertyDifferences(PropertyDifference... propertyDifferences) {
            propertyDifferences(Arrays.asList(propertyDifferences));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder propertyDifferences(Consumer<PropertyDifference.Builder>... propertyDifferences) {
            propertyDifferences(Stream.of(propertyDifferences).map(c -> PropertyDifference.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getStackResourceDriftStatus() {
            return stackResourceDriftStatus;
        }

        public final void setStackResourceDriftStatus(String stackResourceDriftStatus) {
            this.stackResourceDriftStatus = stackResourceDriftStatus;
        }

        @Override
        public final Builder stackResourceDriftStatus(String stackResourceDriftStatus) {
            this.stackResourceDriftStatus = stackResourceDriftStatus;
            return this;
        }

        @Override
        public final Builder stackResourceDriftStatus(StackResourceDriftStatus stackResourceDriftStatus) {
            this.stackResourceDriftStatus(stackResourceDriftStatus == null ? null : stackResourceDriftStatus.toString());
            return this;
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        @Override
        public StackInstanceResourceDriftsSummary build() {
            return new StackInstanceResourceDriftsSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
