/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directconnect.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a link aggregation group (LAG).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateLagResponse extends DirectConnectResponse implements
        ToCopyableBuilder<CreateLagResponse.Builder, CreateLagResponse> {
    private static final SdkField<String> CONNECTIONS_BANDWIDTH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::connectionsBandwidth)).setter(setter(Builder::connectionsBandwidth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("connectionsBandwidth").build())
            .build();

    private static final SdkField<Integer> NUMBER_OF_CONNECTIONS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateLagResponse::numberOfConnections)).setter(setter(Builder::numberOfConnections))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numberOfConnections").build())
            .build();

    private static final SdkField<String> LAG_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::lagId)).setter(setter(Builder::lagId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lagId").build()).build();

    private static final SdkField<String> OWNER_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::ownerAccount)).setter(setter(Builder::ownerAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ownerAccount").build()).build();

    private static final SdkField<String> LAG_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::lagName)).setter(setter(Builder::lagName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lagName").build()).build();

    private static final SdkField<String> LAG_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::lagStateAsString)).setter(setter(Builder::lagState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lagState").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<Integer> MINIMUM_LINKS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateLagResponse::minimumLinks)).setter(setter(Builder::minimumLinks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minimumLinks").build()).build();

    private static final SdkField<String> AWS_DEVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::awsDevice)).setter(setter(Builder::awsDevice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsDevice").build()).build();

    private static final SdkField<String> AWS_DEVICE_V2_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::awsDeviceV2)).setter(setter(Builder::awsDeviceV2))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsDeviceV2").build()).build();

    private static final SdkField<List<Connection>> CONNECTIONS_FIELD = SdkField
            .<List<Connection>> builder(MarshallingType.LIST)
            .getter(getter(CreateLagResponse::connections))
            .setter(setter(Builder::connections))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("connections").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Connection> builder(MarshallingType.SDK_POJO)
                                            .constructor(Connection::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ALLOWS_HOSTED_CONNECTIONS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateLagResponse::allowsHostedConnections)).setter(setter(Builder::allowsHostedConnections))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowsHostedConnections").build())
            .build();

    private static final SdkField<Boolean> JUMBO_FRAME_CAPABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateLagResponse::jumboFrameCapable)).setter(setter(Builder::jumboFrameCapable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jumboFrameCapable").build()).build();

    private static final SdkField<String> HAS_LOGICAL_REDUNDANCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::hasLogicalRedundancyAsString)).setter(setter(Builder::hasLogicalRedundancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hasLogicalRedundancy").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateLagResponse::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PROVIDER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateLagResponse::providerName)).setter(setter(Builder::providerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("providerName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONNECTIONS_BANDWIDTH_FIELD,
            NUMBER_OF_CONNECTIONS_FIELD, LAG_ID_FIELD, OWNER_ACCOUNT_FIELD, LAG_NAME_FIELD, LAG_STATE_FIELD, LOCATION_FIELD,
            REGION_FIELD, MINIMUM_LINKS_FIELD, AWS_DEVICE_FIELD, AWS_DEVICE_V2_FIELD, CONNECTIONS_FIELD,
            ALLOWS_HOSTED_CONNECTIONS_FIELD, JUMBO_FRAME_CAPABLE_FIELD, HAS_LOGICAL_REDUNDANCY_FIELD, TAGS_FIELD,
            PROVIDER_NAME_FIELD));

    private final String connectionsBandwidth;

    private final Integer numberOfConnections;

    private final String lagId;

    private final String ownerAccount;

    private final String lagName;

    private final String lagState;

    private final String location;

    private final String region;

    private final Integer minimumLinks;

    private final String awsDevice;

    private final String awsDeviceV2;

    private final List<Connection> connections;

    private final Boolean allowsHostedConnections;

    private final Boolean jumboFrameCapable;

    private final String hasLogicalRedundancy;

    private final List<Tag> tags;

    private final String providerName;

    private CreateLagResponse(BuilderImpl builder) {
        super(builder);
        this.connectionsBandwidth = builder.connectionsBandwidth;
        this.numberOfConnections = builder.numberOfConnections;
        this.lagId = builder.lagId;
        this.ownerAccount = builder.ownerAccount;
        this.lagName = builder.lagName;
        this.lagState = builder.lagState;
        this.location = builder.location;
        this.region = builder.region;
        this.minimumLinks = builder.minimumLinks;
        this.awsDevice = builder.awsDevice;
        this.awsDeviceV2 = builder.awsDeviceV2;
        this.connections = builder.connections;
        this.allowsHostedConnections = builder.allowsHostedConnections;
        this.jumboFrameCapable = builder.jumboFrameCapable;
        this.hasLogicalRedundancy = builder.hasLogicalRedundancy;
        this.tags = builder.tags;
        this.providerName = builder.providerName;
    }

    /**
     * <p>
     * The individual bandwidth of the physical connections bundled by the LAG. The possible values are 1Gbps and
     * 10Gbps.
     * </p>
     * 
     * @return The individual bandwidth of the physical connections bundled by the LAG. The possible values are 1Gbps
     *         and 10Gbps.
     */
    public String connectionsBandwidth() {
        return connectionsBandwidth;
    }

    /**
     * <p>
     * The number of physical connections bundled by the LAG, up to a maximum of 10.
     * </p>
     * 
     * @return The number of physical connections bundled by the LAG, up to a maximum of 10.
     */
    public Integer numberOfConnections() {
        return numberOfConnections;
    }

    /**
     * <p>
     * The ID of the LAG.
     * </p>
     * 
     * @return The ID of the LAG.
     */
    public String lagId() {
        return lagId;
    }

    /**
     * <p>
     * The ID of the AWS account that owns the LAG.
     * </p>
     * 
     * @return The ID of the AWS account that owns the LAG.
     */
    public String ownerAccount() {
        return ownerAccount;
    }

    /**
     * <p>
     * The name of the LAG.
     * </p>
     * 
     * @return The name of the LAG.
     */
    public String lagName() {
        return lagName;
    }

    /**
     * <p>
     * The state of the LAG. The following are the possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>requested</code>: The initial state of a LAG. The LAG stays in the requested state until the Letter of
     * Authorization (LOA) is available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pending</code>: The LAG has been approved and is being initialized.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>available</code>: The network link is established and the LAG is ready for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>down</code>: The network link is down.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleting</code>: The LAG is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted</code>: The LAG is deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>unknown</code>: The state of the LAG is not available.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lagState} will
     * return {@link LagState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #lagStateAsString}.
     * </p>
     * 
     * @return The state of the LAG. The following are the possible values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>requested</code>: The initial state of a LAG. The LAG stays in the requested state until the Letter
     *         of Authorization (LOA) is available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>pending</code>: The LAG has been approved and is being initialized.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>available</code>: The network link is established and the LAG is ready for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>down</code>: The network link is down.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleting</code>: The LAG is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted</code>: The LAG is deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>unknown</code>: The state of the LAG is not available.
     *         </p>
     *         </li>
     * @see LagState
     */
    public LagState lagState() {
        return LagState.fromValue(lagState);
    }

    /**
     * <p>
     * The state of the LAG. The following are the possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>requested</code>: The initial state of a LAG. The LAG stays in the requested state until the Letter of
     * Authorization (LOA) is available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pending</code>: The LAG has been approved and is being initialized.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>available</code>: The network link is established and the LAG is ready for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>down</code>: The network link is down.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleting</code>: The LAG is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted</code>: The LAG is deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>unknown</code>: The state of the LAG is not available.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lagState} will
     * return {@link LagState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #lagStateAsString}.
     * </p>
     * 
     * @return The state of the LAG. The following are the possible values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>requested</code>: The initial state of a LAG. The LAG stays in the requested state until the Letter
     *         of Authorization (LOA) is available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>pending</code>: The LAG has been approved and is being initialized.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>available</code>: The network link is established and the LAG is ready for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>down</code>: The network link is down.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleting</code>: The LAG is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted</code>: The LAG is deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>unknown</code>: The state of the LAG is not available.
     *         </p>
     *         </li>
     * @see LagState
     */
    public String lagStateAsString() {
        return lagState;
    }

    /**
     * <p>
     * The location of the LAG.
     * </p>
     * 
     * @return The location of the LAG.
     */
    public String location() {
        return location;
    }

    /**
     * <p>
     * The AWS Region where the connection is located.
     * </p>
     * 
     * @return The AWS Region where the connection is located.
     */
    public String region() {
        return region;
    }

    /**
     * <p>
     * The minimum number of physical connections that must be operational for the LAG itself to be operational.
     * </p>
     * 
     * @return The minimum number of physical connections that must be operational for the LAG itself to be operational.
     */
    public Integer minimumLinks() {
        return minimumLinks;
    }

    /**
     * <p>
     * The AWS Direct Connect endpoint that hosts the LAG.
     * </p>
     * 
     * @return The AWS Direct Connect endpoint that hosts the LAG.
     */
    public String awsDevice() {
        return awsDevice;
    }

    /**
     * <p>
     * The AWS Direct Connect endpoint that hosts the LAG.
     * </p>
     * 
     * @return The AWS Direct Connect endpoint that hosts the LAG.
     */
    public String awsDeviceV2() {
        return awsDeviceV2;
    }

    /**
     * Returns true if the Connections property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasConnections() {
        return connections != null && !(connections instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The connections bundled by the LAG.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConnections()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The connections bundled by the LAG.
     */
    public List<Connection> connections() {
        return connections;
    }

    /**
     * <p>
     * Indicates whether the LAG can host other connections.
     * </p>
     * 
     * @return Indicates whether the LAG can host other connections.
     */
    public Boolean allowsHostedConnections() {
        return allowsHostedConnections;
    }

    /**
     * <p>
     * Indicates whether jumbo frames (9001 MTU) are supported.
     * </p>
     * 
     * @return Indicates whether jumbo frames (9001 MTU) are supported.
     */
    public Boolean jumboFrameCapable() {
        return jumboFrameCapable;
    }

    /**
     * <p>
     * Indicates whether the LAG supports a secondary BGP peer in the same address family (IPv4/IPv6).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #hasLogicalRedundancy} will return {@link HasLogicalRedundancy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #hasLogicalRedundancyAsString}.
     * </p>
     * 
     * @return Indicates whether the LAG supports a secondary BGP peer in the same address family (IPv4/IPv6).
     * @see HasLogicalRedundancy
     */
    public HasLogicalRedundancy hasLogicalRedundancy() {
        return HasLogicalRedundancy.fromValue(hasLogicalRedundancy);
    }

    /**
     * <p>
     * Indicates whether the LAG supports a secondary BGP peer in the same address family (IPv4/IPv6).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #hasLogicalRedundancy} will return {@link HasLogicalRedundancy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #hasLogicalRedundancyAsString}.
     * </p>
     * 
     * @return Indicates whether the LAG supports a secondary BGP peer in the same address family (IPv4/IPv6).
     * @see HasLogicalRedundancy
     */
    public String hasLogicalRedundancyAsString() {
        return hasLogicalRedundancy;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags associated with the LAG.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags associated with the LAG.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the service provider associated with the LAG.
     * </p>
     * 
     * @return The name of the service provider associated with the LAG.
     */
    public String providerName() {
        return providerName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(connectionsBandwidth());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfConnections());
        hashCode = 31 * hashCode + Objects.hashCode(lagId());
        hashCode = 31 * hashCode + Objects.hashCode(ownerAccount());
        hashCode = 31 * hashCode + Objects.hashCode(lagName());
        hashCode = 31 * hashCode + Objects.hashCode(lagStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(minimumLinks());
        hashCode = 31 * hashCode + Objects.hashCode(awsDevice());
        hashCode = 31 * hashCode + Objects.hashCode(awsDeviceV2());
        hashCode = 31 * hashCode + Objects.hashCode(connections());
        hashCode = 31 * hashCode + Objects.hashCode(allowsHostedConnections());
        hashCode = 31 * hashCode + Objects.hashCode(jumboFrameCapable());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogicalRedundancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(providerName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateLagResponse)) {
            return false;
        }
        CreateLagResponse other = (CreateLagResponse) obj;
        return Objects.equals(connectionsBandwidth(), other.connectionsBandwidth())
                && Objects.equals(numberOfConnections(), other.numberOfConnections()) && Objects.equals(lagId(), other.lagId())
                && Objects.equals(ownerAccount(), other.ownerAccount()) && Objects.equals(lagName(), other.lagName())
                && Objects.equals(lagStateAsString(), other.lagStateAsString()) && Objects.equals(location(), other.location())
                && Objects.equals(region(), other.region()) && Objects.equals(minimumLinks(), other.minimumLinks())
                && Objects.equals(awsDevice(), other.awsDevice()) && Objects.equals(awsDeviceV2(), other.awsDeviceV2())
                && Objects.equals(connections(), other.connections())
                && Objects.equals(allowsHostedConnections(), other.allowsHostedConnections())
                && Objects.equals(jumboFrameCapable(), other.jumboFrameCapable())
                && Objects.equals(hasLogicalRedundancyAsString(), other.hasLogicalRedundancyAsString())
                && Objects.equals(tags(), other.tags()) && Objects.equals(providerName(), other.providerName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateLagResponse").add("ConnectionsBandwidth", connectionsBandwidth())
                .add("NumberOfConnections", numberOfConnections()).add("LagId", lagId()).add("OwnerAccount", ownerAccount())
                .add("LagName", lagName()).add("LagState", lagStateAsString()).add("Location", location())
                .add("Region", region()).add("MinimumLinks", minimumLinks()).add("AwsDevice", awsDevice())
                .add("AwsDeviceV2", awsDeviceV2()).add("Connections", connections())
                .add("AllowsHostedConnections", allowsHostedConnections()).add("JumboFrameCapable", jumboFrameCapable())
                .add("HasLogicalRedundancy", hasLogicalRedundancyAsString()).add("Tags", tags())
                .add("ProviderName", providerName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "connectionsBandwidth":
            return Optional.ofNullable(clazz.cast(connectionsBandwidth()));
        case "numberOfConnections":
            return Optional.ofNullable(clazz.cast(numberOfConnections()));
        case "lagId":
            return Optional.ofNullable(clazz.cast(lagId()));
        case "ownerAccount":
            return Optional.ofNullable(clazz.cast(ownerAccount()));
        case "lagName":
            return Optional.ofNullable(clazz.cast(lagName()));
        case "lagState":
            return Optional.ofNullable(clazz.cast(lagStateAsString()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "minimumLinks":
            return Optional.ofNullable(clazz.cast(minimumLinks()));
        case "awsDevice":
            return Optional.ofNullable(clazz.cast(awsDevice()));
        case "awsDeviceV2":
            return Optional.ofNullable(clazz.cast(awsDeviceV2()));
        case "connections":
            return Optional.ofNullable(clazz.cast(connections()));
        case "allowsHostedConnections":
            return Optional.ofNullable(clazz.cast(allowsHostedConnections()));
        case "jumboFrameCapable":
            return Optional.ofNullable(clazz.cast(jumboFrameCapable()));
        case "hasLogicalRedundancy":
            return Optional.ofNullable(clazz.cast(hasLogicalRedundancyAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "providerName":
            return Optional.ofNullable(clazz.cast(providerName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateLagResponse, T> g) {
        return obj -> g.apply((CreateLagResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DirectConnectResponse.Builder, SdkPojo, CopyableBuilder<Builder, CreateLagResponse> {
        /**
         * <p>
         * The individual bandwidth of the physical connections bundled by the LAG. The possible values are 1Gbps and
         * 10Gbps.
         * </p>
         * 
         * @param connectionsBandwidth
         *        The individual bandwidth of the physical connections bundled by the LAG. The possible values are 1Gbps
         *        and 10Gbps.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionsBandwidth(String connectionsBandwidth);

        /**
         * <p>
         * The number of physical connections bundled by the LAG, up to a maximum of 10.
         * </p>
         * 
         * @param numberOfConnections
         *        The number of physical connections bundled by the LAG, up to a maximum of 10.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfConnections(Integer numberOfConnections);

        /**
         * <p>
         * The ID of the LAG.
         * </p>
         * 
         * @param lagId
         *        The ID of the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lagId(String lagId);

        /**
         * <p>
         * The ID of the AWS account that owns the LAG.
         * </p>
         * 
         * @param ownerAccount
         *        The ID of the AWS account that owns the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerAccount(String ownerAccount);

        /**
         * <p>
         * The name of the LAG.
         * </p>
         * 
         * @param lagName
         *        The name of the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lagName(String lagName);

        /**
         * <p>
         * The state of the LAG. The following are the possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>requested</code>: The initial state of a LAG. The LAG stays in the requested state until the Letter of
         * Authorization (LOA) is available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>pending</code>: The LAG has been approved and is being initialized.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>available</code>: The network link is established and the LAG is ready for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>down</code>: The network link is down.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleting</code>: The LAG is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code>: The LAG is deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>unknown</code>: The state of the LAG is not available.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lagState
         *        The state of the LAG. The following are the possible values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>requested</code>: The initial state of a LAG. The LAG stays in the requested state until the
         *        Letter of Authorization (LOA) is available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>pending</code>: The LAG has been approved and is being initialized.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>available</code>: The network link is established and the LAG is ready for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>down</code>: The network link is down.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleting</code>: The LAG is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code>: The LAG is deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>unknown</code>: The state of the LAG is not available.
         *        </p>
         *        </li>
         * @see LagState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LagState
         */
        Builder lagState(String lagState);

        /**
         * <p>
         * The state of the LAG. The following are the possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>requested</code>: The initial state of a LAG. The LAG stays in the requested state until the Letter of
         * Authorization (LOA) is available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>pending</code>: The LAG has been approved and is being initialized.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>available</code>: The network link is established and the LAG is ready for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>down</code>: The network link is down.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleting</code>: The LAG is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code>: The LAG is deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>unknown</code>: The state of the LAG is not available.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lagState
         *        The state of the LAG. The following are the possible values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>requested</code>: The initial state of a LAG. The LAG stays in the requested state until the
         *        Letter of Authorization (LOA) is available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>pending</code>: The LAG has been approved and is being initialized.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>available</code>: The network link is established and the LAG is ready for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>down</code>: The network link is down.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleting</code>: The LAG is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code>: The LAG is deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>unknown</code>: The state of the LAG is not available.
         *        </p>
         *        </li>
         * @see LagState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LagState
         */
        Builder lagState(LagState lagState);

        /**
         * <p>
         * The location of the LAG.
         * </p>
         * 
         * @param location
         *        The location of the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * The AWS Region where the connection is located.
         * </p>
         * 
         * @param region
         *        The AWS Region where the connection is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The minimum number of physical connections that must be operational for the LAG itself to be operational.
         * </p>
         * 
         * @param minimumLinks
         *        The minimum number of physical connections that must be operational for the LAG itself to be
         *        operational.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumLinks(Integer minimumLinks);

        /**
         * <p>
         * The AWS Direct Connect endpoint that hosts the LAG.
         * </p>
         * 
         * @param awsDevice
         *        The AWS Direct Connect endpoint that hosts the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsDevice(String awsDevice);

        /**
         * <p>
         * The AWS Direct Connect endpoint that hosts the LAG.
         * </p>
         * 
         * @param awsDeviceV2
         *        The AWS Direct Connect endpoint that hosts the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsDeviceV2(String awsDeviceV2);

        /**
         * <p>
         * The connections bundled by the LAG.
         * </p>
         * 
         * @param connections
         *        The connections bundled by the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connections(Collection<Connection> connections);

        /**
         * <p>
         * The connections bundled by the LAG.
         * </p>
         * 
         * @param connections
         *        The connections bundled by the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connections(Connection... connections);

        /**
         * <p>
         * The connections bundled by the LAG.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Connection>.Builder} avoiding the need to
         * create one manually via {@link List<Connection>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Connection>.Builder#build()} is called immediately and its
         * result is passed to {@link #connections(List<Connection>)}.
         * 
         * @param connections
         *        a consumer that will call methods on {@link List<Connection>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connections(List<Connection>)
         */
        Builder connections(Consumer<Connection.Builder>... connections);

        /**
         * <p>
         * Indicates whether the LAG can host other connections.
         * </p>
         * 
         * @param allowsHostedConnections
         *        Indicates whether the LAG can host other connections.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowsHostedConnections(Boolean allowsHostedConnections);

        /**
         * <p>
         * Indicates whether jumbo frames (9001 MTU) are supported.
         * </p>
         * 
         * @param jumboFrameCapable
         *        Indicates whether jumbo frames (9001 MTU) are supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jumboFrameCapable(Boolean jumboFrameCapable);

        /**
         * <p>
         * Indicates whether the LAG supports a secondary BGP peer in the same address family (IPv4/IPv6).
         * </p>
         * 
         * @param hasLogicalRedundancy
         *        Indicates whether the LAG supports a secondary BGP peer in the same address family (IPv4/IPv6).
         * @see HasLogicalRedundancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HasLogicalRedundancy
         */
        Builder hasLogicalRedundancy(String hasLogicalRedundancy);

        /**
         * <p>
         * Indicates whether the LAG supports a secondary BGP peer in the same address family (IPv4/IPv6).
         * </p>
         * 
         * @param hasLogicalRedundancy
         *        Indicates whether the LAG supports a secondary BGP peer in the same address family (IPv4/IPv6).
         * @see HasLogicalRedundancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HasLogicalRedundancy
         */
        Builder hasLogicalRedundancy(HasLogicalRedundancy hasLogicalRedundancy);

        /**
         * <p>
         * The tags associated with the LAG.
         * </p>
         * 
         * @param tags
         *        The tags associated with the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags associated with the LAG.
         * </p>
         * 
         * @param tags
         *        The tags associated with the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags associated with the LAG.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The name of the service provider associated with the LAG.
         * </p>
         * 
         * @param providerName
         *        The name of the service provider associated with the LAG.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerName(String providerName);
    }

    static final class BuilderImpl extends DirectConnectResponse.BuilderImpl implements Builder {
        private String connectionsBandwidth;

        private Integer numberOfConnections;

        private String lagId;

        private String ownerAccount;

        private String lagName;

        private String lagState;

        private String location;

        private String region;

        private Integer minimumLinks;

        private String awsDevice;

        private String awsDeviceV2;

        private List<Connection> connections = DefaultSdkAutoConstructList.getInstance();

        private Boolean allowsHostedConnections;

        private Boolean jumboFrameCapable;

        private String hasLogicalRedundancy;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String providerName;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateLagResponse model) {
            super(model);
            connectionsBandwidth(model.connectionsBandwidth);
            numberOfConnections(model.numberOfConnections);
            lagId(model.lagId);
            ownerAccount(model.ownerAccount);
            lagName(model.lagName);
            lagState(model.lagState);
            location(model.location);
            region(model.region);
            minimumLinks(model.minimumLinks);
            awsDevice(model.awsDevice);
            awsDeviceV2(model.awsDeviceV2);
            connections(model.connections);
            allowsHostedConnections(model.allowsHostedConnections);
            jumboFrameCapable(model.jumboFrameCapable);
            hasLogicalRedundancy(model.hasLogicalRedundancy);
            tags(model.tags);
            providerName(model.providerName);
        }

        public final String getConnectionsBandwidth() {
            return connectionsBandwidth;
        }

        @Override
        public final Builder connectionsBandwidth(String connectionsBandwidth) {
            this.connectionsBandwidth = connectionsBandwidth;
            return this;
        }

        public final void setConnectionsBandwidth(String connectionsBandwidth) {
            this.connectionsBandwidth = connectionsBandwidth;
        }

        public final Integer getNumberOfConnections() {
            return numberOfConnections;
        }

        @Override
        public final Builder numberOfConnections(Integer numberOfConnections) {
            this.numberOfConnections = numberOfConnections;
            return this;
        }

        public final void setNumberOfConnections(Integer numberOfConnections) {
            this.numberOfConnections = numberOfConnections;
        }

        public final String getLagId() {
            return lagId;
        }

        @Override
        public final Builder lagId(String lagId) {
            this.lagId = lagId;
            return this;
        }

        public final void setLagId(String lagId) {
            this.lagId = lagId;
        }

        public final String getOwnerAccount() {
            return ownerAccount;
        }

        @Override
        public final Builder ownerAccount(String ownerAccount) {
            this.ownerAccount = ownerAccount;
            return this;
        }

        public final void setOwnerAccount(String ownerAccount) {
            this.ownerAccount = ownerAccount;
        }

        public final String getLagName() {
            return lagName;
        }

        @Override
        public final Builder lagName(String lagName) {
            this.lagName = lagName;
            return this;
        }

        public final void setLagName(String lagName) {
            this.lagName = lagName;
        }

        public final String getLagStateAsString() {
            return lagState;
        }

        @Override
        public final Builder lagState(String lagState) {
            this.lagState = lagState;
            return this;
        }

        @Override
        public final Builder lagState(LagState lagState) {
            this.lagState(lagState == null ? null : lagState.toString());
            return this;
        }

        public final void setLagState(String lagState) {
            this.lagState = lagState;
        }

        public final String getLocation() {
            return location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        public final Integer getMinimumLinks() {
            return minimumLinks;
        }

        @Override
        public final Builder minimumLinks(Integer minimumLinks) {
            this.minimumLinks = minimumLinks;
            return this;
        }

        public final void setMinimumLinks(Integer minimumLinks) {
            this.minimumLinks = minimumLinks;
        }

        public final String getAwsDevice() {
            return awsDevice;
        }

        @Override
        public final Builder awsDevice(String awsDevice) {
            this.awsDevice = awsDevice;
            return this;
        }

        public final void setAwsDevice(String awsDevice) {
            this.awsDevice = awsDevice;
        }

        public final String getAwsDeviceV2() {
            return awsDeviceV2;
        }

        @Override
        public final Builder awsDeviceV2(String awsDeviceV2) {
            this.awsDeviceV2 = awsDeviceV2;
            return this;
        }

        public final void setAwsDeviceV2(String awsDeviceV2) {
            this.awsDeviceV2 = awsDeviceV2;
        }

        public final Collection<Connection.Builder> getConnections() {
            return connections != null ? connections.stream().map(Connection::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder connections(Collection<Connection> connections) {
            this.connections = ConnectionListCopier.copy(connections);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder connections(Connection... connections) {
            connections(Arrays.asList(connections));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder connections(Consumer<Connection.Builder>... connections) {
            connections(Stream.of(connections).map(c -> Connection.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setConnections(Collection<Connection.BuilderImpl> connections) {
            this.connections = ConnectionListCopier.copyFromBuilder(connections);
        }

        public final Boolean getAllowsHostedConnections() {
            return allowsHostedConnections;
        }

        @Override
        public final Builder allowsHostedConnections(Boolean allowsHostedConnections) {
            this.allowsHostedConnections = allowsHostedConnections;
            return this;
        }

        public final void setAllowsHostedConnections(Boolean allowsHostedConnections) {
            this.allowsHostedConnections = allowsHostedConnections;
        }

        public final Boolean getJumboFrameCapable() {
            return jumboFrameCapable;
        }

        @Override
        public final Builder jumboFrameCapable(Boolean jumboFrameCapable) {
            this.jumboFrameCapable = jumboFrameCapable;
            return this;
        }

        public final void setJumboFrameCapable(Boolean jumboFrameCapable) {
            this.jumboFrameCapable = jumboFrameCapable;
        }

        public final String getHasLogicalRedundancyAsString() {
            return hasLogicalRedundancy;
        }

        @Override
        public final Builder hasLogicalRedundancy(String hasLogicalRedundancy) {
            this.hasLogicalRedundancy = hasLogicalRedundancy;
            return this;
        }

        @Override
        public final Builder hasLogicalRedundancy(HasLogicalRedundancy hasLogicalRedundancy) {
            this.hasLogicalRedundancy(hasLogicalRedundancy == null ? null : hasLogicalRedundancy.toString());
            return this;
        }

        public final void setHasLogicalRedundancy(String hasLogicalRedundancy) {
            this.hasLogicalRedundancy = hasLogicalRedundancy;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getProviderName() {
            return providerName;
        }

        @Override
        public final Builder providerName(String providerName) {
            this.providerName = providerName;
            return this;
        }

        public final void setProviderName(String providerName) {
            this.providerName = providerName;
        }

        @Override
        public CreateLagResponse build() {
            return new CreateLagResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
