/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.efs.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateMountTargetRequest extends EfsRequest implements
        ToCopyableBuilder<CreateMountTargetRequest.Builder, CreateMountTargetRequest> {
    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(CreateMountTargetRequest::fileSystemId))
            .setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetId").getter(getter(CreateMountTargetRequest::subnetId)).setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(CreateMountTargetRequest::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> IPV6_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Ipv6Address").getter(getter(CreateMountTargetRequest::ipv6Address)).setter(setter(Builder::ipv6Address))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6Address").build()).build();

    private static final SdkField<String> IP_ADDRESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddressType").getter(getter(CreateMountTargetRequest::ipAddressTypeAsString))
            .setter(setter(Builder::ipAddressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddressType").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(CreateMountTargetRequest::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FILE_SYSTEM_ID_FIELD,
            SUBNET_ID_FIELD, IP_ADDRESS_FIELD, IPV6_ADDRESS_FIELD, IP_ADDRESS_TYPE_FIELD, SECURITY_GROUPS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String fileSystemId;

    private final String subnetId;

    private final String ipAddress;

    private final String ipv6Address;

    private final String ipAddressType;

    private final List<String> securityGroups;

    private CreateMountTargetRequest(BuilderImpl builder) {
        super(builder);
        this.fileSystemId = builder.fileSystemId;
        this.subnetId = builder.subnetId;
        this.ipAddress = builder.ipAddress;
        this.ipv6Address = builder.ipv6Address;
        this.ipAddressType = builder.ipAddressType;
        this.securityGroups = builder.securityGroups;
    }

    /**
     * <p>
     * The ID of the file system for which to create the mount target.
     * </p>
     * 
     * @return The ID of the file system for which to create the mount target.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * The ID of the subnet to add the mount target in. For One Zone file systems, use the subnet that is associated
     * with the file system's Availability Zone.
     * </p>
     * 
     * @return The ID of the subnet to add the mount target in. For One Zone file systems, use the subnet that is
     *         associated with the file system's Availability Zone.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * If the IP address type for the mount target is IPv4, then specify the IPv4 address within the address range of
     * the specified subnet.
     * </p>
     * 
     * @return If the IP address type for the mount target is IPv4, then specify the IPv4 address within the address
     *         range of the specified subnet.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * If the IP address type for the mount target is IPv6, then specify the IPv6 address within the address range of
     * the specified subnet.
     * </p>
     * 
     * @return If the IP address type for the mount target is IPv6, then specify the IPv6 address within the address
     *         range of the specified subnet.
     */
    public final String ipv6Address() {
        return ipv6Address;
    }

    /**
     * <p>
     * Specify the type of IP address of the mount target you are creating. Options are IPv4, dual stack, or IPv6. If
     * you don’t specify an IpAddressType, then IPv4 is used.
     * </p>
     * <ul>
     * <li>
     * <p>
     * IPV4_ONLY – Create mount target with IPv4 only subnet or dual-stack subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * DUAL_STACK – Create mount target with dual-stack subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * IPV6_ONLY – Create mount target with IPv6 only subnet.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Creating IPv6 mount target only ENI in dual-stack subnet is not supported.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return Specify the type of IP address of the mount target you are creating. Options are IPv4, dual stack, or
     *         IPv6. If you don’t specify an IpAddressType, then IPv4 is used.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         IPV4_ONLY – Create mount target with IPv4 only subnet or dual-stack subnet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DUAL_STACK – Create mount target with dual-stack subnet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IPV6_ONLY – Create mount target with IPv6 only subnet.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Creating IPv6 mount target only ENI in dual-stack subnet is not supported.
     *         </p>
     * @see IpAddressType
     */
    public final IpAddressType ipAddressType() {
        return IpAddressType.fromValue(ipAddressType);
    }

    /**
     * <p>
     * Specify the type of IP address of the mount target you are creating. Options are IPv4, dual stack, or IPv6. If
     * you don’t specify an IpAddressType, then IPv4 is used.
     * </p>
     * <ul>
     * <li>
     * <p>
     * IPV4_ONLY – Create mount target with IPv4 only subnet or dual-stack subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * DUAL_STACK – Create mount target with dual-stack subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * IPV6_ONLY – Create mount target with IPv6 only subnet.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Creating IPv6 mount target only ENI in dual-stack subnet is not supported.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return Specify the type of IP address of the mount target you are creating. Options are IPv4, dual stack, or
     *         IPv6. If you don’t specify an IpAddressType, then IPv4 is used.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         IPV4_ONLY – Create mount target with IPv4 only subnet or dual-stack subnet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DUAL_STACK – Create mount target with dual-stack subnet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IPV6_ONLY – Create mount target with IPv6 only subnet.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Creating IPv6 mount target only ENI in dual-stack subnet is not supported.
     *         </p>
     * @see IpAddressType
     */
    public final String ipAddressTypeAsString() {
        return ipAddressType;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * VPC security group IDs, of the form <code>sg-xxxxxxxx</code>. These must be for the same VPC as the subnet
     * specified. The maximum number of security groups depends on account quota. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon VPC Quotas</a> in the
     * <i>Amazon VPC User Guide</i> (see the <b>Security Groups</b> table).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return VPC security group IDs, of the form <code>sg-xxxxxxxx</code>. These must be for the same VPC as the
     *         subnet specified. The maximum number of security groups depends on account quota. For more information,
     *         see <a href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon VPC
     *         Quotas</a> in the <i>Amazon VPC User Guide</i> (see the <b>Security Groups</b> table).
     */
    public final List<String> securityGroups() {
        return securityGroups;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6Address());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateMountTargetRequest)) {
            return false;
        }
        CreateMountTargetRequest other = (CreateMountTargetRequest) obj;
        return Objects.equals(fileSystemId(), other.fileSystemId()) && Objects.equals(subnetId(), other.subnetId())
                && Objects.equals(ipAddress(), other.ipAddress()) && Objects.equals(ipv6Address(), other.ipv6Address())
                && Objects.equals(ipAddressTypeAsString(), other.ipAddressTypeAsString())
                && hasSecurityGroups() == other.hasSecurityGroups() && Objects.equals(securityGroups(), other.securityGroups());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateMountTargetRequest").add("FileSystemId", fileSystemId()).add("SubnetId", subnetId())
                .add("IpAddress", ipAddress()).add("Ipv6Address", ipv6Address()).add("IpAddressType", ipAddressTypeAsString())
                .add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "Ipv6Address":
            return Optional.ofNullable(clazz.cast(ipv6Address()));
        case "IpAddressType":
            return Optional.ofNullable(clazz.cast(ipAddressTypeAsString()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FileSystemId", FILE_SYSTEM_ID_FIELD);
        map.put("SubnetId", SUBNET_ID_FIELD);
        map.put("IpAddress", IP_ADDRESS_FIELD);
        map.put("Ipv6Address", IPV6_ADDRESS_FIELD);
        map.put("IpAddressType", IP_ADDRESS_TYPE_FIELD);
        map.put("SecurityGroups", SECURITY_GROUPS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateMountTargetRequest, T> g) {
        return obj -> g.apply((CreateMountTargetRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends EfsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateMountTargetRequest> {
        /**
         * <p>
         * The ID of the file system for which to create the mount target.
         * </p>
         * 
         * @param fileSystemId
         *        The ID of the file system for which to create the mount target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * The ID of the subnet to add the mount target in. For One Zone file systems, use the subnet that is associated
         * with the file system's Availability Zone.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet to add the mount target in. For One Zone file systems, use the subnet that is
         *        associated with the file system's Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * If the IP address type for the mount target is IPv4, then specify the IPv4 address within the address range
         * of the specified subnet.
         * </p>
         * 
         * @param ipAddress
         *        If the IP address type for the mount target is IPv4, then specify the IPv4 address within the address
         *        range of the specified subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * If the IP address type for the mount target is IPv6, then specify the IPv6 address within the address range
         * of the specified subnet.
         * </p>
         * 
         * @param ipv6Address
         *        If the IP address type for the mount target is IPv6, then specify the IPv6 address within the address
         *        range of the specified subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Address(String ipv6Address);

        /**
         * <p>
         * Specify the type of IP address of the mount target you are creating. Options are IPv4, dual stack, or IPv6.
         * If you don’t specify an IpAddressType, then IPv4 is used.
         * </p>
         * <ul>
         * <li>
         * <p>
         * IPV4_ONLY – Create mount target with IPv4 only subnet or dual-stack subnet.
         * </p>
         * </li>
         * <li>
         * <p>
         * DUAL_STACK – Create mount target with dual-stack subnet.
         * </p>
         * </li>
         * <li>
         * <p>
         * IPV6_ONLY – Create mount target with IPv6 only subnet.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Creating IPv6 mount target only ENI in dual-stack subnet is not supported.
         * </p>
         * </note>
         * 
         * @param ipAddressType
         *        Specify the type of IP address of the mount target you are creating. Options are IPv4, dual stack, or
         *        IPv6. If you don’t specify an IpAddressType, then IPv4 is used.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        IPV4_ONLY – Create mount target with IPv4 only subnet or dual-stack subnet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DUAL_STACK – Create mount target with dual-stack subnet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IPV6_ONLY – Create mount target with IPv6 only subnet.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Creating IPv6 mount target only ENI in dual-stack subnet is not supported.
         *        </p>
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(String ipAddressType);

        /**
         * <p>
         * Specify the type of IP address of the mount target you are creating. Options are IPv4, dual stack, or IPv6.
         * If you don’t specify an IpAddressType, then IPv4 is used.
         * </p>
         * <ul>
         * <li>
         * <p>
         * IPV4_ONLY – Create mount target with IPv4 only subnet or dual-stack subnet.
         * </p>
         * </li>
         * <li>
         * <p>
         * DUAL_STACK – Create mount target with dual-stack subnet.
         * </p>
         * </li>
         * <li>
         * <p>
         * IPV6_ONLY – Create mount target with IPv6 only subnet.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Creating IPv6 mount target only ENI in dual-stack subnet is not supported.
         * </p>
         * </note>
         * 
         * @param ipAddressType
         *        Specify the type of IP address of the mount target you are creating. Options are IPv4, dual stack, or
         *        IPv6. If you don’t specify an IpAddressType, then IPv4 is used.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        IPV4_ONLY – Create mount target with IPv4 only subnet or dual-stack subnet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DUAL_STACK – Create mount target with dual-stack subnet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IPV6_ONLY – Create mount target with IPv6 only subnet.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Creating IPv6 mount target only ENI in dual-stack subnet is not supported.
         *        </p>
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(IpAddressType ipAddressType);

        /**
         * <p>
         * VPC security group IDs, of the form <code>sg-xxxxxxxx</code>. These must be for the same VPC as the subnet
         * specified. The maximum number of security groups depends on account quota. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon VPC Quotas</a> in the
         * <i>Amazon VPC User Guide</i> (see the <b>Security Groups</b> table).
         * </p>
         * 
         * @param securityGroups
         *        VPC security group IDs, of the form <code>sg-xxxxxxxx</code>. These must be for the same VPC as the
         *        subnet specified. The maximum number of security groups depends on account quota. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon VPC Quotas</a>
         *        in the <i>Amazon VPC User Guide</i> (see the <b>Security Groups</b> table).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * <p>
         * VPC security group IDs, of the form <code>sg-xxxxxxxx</code>. These must be for the same VPC as the subnet
         * specified. The maximum number of security groups depends on account quota. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon VPC Quotas</a> in the
         * <i>Amazon VPC User Guide</i> (see the <b>Security Groups</b> table).
         * </p>
         * 
         * @param securityGroups
         *        VPC security group IDs, of the form <code>sg-xxxxxxxx</code>. These must be for the same VPC as the
         *        subnet specified. The maximum number of security groups depends on account quota. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon VPC Quotas</a>
         *        in the <i>Amazon VPC User Guide</i> (see the <b>Security Groups</b> table).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EfsRequest.BuilderImpl implements Builder {
        private String fileSystemId;

        private String subnetId;

        private String ipAddress;

        private String ipv6Address;

        private String ipAddressType;

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateMountTargetRequest model) {
            super(model);
            fileSystemId(model.fileSystemId);
            subnetId(model.subnetId);
            ipAddress(model.ipAddress);
            ipv6Address(model.ipv6Address);
            ipAddressType(model.ipAddressType);
            securityGroups(model.securityGroups);
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getIpv6Address() {
            return ipv6Address;
        }

        public final void setIpv6Address(String ipv6Address) {
            this.ipv6Address = ipv6Address;
        }

        @Override
        public final Builder ipv6Address(String ipv6Address) {
            this.ipv6Address = ipv6Address;
            return this;
        }

        public final String getIpAddressType() {
            return ipAddressType;
        }

        public final void setIpAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
        }

        @Override
        public final Builder ipAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
            return this;
        }

        @Override
        public final Builder ipAddressType(IpAddressType ipAddressType) {
            this.ipAddressType(ipAddressType == null ? null : ipAddressType.toString());
            return this;
        }

        public final Collection<String> getSecurityGroups() {
            if (securityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroups;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupsCopier.copy(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupsCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateMountTargetRequest build() {
            return new CreateMountTargetRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
