/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.fsx.model.ActiveDirectoryErrorException;
import software.amazon.awssdk.services.fsx.model.BackupInProgressException;
import software.amazon.awssdk.services.fsx.model.BackupNotFoundException;
import software.amazon.awssdk.services.fsx.model.BackupRestoringException;
import software.amazon.awssdk.services.fsx.model.BadRequestException;
import software.amazon.awssdk.services.fsx.model.CancelDataRepositoryTaskRequest;
import software.amazon.awssdk.services.fsx.model.CancelDataRepositoryTaskResponse;
import software.amazon.awssdk.services.fsx.model.CreateBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryTaskRequest;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryTaskResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemFromBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemFromBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskEndedException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskExecutingException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskNotFoundException;
import software.amazon.awssdk.services.fsx.model.DeleteBackupRequest;
import software.amazon.awssdk.services.fsx.model.DeleteBackupResponse;
import software.amazon.awssdk.services.fsx.model.DeleteFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.DeleteFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksRequest;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemsResponse;
import software.amazon.awssdk.services.fsx.model.FSxException;
import software.amazon.awssdk.services.fsx.model.FSxRequest;
import software.amazon.awssdk.services.fsx.model.FileSystemNotFoundException;
import software.amazon.awssdk.services.fsx.model.IncompatibleParameterErrorException;
import software.amazon.awssdk.services.fsx.model.InternalServerErrorException;
import software.amazon.awssdk.services.fsx.model.InvalidExportPathException;
import software.amazon.awssdk.services.fsx.model.InvalidImportPathException;
import software.amazon.awssdk.services.fsx.model.InvalidNetworkSettingsException;
import software.amazon.awssdk.services.fsx.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.fsx.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.fsx.model.MissingFileSystemConfigurationException;
import software.amazon.awssdk.services.fsx.model.NotServiceResourceErrorException;
import software.amazon.awssdk.services.fsx.model.ResourceDoesNotSupportTaggingException;
import software.amazon.awssdk.services.fsx.model.ResourceNotFoundException;
import software.amazon.awssdk.services.fsx.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.fsx.model.TagResourceRequest;
import software.amazon.awssdk.services.fsx.model.TagResourceResponse;
import software.amazon.awssdk.services.fsx.model.UnsupportedOperationException;
import software.amazon.awssdk.services.fsx.model.UntagResourceRequest;
import software.amazon.awssdk.services.fsx.model.UntagResourceResponse;
import software.amazon.awssdk.services.fsx.model.UpdateFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.UpdateFileSystemResponse;
import software.amazon.awssdk.services.fsx.paginators.DescribeBackupsPublisher;
import software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryTasksPublisher;
import software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemsPublisher;
import software.amazon.awssdk.services.fsx.transform.CancelDataRepositoryTaskRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateDataRepositoryTaskRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileSystemFromBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeDataRepositoryTasksRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileSystemsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateFileSystemRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link FSxAsyncClient}.
 *
 * @see FSxAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFSxAsyncClient implements FSxAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultFSxAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFSxAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels an existing Amazon FSx for Lustre data repository task if that task is in either the <code>PENDING</code>
     * or <code>EXECUTING</code> state. When you cancel a task, Amazon FSx does the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Any files that FSx has already exported are not reverted.
     * </p>
     * </li>
     * <li>
     * <p>
     * FSx continues to export any files that are "in-flight" when the cancel operation is received.
     * </p>
     * </li>
     * <li>
     * <p>
     * FSx does not export any files that have not yet been exported.
     * </p>
     * </li>
     * </ul>
     *
     * @param cancelDataRepositoryTaskRequest
     *        Cancels a data repository task.
     * @return A Java Future containing the result of the CancelDataRepositoryTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>DataRepositoryTaskNotFoundException The data repository task or tasks you specified could not be
     *         found.</li>
     *         <li>DataRepositoryTaskEndedException The data repository task could not be canceled because the task has
     *         already ended.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CancelDataRepositoryTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CancelDataRepositoryTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelDataRepositoryTaskResponse> cancelDataRepositoryTask(
            CancelDataRepositoryTaskRequest cancelDataRepositoryTaskRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelDataRepositoryTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelDataRepositoryTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelDataRepositoryTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelDataRepositoryTaskRequest, CancelDataRepositoryTaskResponse>()
                            .withOperationName("CancelDataRepositoryTask")
                            .withMarshaller(new CancelDataRepositoryTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(cancelDataRepositoryTaskRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a backup of an existing Amazon FSx for Windows File Server file system. Creating regular backups for your
     * file system is a best practice that complements the replication that Amazon FSx for Windows File Server performs
     * for your file system. It also enables you to restore from user modification of data.
     * </p>
     * <p>
     * If a backup with the specified client request token exists, and the parameters match, this operation returns the
     * description of the existing backup. If a backup specified client request token exists, and the parameters don't
     * match, this operation returns <code>IncompatibleParameterError</code>. If a backup with the specified client
     * request token doesn't exist, <code>CreateBackup</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new Amazon FSx backup with an assigned ID, and an initial lifecycle state of <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the backup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateBackup</code> operation without the risk of
     * creating an extra backup. This approach can be useful when an initial call fails in a way that makes it unclear
     * whether a backup was created. If you use the same client request token and the initial call created a backup, the
     * operation returns a successful result because all the parameters are the same.
     * </p>
     * <p>
     * The <code>CreateFileSystem</code> operation returns while the backup's lifecycle state is still
     * <code>CREATING</code>. You can check the file system creation status by calling the <a>DescribeBackups</a>
     * operation, which returns the backup state along with other information.
     * </p>
     * <note>
     * <p/>
     * </note>
     *
     * @param createBackupRequest
     *        The request object for the <code>CreateBackup</code> operation.
     * @return A Java Future containing the result of the CreateBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li> <li>
     *         UnsupportedOperationException The requested operation is not supported for this resource or API.</li> 
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.
     *         </li> <li>BackupInProgressException Another backup is already under way. Wait for completion before
     *         initiating additional backups of this file system.</li> <li>IncompatibleParameterErrorException The error
     *         returned when a second request is received with the same client request token but different parameters
     *         settings. A client request token should always uniquely identify a single request.</li> <li>
     *         ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You can
     *         increase some service limits by contacting AWS Support.</li> <li>InternalServerErrorException A generic
     *         error indicating a server-side failure.</li> <li>SdkException Base class for all exceptions that can be
     *         thrown by the SDK (both service and client). Can be used for catch all scenarios.</li> <li>
     *         SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li> <li>FSxException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackupResponse> createBackup(CreateBackupRequest createBackupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackupRequest, CreateBackupResponse>()
                            .withOperationName("CreateBackup").withMarshaller(new CreateBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createBackupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon FSx for Lustre data repository task. You use data repository tasks to perform bulk operations
     * between your Amazon FSx file system and its linked data repository. An example of a data repository task is
     * exporting any data and metadata changes, including POSIX metadata, to files, directories, and symbolic links
     * (symlinks) from your FSx file system to its linked data repository. A <code>CreateDataRepositoryTask</code>
     * operation will fail if a data repository is not linked to the FSx file system. To learn more about data
     * repository tasks, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-repository-tasks.html">Using Data Repository
     * Tasks</a>. To learn more about linking a data repository to your file system, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/getting-started-step1.html">Step 1: Create Your Amazon
     * FSx for Lustre File System</a>.
     * </p>
     *
     * @param createDataRepositoryTaskRequest
     * @return A Java Future containing the result of the CreateDataRepositoryTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting AWS Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>DataRepositoryTaskExecutingException An existing data repository task is currently executing on the
     *         file system. Wait until the existing task has completed, then create the new task.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateDataRepositoryTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateDataRepositoryTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataRepositoryTaskResponse> createDataRepositoryTask(
            CreateDataRepositoryTaskRequest createDataRepositoryTaskRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataRepositoryTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataRepositoryTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataRepositoryTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataRepositoryTaskRequest, CreateDataRepositoryTaskResponse>()
                            .withOperationName("CreateDataRepositoryTask")
                            .withMarshaller(new CreateDataRepositoryTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDataRepositoryTaskRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new, empty Amazon FSx file system.
     * </p>
     * <p>
     * If a file system with the specified client request token exists and the parameters match,
     * <code>CreateFileSystem</code> returns the description of the existing file system. If a file system specified
     * client request token exists and the parameters don't match, this call returns
     * <code>IncompatibleParameterError</code>. If a file system with the specified client request token doesn't exist,
     * <code>CreateFileSystem</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty Amazon FSx file system with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires a client request token in the request that Amazon FSx uses to ensure idempotent creation.
     * This means that calling the operation multiple times with the same client request token has no effect. By using
     * the idempotent operation, you can retry a <code>CreateFileSystem</code> operation without the risk of creating an
     * extra file system. This approach can be useful when an initial call fails in a way that makes it unclear whether
     * a file system was created. Examples are if a transport level timeout occurred, or your connection was reset. If
     * you use the same client request token and the initial call created a file system, the client receives success as
     * long as the parameters are the same.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>CREATING</code>. You can check the file-system creation status by calling the <a>DescribeFileSystems</a>
     * operation, which returns the file system state along with other information.
     * </p>
     * </note>
     *
     * @param createFileSystemRequest
     *        The request object used to create a new Amazon FSx file system.
     * @return A Java Future containing the result of the CreateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>ActiveDirectoryErrorException An Active Directory error.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InvalidImportPathException The path provided for data repository import isn't valid.</li>
     *         <li>InvalidExportPathException The path provided for data repository export isn't valid.</li>
     *         <li>InvalidNetworkSettingsException One or more network settings specified in the request are invalid.
     *         <code>InvalidVpcId</code> means that the ID passed for the virtual private cloud (VPC) is invalid.
     *         <code>InvalidSubnetIds</code> returns the list of IDs for subnets that are either invalid or not part of
     *         the VPC specified. <code>InvalidSecurityGroupIds</code> returns the list of IDs for security groups that
     *         are either invalid or not part of the VPC specified.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting AWS Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>MissingFileSystemConfigurationException File system configuration is required for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateFileSystem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFileSystemResponse> createFileSystem(CreateFileSystemRequest createFileSystemRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFileSystemRequest, CreateFileSystemResponse>()
                            .withOperationName("CreateFileSystem")
                            .withMarshaller(new CreateFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createFileSystemRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon FSx file system from an existing Amazon FSx for Windows File Server backup.
     * </p>
     * <p>
     * If a file system with the specified client request token exists and the parameters match, this operation returns
     * the description of the file system. If a client request token specified by the file system exists and the
     * parameters don't match, this call returns <code>IncompatibleParameterError</code>. If a file system with the
     * specified client request token doesn't exist, this operation does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new Amazon FSx file system from backup with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Parameters like Active Directory, default share name, automatic backup, and backup settings default to the
     * parameters of the file system that was backed up, unless overridden. You can explicitly supply other settings.
     * </p>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateFileSystemFromBackup</code> call without the risk
     * of creating an extra file system. This approach can be useful when an initial call fails in a way that makes it
     * unclear whether a file system was created. Examples are if a transport level timeout occurred, or your connection
     * was reset. If you use the same client request token and the initial call created a file system, the client
     * receives success as long as the parameters are the same.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystemFromBackup</code> call returns while the file system's lifecycle state is still
     * <code>CREATING</code>. You can check the file-system creation status by calling the <a>DescribeFileSystems</a>
     * operation, which returns the file system state along with other information.
     * </p>
     * </note>
     *
     * @param createFileSystemFromBackupRequest
     *        The request object for the <code>CreateFileSystemFromBackup</code> operation.
     * @return A Java Future containing the result of the CreateFileSystemFromBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>ActiveDirectoryErrorException An Active Directory error.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InvalidNetworkSettingsException One or more network settings specified in the request are invalid.
     *         <code>InvalidVpcId</code> means that the ID passed for the virtual private cloud (VPC) is invalid.
     *         <code>InvalidSubnetIds</code> returns the list of IDs for subnets that are either invalid or not part of
     *         the VPC specified. <code>InvalidSecurityGroupIds</code> returns the list of IDs for security groups that
     *         are either invalid or not part of the VPC specified.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting AWS Support.</li>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>MissingFileSystemConfigurationException File system configuration is required for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateFileSystemFromBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileSystemFromBackup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFileSystemFromBackupResponse> createFileSystemFromBackup(
            CreateFileSystemFromBackupRequest createFileSystemFromBackupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFileSystemFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFileSystemFromBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFileSystemFromBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFileSystemFromBackupRequest, CreateFileSystemFromBackupResponse>()
                            .withOperationName("CreateFileSystemFromBackup")
                            .withMarshaller(new CreateFileSystemFromBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createFileSystemFromBackupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx for Windows File Server backup, deleting its contents. After deletion, the backup no longer
     * exists, and its data is gone.
     * </p>
     * <p>
     * The <code>DeleteBackup</code> call returns instantly. The backup will not show up in later
     * <code>DescribeBackups</code> calls.
     * </p>
     * <important>
     * <p>
     * The data in a deleted backup is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteBackupRequest
     *        The request object for <code>DeleteBackup</code> operation.
     * @return A Java Future containing the result of the DeleteBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>BackupInProgressException Another backup is already under way. Wait for completion before initiating
     *         additional backups of this file system.</li>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>BackupRestoringException You can't delete a backup while it's being used to restore a file system.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackupResponse> deleteBackup(DeleteBackupRequest deleteBackupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                            .withOperationName("DeleteBackup").withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteBackupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a file system, deleting its contents. After deletion, the file system no longer exists, and its data is
     * gone. Any existing automatic backups will also be deleted.
     * </p>
     * <p>
     * By default, when you delete an Amazon FSx for Windows File Server file system, a final backup is created upon
     * deletion. This final backup is not subject to the file system's retention policy, and must be manually deleted.
     * </p>
     * <p>
     * The <code>DeleteFileSystem</code> action returns while the file system has the <code>DELETING</code> status. You
     * can check the file system deletion status by calling the <a>DescribeFileSystems</a> action, which returns a list
     * of file systems in your account. If you pass the file system ID for a deleted file system, the
     * <a>DescribeFileSystems</a> returns a <code>FileSystemNotFound</code> error.
     * </p>
     * <note>
     * <p>
     * Deleting an Amazon FSx for Lustre file system will fail with a 400 BadRequest if a data repository task is in a
     * <code>PENDING</code> or <code>EXECUTING</code> state.
     * </p>
     * </note> <important>
     * <p>
     * The data in a deleted file system is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteFileSystemRequest
     *        The request object for <code>DeleteFileSystem</code> operation.
     * @return A Java Future containing the result of the DeleteFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting AWS Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteFileSystem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFileSystemResponse> deleteFileSystem(DeleteFileSystemRequest deleteFileSystemRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFileSystemRequest, DeleteFileSystemResponse>()
                            .withOperationName("DeleteFileSystem")
                            .withMarshaller(new DeleteFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteFileSystemRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Windows File Server backups, if a <code>BackupIds</code> value
     * is provided for that backup. Otherwise, it returns all backups owned by your AWS account in the AWS Region of the
     * endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all backups, you can optionally specify the <code>MaxResults</code> parameter to limit the number
     * of backups in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in the
     * response. In this case, send a later request with the <code>NextToken</code> request parameter set to the value
     * of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your backups. <code>DescribeBackups</code> is
     * called first without a <code>NextToken</code>value. Then the action continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of backups returned in the response of one <code>DescribeBackups</code> call and the order of backups
     * returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeBackupsRequest
     *        The request object for <code>DescribeBackups</code> operation.
     * @return A Java Future containing the result of the DescribeBackups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeBackups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBackupsResponse> describeBackups(DescribeBackupsRequest describeBackupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBackupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBackupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                            .withOperationName("DescribeBackups")
                            .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeBackupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Windows File Server backups, if a <code>BackupIds</code> value
     * is provided for that backup. Otherwise, it returns all backups owned by your AWS account in the AWS Region of the
     * endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all backups, you can optionally specify the <code>MaxResults</code> parameter to limit the number
     * of backups in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in the
     * response. In this case, send a later request with the <code>NextToken</code> request parameter set to the value
     * of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your backups. <code>DescribeBackups</code> is
     * called first without a <code>NextToken</code>value. Then the action continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of backups returned in the response of one <code>DescribeBackups</code> call and the order of backups
     * returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #describeBackups(software.amazon.awssdk.services.fsx.model.DescribeBackupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeBackupsPublisher publisher = client.describeBackupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeBackupsPublisher publisher = client.describeBackupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.fsx.model.DescribeBackupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.fsx.model.DescribeBackupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBackups(software.amazon.awssdk.services.fsx.model.DescribeBackupsRequest)} operation.</b>
     * </p>
     *
     * @param describeBackupsRequest
     *        The request object for <code>DescribeBackups</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeBackups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeBackupsPublisher describeBackupsPaginator(DescribeBackupsRequest describeBackupsRequest) {
        return new DescribeBackupsPublisher(this, applyPaginatorUserAgent(describeBackupsRequest));
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre data repository tasks, if one or more
     * <code>TaskIds</code> values are provided in the request, or if filters are used in the request. You can use
     * filters to narrow the response to include just tasks for specific file systems, or tasks in a specific lifecycle
     * state. Otherwise, it returns all data repository tasks owned by your AWS account in the AWS Region of the
     * endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all tasks, you can paginate the response by using the optional <code>MaxResults</code> parameter
     * to limit the number of tasks returned in a response. If more tasks remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     *
     * @param describeDataRepositoryTasksRequest
     * @return A Java Future containing the result of the DescribeDataRepositoryTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>DataRepositoryTaskNotFoundException The data repository task or tasks you specified could not be
     *         found.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeDataRepositoryTasks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDataRepositoryTasksResponse> describeDataRepositoryTasks(
            DescribeDataRepositoryTasksRequest describeDataRepositoryTasksRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDataRepositoryTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDataRepositoryTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDataRepositoryTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDataRepositoryTasksRequest, DescribeDataRepositoryTasksResponse>()
                            .withOperationName("DescribeDataRepositoryTasks")
                            .withMarshaller(new DescribeDataRepositoryTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDataRepositoryTasksRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre data repository tasks, if one or more
     * <code>TaskIds</code> values are provided in the request, or if filters are used in the request. You can use
     * filters to narrow the response to include just tasks for specific file systems, or tasks in a specific lifecycle
     * state. Otherwise, it returns all data repository tasks owned by your AWS account in the AWS Region of the
     * endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all tasks, you can paginate the response by using the optional <code>MaxResults</code> parameter
     * to limit the number of tasks returned in a response. If more tasks remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeDataRepositoryTasks(software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryTasksPublisher publisher = client.describeDataRepositoryTasksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryTasksPublisher publisher = client.describeDataRepositoryTasksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDataRepositoryTasks(software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDataRepositoryTasksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>DataRepositoryTaskNotFoundException The data repository task or tasks you specified could not be
     *         found.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeDataRepositoryTasks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryTasks"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeDataRepositoryTasksPublisher describeDataRepositoryTasksPaginator(
            DescribeDataRepositoryTasksRequest describeDataRepositoryTasksRequest) {
        return new DescribeDataRepositoryTasksPublisher(this, applyPaginatorUserAgent(describeDataRepositoryTasksRequest));
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx file systems, if a <code>FileSystemIds</code> value is provided
     * for that file system. Otherwise, it returns descriptions of all file systems owned by your AWS account in the AWS
     * Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxResults</code> parameter to
     * limit the number of descriptions in a response. If more file system descriptions remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your file system descriptions.
     * <code>DescribeFileSystems</code> is called first without a <code>NextToken</code>value. Then the action continues
     * to be called with the <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value
     * until a response has no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multicall iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFileSystemsRequest
     *        The request object for <code>DescribeFileSystems</code> operation.
     * @return A Java Future containing the result of the DescribeFileSystems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeFileSystems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFileSystemsResponse> describeFileSystems(
            DescribeFileSystemsRequest describeFileSystemsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFileSystemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFileSystemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFileSystemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFileSystemsRequest, DescribeFileSystemsResponse>()
                            .withOperationName("DescribeFileSystems")
                            .withMarshaller(new DescribeFileSystemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeFileSystemsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx file systems, if a <code>FileSystemIds</code> value is provided
     * for that file system. Otherwise, it returns descriptions of all file systems owned by your AWS account in the AWS
     * Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxResults</code> parameter to
     * limit the number of descriptions in a response. If more file system descriptions remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your file system descriptions.
     * <code>DescribeFileSystems</code> is called first without a <code>NextToken</code>value. Then the action continues
     * to be called with the <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value
     * until a response has no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multicall iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFileSystems(software.amazon.awssdk.services.fsx.model.DescribeFileSystemsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemsPublisher publisher = client.describeFileSystemsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemsPublisher publisher = client.describeFileSystemsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.fsx.model.DescribeFileSystemsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.fsx.model.DescribeFileSystemsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFileSystems(software.amazon.awssdk.services.fsx.model.DescribeFileSystemsRequest)} operation.</b>
     * </p>
     *
     * @param describeFileSystemsRequest
     *        The request object for <code>DescribeFileSystems</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeFileSystems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystems" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeFileSystemsPublisher describeFileSystemsPaginator(DescribeFileSystemsRequest describeFileSystemsRequest) {
        return new DescribeFileSystemsPublisher(this, applyPaginatorUserAgent(describeFileSystemsRequest));
    }

    /**
     * <p>
     * Lists tags for an Amazon FSx file systems and backups in the case of Amazon FSx for Windows File Server.
     * </p>
     * <p>
     * When retrieving all tags, you can optionally specify the <code>MaxResults</code> parameter to limit the number of
     * tags in a response. If more tags remain, Amazon FSx returns a <code>NextToken</code> value in the response. In
     * this case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your tags. <code>ListTagsForResource</code> is
     * called first without a <code>NextToken</code>value. Then the action continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of tags returned in the response of one <code>ListTagsForResource</code> call and the order of tags
     * returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param listTagsForResourceRequest
     *        The request object for <code>ListTagsForResource</code> operation.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>ResourceNotFoundException The resource specified by the Amazon Resource Name (ARN) can't be found.</li>
     *         <li>NotServiceResourceErrorException The resource specified for the tagging operation is not a resource
     *         type owned by Amazon FSx. Use the API of the relevant service to perform the operation.</li>
     *         <li>ResourceDoesNotSupportTaggingException The resource specified does not support tagging.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags an Amazon FSx resource.
     * </p>
     *
     * @param tagResourceRequest
     *        The request object for the <code>TagResource</code> operation.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>ResourceNotFoundException The resource specified by the Amazon Resource Name (ARN) can't be found.</li>
     *         <li>NotServiceResourceErrorException The resource specified for the tagging operation is not a resource
     *         type owned by Amazon FSx. Use the API of the relevant service to perform the operation.</li>
     *         <li>ResourceDoesNotSupportTaggingException The resource specified does not support tagging.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action removes a tag from an Amazon FSx resource.
     * </p>
     *
     * @param untagResourceRequest
     *        The request object for <code>UntagResource</code> action.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>ResourceNotFoundException The resource specified by the Amazon Resource Name (ARN) can't be found.</li>
     *         <li>NotServiceResourceErrorException The resource specified for the tagging operation is not a resource
     *         type owned by Amazon FSx. Use the API of the relevant service to perform the operation.</li>
     *         <li>ResourceDoesNotSupportTaggingException The resource specified does not support tagging.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a file system configuration.
     * </p>
     *
     * @param updateFileSystemRequest
     *        The request object for the <code>UpdateFileSystem</code> operation.
     * @return A Java Future containing the result of the UpdateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>MissingFileSystemConfigurationException File system configuration is required for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UpdateFileSystem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFileSystemResponse> updateFileSystem(UpdateFileSystemRequest updateFileSystemRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFileSystemRequest, UpdateFileSystemResponse>()
                            .withOperationName("UpdateFileSystem")
                            .withMarshaller(new UpdateFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateFileSystemRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FSxException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLimitExceeded")
                                .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotServiceResourceError")
                                .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperation")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidImportPath")
                                .exceptionBuilderSupplier(InvalidImportPathException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNetworkSettings")
                                .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration")
                                .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting")
                                .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemNotFound")
                                .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupRestoring")
                                .exceptionBuilderSupplier(BackupRestoringException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncompatibleParameterError")
                                .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded")
                                .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupInProgress")
                                .exceptionBuilderSupplier(BackupInProgressException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidExportPath")
                                .exceptionBuilderSupplier(InvalidExportPathException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging")
                                .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound")
                                .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupNotFound")
                                .exceptionBuilderSupplier(BackupNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveDirectoryError")
                                .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequest")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
    }

    private <T extends FSxRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
