/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that Amazon FSx uses to join the Windows File Server instance to your self-managed (including
 * on-premises) Microsoft Active Directory (AD) directory.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SelfManagedActiveDirectoryConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<SelfManagedActiveDirectoryConfiguration.Builder, SelfManagedActiveDirectoryConfiguration> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SelfManagedActiveDirectoryConfiguration::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> ORGANIZATIONAL_UNIT_DISTINGUISHED_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(SelfManagedActiveDirectoryConfiguration::organizationalUnitDistinguishedName))
            .setter(setter(Builder::organizationalUnitDistinguishedName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("OrganizationalUnitDistinguishedName").build()).build();

    private static final SdkField<String> FILE_SYSTEM_ADMINISTRATORS_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(SelfManagedActiveDirectoryConfiguration::fileSystemAdministratorsGroup))
            .setter(setter(Builder::fileSystemAdministratorsGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemAdministratorsGroup")
                    .build()).build();

    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SelfManagedActiveDirectoryConfiguration::userName)).setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SelfManagedActiveDirectoryConfiguration::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<List<String>> DNS_IPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(SelfManagedActiveDirectoryConfiguration::dnsIps))
            .setter(setter(Builder::dnsIps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            ORGANIZATIONAL_UNIT_DISTINGUISHED_NAME_FIELD, FILE_SYSTEM_ADMINISTRATORS_GROUP_FIELD, USER_NAME_FIELD,
            PASSWORD_FIELD, DNS_IPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String domainName;

    private final String organizationalUnitDistinguishedName;

    private final String fileSystemAdministratorsGroup;

    private final String userName;

    private final String password;

    private final List<String> dnsIps;

    private SelfManagedActiveDirectoryConfiguration(BuilderImpl builder) {
        this.domainName = builder.domainName;
        this.organizationalUnitDistinguishedName = builder.organizationalUnitDistinguishedName;
        this.fileSystemAdministratorsGroup = builder.fileSystemAdministratorsGroup;
        this.userName = builder.userName;
        this.password = builder.password;
        this.dnsIps = builder.dnsIps;
    }

    /**
     * <p>
     * The fully qualified domain name of the self-managed AD directory, such as <code>corp.example.com</code>.
     * </p>
     * 
     * @return The fully qualified domain name of the self-managed AD directory, such as <code>corp.example.com</code>.
     */
    public String domainName() {
        return domainName;
    }

    /**
     * <p>
     * (Optional) The fully qualified distinguished name of the organizational unit within your self-managed AD
     * directory that the Windows File Server instance will join. Amazon FSx only accepts OU as the direct parent of the
     * file system. An example is <code>OU=FSx,DC=yourdomain,DC=corp,DC=com</code>. To learn more, see <a
     * href="https://tools.ietf.org/html/rfc2253">RFC 2253</a>. If none is provided, the FSx file system is created in
     * the default location of your self-managed AD directory.
     * </p>
     * <important>
     * <p>
     * Only Organizational Unit (OU) objects can be the direct parent of the file system that you're creating.
     * </p>
     * </important>
     * 
     * @return (Optional) The fully qualified distinguished name of the organizational unit within your self-managed AD
     *         directory that the Windows File Server instance will join. Amazon FSx only accepts OU as the direct
     *         parent of the file system. An example is <code>OU=FSx,DC=yourdomain,DC=corp,DC=com</code>. To learn more,
     *         see <a href="https://tools.ietf.org/html/rfc2253">RFC 2253</a>. If none is provided, the FSx file system
     *         is created in the default location of your self-managed AD directory. </p> <important>
     *         <p>
     *         Only Organizational Unit (OU) objects can be the direct parent of the file system that you're creating.
     *         </p>
     */
    public String organizationalUnitDistinguishedName() {
        return organizationalUnitDistinguishedName;
    }

    /**
     * <p>
     * (Optional) The name of the domain group whose members are granted administrative privileges for the file system.
     * Administrative privileges include taking ownership of files and folders, setting audit controls (audit ACLs) on
     * files and folders, and administering the file system remotely by using the FSx Remote PowerShell. The group that
     * you specify must already exist in your domain. If you don't provide one, your AD domain's Domain Admins group is
     * used.
     * </p>
     * 
     * @return (Optional) The name of the domain group whose members are granted administrative privileges for the file
     *         system. Administrative privileges include taking ownership of files and folders, setting audit controls
     *         (audit ACLs) on files and folders, and administering the file system remotely by using the FSx Remote
     *         PowerShell. The group that you specify must already exist in your domain. If you don't provide one, your
     *         AD domain's Domain Admins group is used.
     */
    public String fileSystemAdministratorsGroup() {
        return fileSystemAdministratorsGroup;
    }

    /**
     * <p>
     * The user name for the service account on your self-managed AD domain that Amazon FSx will use to join to your AD
     * domain. This account must have the permission to join computers to the domain in the organizational unit provided
     * in <code>OrganizationalUnitDistinguishedName</code>, or in the default location of your AD domain.
     * </p>
     * 
     * @return The user name for the service account on your self-managed AD domain that Amazon FSx will use to join to
     *         your AD domain. This account must have the permission to join computers to the domain in the
     *         organizational unit provided in <code>OrganizationalUnitDistinguishedName</code>, or in the default
     *         location of your AD domain.
     */
    public String userName() {
        return userName;
    }

    /**
     * <p>
     * The password for the service account on your self-managed AD domain that Amazon FSx will use to join to your AD
     * domain.
     * </p>
     * 
     * @return The password for the service account on your self-managed AD domain that Amazon FSx will use to join to
     *         your AD domain.
     */
    public String password() {
        return password;
    }

    /**
     * Returns true if the DnsIps property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDnsIps() {
        return dnsIps != null && !(dnsIps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of up to two IP addresses of DNS servers or domain controllers in the self-managed AD directory. The IP
     * addresses need to be either in the same VPC CIDR range as the one in which your Amazon FSx file system is being
     * created, or in the private IP version 4 (IPv4) address ranges, as specified in <a
     * href="http://www.faqs.org/rfcs/rfc1918.html">RFC 1918</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * 10.0.0.0 - 10.255.255.255 (10/8 prefix)
     * </p>
     * </li>
     * <li>
     * <p>
     * 172.16.0.0 - 172.31.255.255 (172.16/12 prefix)
     * </p>
     * </li>
     * <li>
     * <p>
     * 192.168.0.0 - 192.168.255.255 (192.168/16 prefix)
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDnsIps()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of up to two IP addresses of DNS servers or domain controllers in the self-managed AD directory.
     *         The IP addresses need to be either in the same VPC CIDR range as the one in which your Amazon FSx file
     *         system is being created, or in the private IP version 4 (IPv4) address ranges, as specified in <a
     *         href="http://www.faqs.org/rfcs/rfc1918.html">RFC 1918</a>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         10.0.0.0 - 10.255.255.255 (10/8 prefix)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         172.16.0.0 - 172.31.255.255 (172.16/12 prefix)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         192.168.0.0 - 192.168.255.255 (192.168/16 prefix)
     *         </p>
     *         </li>
     */
    public List<String> dnsIps() {
        return dnsIps;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(organizationalUnitDistinguishedName());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemAdministratorsGroup());
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(dnsIps());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SelfManagedActiveDirectoryConfiguration)) {
            return false;
        }
        SelfManagedActiveDirectoryConfiguration other = (SelfManagedActiveDirectoryConfiguration) obj;
        return Objects.equals(domainName(), other.domainName())
                && Objects.equals(organizationalUnitDistinguishedName(), other.organizationalUnitDistinguishedName())
                && Objects.equals(fileSystemAdministratorsGroup(), other.fileSystemAdministratorsGroup())
                && Objects.equals(userName(), other.userName()) && Objects.equals(password(), other.password())
                && Objects.equals(dnsIps(), other.dnsIps());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SelfManagedActiveDirectoryConfiguration").add("DomainName", domainName())
                .add("OrganizationalUnitDistinguishedName", organizationalUnitDistinguishedName())
                .add("FileSystemAdministratorsGroup", fileSystemAdministratorsGroup()).add("UserName", userName())
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***").add("DnsIps", dnsIps()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "OrganizationalUnitDistinguishedName":
            return Optional.ofNullable(clazz.cast(organizationalUnitDistinguishedName()));
        case "FileSystemAdministratorsGroup":
            return Optional.ofNullable(clazz.cast(fileSystemAdministratorsGroup()));
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "DnsIps":
            return Optional.ofNullable(clazz.cast(dnsIps()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SelfManagedActiveDirectoryConfiguration, T> g) {
        return obj -> g.apply((SelfManagedActiveDirectoryConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SelfManagedActiveDirectoryConfiguration> {
        /**
         * <p>
         * The fully qualified domain name of the self-managed AD directory, such as <code>corp.example.com</code>.
         * </p>
         * 
         * @param domainName
         *        The fully qualified domain name of the self-managed AD directory, such as
         *        <code>corp.example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * (Optional) The fully qualified distinguished name of the organizational unit within your self-managed AD
         * directory that the Windows File Server instance will join. Amazon FSx only accepts OU as the direct parent of
         * the file system. An example is <code>OU=FSx,DC=yourdomain,DC=corp,DC=com</code>. To learn more, see <a
         * href="https://tools.ietf.org/html/rfc2253">RFC 2253</a>. If none is provided, the FSx file system is created
         * in the default location of your self-managed AD directory.
         * </p>
         * <important>
         * <p>
         * Only Organizational Unit (OU) objects can be the direct parent of the file system that you're creating.
         * </p>
         * </important>
         * 
         * @param organizationalUnitDistinguishedName
         *        (Optional) The fully qualified distinguished name of the organizational unit within your self-managed
         *        AD directory that the Windows File Server instance will join. Amazon FSx only accepts OU as the direct
         *        parent of the file system. An example is <code>OU=FSx,DC=yourdomain,DC=corp,DC=com</code>. To learn
         *        more, see <a href="https://tools.ietf.org/html/rfc2253">RFC 2253</a>. If none is provided, the FSx
         *        file system is created in the default location of your self-managed AD directory. </p> <important>
         *        <p>
         *        Only Organizational Unit (OU) objects can be the direct parent of the file system that you're
         *        creating.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitDistinguishedName(String organizationalUnitDistinguishedName);

        /**
         * <p>
         * (Optional) The name of the domain group whose members are granted administrative privileges for the file
         * system. Administrative privileges include taking ownership of files and folders, setting audit controls
         * (audit ACLs) on files and folders, and administering the file system remotely by using the FSx Remote
         * PowerShell. The group that you specify must already exist in your domain. If you don't provide one, your AD
         * domain's Domain Admins group is used.
         * </p>
         * 
         * @param fileSystemAdministratorsGroup
         *        (Optional) The name of the domain group whose members are granted administrative privileges for the
         *        file system. Administrative privileges include taking ownership of files and folders, setting audit
         *        controls (audit ACLs) on files and folders, and administering the file system remotely by using the
         *        FSx Remote PowerShell. The group that you specify must already exist in your domain. If you don't
         *        provide one, your AD domain's Domain Admins group is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemAdministratorsGroup(String fileSystemAdministratorsGroup);

        /**
         * <p>
         * The user name for the service account on your self-managed AD domain that Amazon FSx will use to join to your
         * AD domain. This account must have the permission to join computers to the domain in the organizational unit
         * provided in <code>OrganizationalUnitDistinguishedName</code>, or in the default location of your AD domain.
         * </p>
         * 
         * @param userName
         *        The user name for the service account on your self-managed AD domain that Amazon FSx will use to join
         *        to your AD domain. This account must have the permission to join computers to the domain in the
         *        organizational unit provided in <code>OrganizationalUnitDistinguishedName</code>, or in the default
         *        location of your AD domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        /**
         * <p>
         * The password for the service account on your self-managed AD domain that Amazon FSx will use to join to your
         * AD domain.
         * </p>
         * 
         * @param password
         *        The password for the service account on your self-managed AD domain that Amazon FSx will use to join
         *        to your AD domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * A list of up to two IP addresses of DNS servers or domain controllers in the self-managed AD directory. The
         * IP addresses need to be either in the same VPC CIDR range as the one in which your Amazon FSx file system is
         * being created, or in the private IP version 4 (IPv4) address ranges, as specified in <a
         * href="http://www.faqs.org/rfcs/rfc1918.html">RFC 1918</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * 10.0.0.0 - 10.255.255.255 (10/8 prefix)
         * </p>
         * </li>
         * <li>
         * <p>
         * 172.16.0.0 - 172.31.255.255 (172.16/12 prefix)
         * </p>
         * </li>
         * <li>
         * <p>
         * 192.168.0.0 - 192.168.255.255 (192.168/16 prefix)
         * </p>
         * </li>
         * </ul>
         * 
         * @param dnsIps
         *        A list of up to two IP addresses of DNS servers or domain controllers in the self-managed AD
         *        directory. The IP addresses need to be either in the same VPC CIDR range as the one in which your
         *        Amazon FSx file system is being created, or in the private IP version 4 (IPv4) address ranges, as
         *        specified in <a href="http://www.faqs.org/rfcs/rfc1918.html">RFC 1918</a>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        10.0.0.0 - 10.255.255.255 (10/8 prefix)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        172.16.0.0 - 172.31.255.255 (172.16/12 prefix)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        192.168.0.0 - 192.168.255.255 (192.168/16 prefix)
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIps(Collection<String> dnsIps);

        /**
         * <p>
         * A list of up to two IP addresses of DNS servers or domain controllers in the self-managed AD directory. The
         * IP addresses need to be either in the same VPC CIDR range as the one in which your Amazon FSx file system is
         * being created, or in the private IP version 4 (IPv4) address ranges, as specified in <a
         * href="http://www.faqs.org/rfcs/rfc1918.html">RFC 1918</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * 10.0.0.0 - 10.255.255.255 (10/8 prefix)
         * </p>
         * </li>
         * <li>
         * <p>
         * 172.16.0.0 - 172.31.255.255 (172.16/12 prefix)
         * </p>
         * </li>
         * <li>
         * <p>
         * 192.168.0.0 - 192.168.255.255 (192.168/16 prefix)
         * </p>
         * </li>
         * </ul>
         * 
         * @param dnsIps
         *        A list of up to two IP addresses of DNS servers or domain controllers in the self-managed AD
         *        directory. The IP addresses need to be either in the same VPC CIDR range as the one in which your
         *        Amazon FSx file system is being created, or in the private IP version 4 (IPv4) address ranges, as
         *        specified in <a href="http://www.faqs.org/rfcs/rfc1918.html">RFC 1918</a>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        10.0.0.0 - 10.255.255.255 (10/8 prefix)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        172.16.0.0 - 172.31.255.255 (172.16/12 prefix)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        192.168.0.0 - 192.168.255.255 (192.168/16 prefix)
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIps(String... dnsIps);
    }

    static final class BuilderImpl implements Builder {
        private String domainName;

        private String organizationalUnitDistinguishedName;

        private String fileSystemAdministratorsGroup;

        private String userName;

        private String password;

        private List<String> dnsIps = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SelfManagedActiveDirectoryConfiguration model) {
            domainName(model.domainName);
            organizationalUnitDistinguishedName(model.organizationalUnitDistinguishedName);
            fileSystemAdministratorsGroup(model.fileSystemAdministratorsGroup);
            userName(model.userName);
            password(model.password);
            dnsIps(model.dnsIps);
        }

        public final String getDomainName() {
            return domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        public final String getOrganizationalUnitDistinguishedName() {
            return organizationalUnitDistinguishedName;
        }

        @Override
        public final Builder organizationalUnitDistinguishedName(String organizationalUnitDistinguishedName) {
            this.organizationalUnitDistinguishedName = organizationalUnitDistinguishedName;
            return this;
        }

        public final void setOrganizationalUnitDistinguishedName(String organizationalUnitDistinguishedName) {
            this.organizationalUnitDistinguishedName = organizationalUnitDistinguishedName;
        }

        public final String getFileSystemAdministratorsGroup() {
            return fileSystemAdministratorsGroup;
        }

        @Override
        public final Builder fileSystemAdministratorsGroup(String fileSystemAdministratorsGroup) {
            this.fileSystemAdministratorsGroup = fileSystemAdministratorsGroup;
            return this;
        }

        public final void setFileSystemAdministratorsGroup(String fileSystemAdministratorsGroup) {
            this.fileSystemAdministratorsGroup = fileSystemAdministratorsGroup;
        }

        public final String getUserName() {
            return userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        public final String getPassword() {
            return password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        public final Collection<String> getDnsIps() {
            return dnsIps;
        }

        @Override
        public final Builder dnsIps(Collection<String> dnsIps) {
            this.dnsIps = DnsIpsCopier.copy(dnsIps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIps(String... dnsIps) {
            dnsIps(Arrays.asList(dnsIps));
            return this;
        }

        public final void setDnsIps(Collection<String> dnsIps) {
            this.dnsIps = DnsIpsCopier.copy(dnsIps);
        }

        @Override
        public SelfManagedActiveDirectoryConfiguration build() {
            return new SelfManagedActiveDirectoryConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
