/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.services.sqs.internal.batchmanager;

import java.time.Duration;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.services.sqs.SqsAsyncClient;
import software.amazon.awssdk.services.sqs.internal.batchmanager.QueueAttributesManager;
import software.amazon.awssdk.services.sqs.internal.batchmanager.ReceiveSqsMessageHelper;
import software.amazon.awssdk.services.sqs.internal.batchmanager.ResponseBatchConfiguration;
import software.amazon.awssdk.services.sqs.model.Message;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@SdkInternalApi
public class ReceiveQueueBuffer
implements SdkAutoCloseable {
    private final ScheduledExecutorService executor;
    private final SqsAsyncClient sqsClient;
    private final ResponseBatchConfiguration config;
    private final String queueUrl;
    private final QueueAttributesManager queueAttributesManager;
    private final Queue<ReceiveSqsMessageHelper> finishedTasks = new ConcurrentLinkedQueue<ReceiveSqsMessageHelper>();
    private final Queue<FutureRequestWrapper> futures = new ConcurrentLinkedQueue<FutureRequestWrapper>();
    private final AtomicInteger inflightReceiveMessageBatches = new AtomicInteger(0);
    private final AtomicBoolean shutDown = new AtomicBoolean(false);
    private final AtomicBoolean processingFutures = new AtomicBoolean(false);

    private ReceiveQueueBuffer(Builder builder) {
        this.executor = builder.executor;
        this.sqsClient = builder.sqsClient;
        this.config = builder.config;
        this.queueUrl = builder.queueUrl;
        this.queueAttributesManager = builder.queueAttributesManager;
    }

    public static Builder builder() {
        return new Builder();
    }

    public void receiveMessage(CompletableFuture<ReceiveMessageResponse> receiveMessageFuture, int numMessages) {
        this.futures.add(new FutureRequestWrapper(receiveMessageFuture, numMessages));
        this.satisfyFuturesFromBuffer();
        this.spawnMoreReceiveTasks();
    }

    public boolean isShutDown() {
        return this.shutDown.get();
    }

    public void close() {
        if (this.shutDown.compareAndSet(false, true)) {
            while (!this.finishedTasks.isEmpty()) {
                ReceiveSqsMessageHelper batch = this.finishedTasks.poll();
                if (this.inflightReceiveMessageBatches.get() > 0) {
                    this.inflightReceiveMessageBatches.decrementAndGet();
                }
                if (batch == null) continue;
                batch.clear();
            }
            this.futures.forEach(futureWrapper -> {
                if (!futureWrapper.getFuture().isDone()) {
                    futureWrapper.getFuture().completeExceptionally(new CancellationException("Shutdown in progress"));
                }
            });
            this.futures.clear();
        }
    }

    private void spawnMoreReceiveTasks() {
        if (this.shutDown.get()) {
            return;
        }
        int desiredBatches = this.determineDesiredBatches();
        if (this.finishedTasks.size() >= desiredBatches) {
            return;
        }
        if (!this.finishedTasks.isEmpty() && this.finishedTasks.size() + this.inflightReceiveMessageBatches.get() >= desiredBatches) {
            return;
        }
        this.queueAttributesManager.getVisibilityTimeout().thenAccept(visibilityTimeout -> {
            int max = Math.max(this.config.maxInflightReceiveBatches(), 1);
            int toSpawn = max - this.inflightReceiveMessageBatches.get();
            if (toSpawn > 0) {
                ReceiveSqsMessageHelper receiveSqsMessageHelper = new ReceiveSqsMessageHelper(this.queueUrl, this.sqsClient, (Duration)visibilityTimeout, this.config);
                this.inflightReceiveMessageBatches.incrementAndGet();
                receiveSqsMessageHelper.asyncReceiveMessage().whenComplete((response, exception) -> this.reportBatchFinished((ReceiveSqsMessageHelper)response));
            }
        });
    }

    private int determineDesiredBatches() {
        int desiredBatches = Math.max(this.config.maxDoneReceiveBatches(), 1);
        int totalRequested = this.futures.stream().mapToInt(FutureRequestWrapper::getRequestedSize).sum();
        int batchesNeededToFulfillFutures = (int)Math.ceil((float)totalRequested / 10.0f);
        desiredBatches = Math.min(batchesNeededToFulfillFutures, desiredBatches);
        return desiredBatches;
    }

    private void fulfillFuture(FutureRequestWrapper futureWrapper) {
        ReceiveSqsMessageHelper peekedMessage = this.finishedTasks.peek();
        LinkedList<Message> messages = new LinkedList<Message>();
        Throwable exception = peekedMessage.getException();
        boolean batchDone = false;
        if (exception != null) {
            futureWrapper.getFuture().completeExceptionally(exception);
            this.finishedTasks.poll();
            return;
        }
        for (int numRetrieved = 0; numRetrieved < futureWrapper.getRequestedSize(); ++numRetrieved) {
            Message msg = peekedMessage.removeMessage();
            if (msg != null) {
                messages.add(msg);
                continue;
            }
            batchDone = true;
            break;
        }
        boolean bl = batchDone = batchDone || peekedMessage.isEmpty();
        if (batchDone) {
            this.finishedTasks.poll();
        }
        futureWrapper.getFuture().complete((ReceiveMessageResponse)((Object)ReceiveMessageResponse.builder().messages(messages).build()));
    }

    private void satisfyFuturesFromBuffer() {
        if (!this.processingFutures.compareAndSet(false, true)) {
            return;
        }
        try {
            do {
                this.futures.removeIf(future -> {
                    if (future.getFuture().isDone()) {
                        return true;
                    }
                    if (!this.finishedTasks.isEmpty()) {
                        this.fulfillFuture((FutureRequestWrapper)future);
                        return true;
                    }
                    return false;
                });
            } while (!this.futures.isEmpty() && !this.finishedTasks.isEmpty());
        }
        finally {
            this.processingFutures.set(false);
        }
    }

    private void reportBatchFinished(ReceiveSqsMessageHelper batch) {
        this.finishedTasks.offer(batch);
        this.inflightReceiveMessageBatches.decrementAndGet();
        this.satisfyFuturesFromBuffer();
        this.spawnMoreReceiveTasks();
    }

    public static class Builder {
        private ScheduledExecutorService executor;
        private SqsAsyncClient sqsClient;
        private ResponseBatchConfiguration config;
        private String queueUrl;
        private QueueAttributesManager queueAttributesManager;

        public Builder executor(ScheduledExecutorService executor) {
            this.executor = executor;
            return this;
        }

        public Builder sqsClient(SqsAsyncClient sqsClient) {
            this.sqsClient = sqsClient;
            return this;
        }

        public Builder config(ResponseBatchConfiguration config) {
            this.config = config;
            return this;
        }

        public Builder queueUrl(String queueUrl) {
            this.queueUrl = queueUrl;
            return this;
        }

        public Builder queueAttributesManager(QueueAttributesManager queueAttributesManager) {
            this.queueAttributesManager = queueAttributesManager;
            return this;
        }

        public ReceiveQueueBuffer build() {
            return new ReceiveQueueBuffer(this);
        }
    }

    private static class FutureRequestWrapper {
        private final CompletableFuture<ReceiveMessageResponse> future;
        private final int requestedSize;

        FutureRequestWrapper(CompletableFuture<ReceiveMessageResponse> future, int requestedSize) {
            this.future = future;
            this.requestedSize = requestedSize;
        }

        public CompletableFuture<ReceiveMessageResponse> getFuture() {
            return this.future;
        }

        public int getRequestedSize() {
            return this.requestedSize;
        }
    }
}

