/*
  * JBoss, Home of Professional Open Source
  * Copyright 2007, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.integration.web;

import java.security.Principal;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.jacc.PolicyContext;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import org.jboss.security.AuthorizationManager;
import org.jboss.security.SecurityContext;
import org.jboss.security.audit.AuditLevel;
import org.jboss.security.authorization.AuthorizationContext;
import org.jboss.security.authorization.AuthorizationException;
import org.jboss.security.authorization.ResourceKeys;
import org.jboss.security.authorization.resources.WebResource; 
import org.jboss.security.callbacks.SecurityContextCallbackHandler;
import org.jboss.security.identity.RoleGroup;
import org.jboss.security.integration.ejb.SecurityHelper;

//$Id$

/**
 *  Helper Class for Web Authorization
 *  @author Anil.Saldhana@redhat.com
 *  @since  Nov 26, 2007 
 *  @version $Revision$
 */
public class WebAuthorizationHelper extends SecurityHelper
{ 
   private boolean enableAudit = true;
   
   public WebAuthorizationHelper(SecurityContext sc, boolean enableAudit)
   { 
      super(sc);
      this.enableAudit = enableAudit;
   } 
   
   public boolean checkResourcePermission(Map<String, Object> contextMap,
         ServletRequest request, ServletResponse response,
         Subject callerSubject, AuthorizationManager authzMgr, String canonicalRequestURI)
   {
      boolean isAuthorized = false; 
      
      WebResource webResource = new WebResource(Collections.unmodifiableMap(contextMap));
      webResource.setPolicyContextID(PolicyContext.getContextID());
      webResource.setServletRequest(request);
      webResource.setServletResponse(response);
      webResource.setCallerSubject(callerSubject);
      webResource.setCanonicalRequestURI(canonicalRequestURI);
      
      SecurityContextCallbackHandler sch = new SecurityContextCallbackHandler(this.securityContext); 
      RoleGroup callerRoles = authzMgr.getSubjectRoles(callerSubject, sch);
      
      try
      {
         int permit = authzMgr.authorize(webResource, callerSubject, callerRoles);
         isAuthorized = (permit == AuthorizationContext.PERMIT);
         String level = (permit == AuthorizationContext.PERMIT ? AuditLevel.SUCCESS : AuditLevel.FAILURE);
         if(this.enableAudit)
           this.authorizationAudit(level,webResource, null); 
      }
      catch (AuthorizationException e)
      {
         isAuthorized = false; 
         if(log.isTraceEnabled()) 
            log.trace("hasResourcePermission check failed:"+e.getLocalizedMessage()); 
         if(this.enableAudit)
            authorizationAudit(AuditLevel.ERROR,webResource,e); 
      }
      return isAuthorized; 
   }
   
   public boolean hasRole(String roleName, Principal principal, String servletName,
         Set<Principal> principalRoles, AuthorizationManager authzMgr)
   { 
      boolean hasTheRole = false;
      Map<String,Object> map =  new HashMap<String,Object>();  
      map.put(ResourceKeys.ROLENAME, roleName); 
      map.put(ResourceKeys.ROLEREF_PERM_CHECK, Boolean.TRUE);  
      map.put(ResourceKeys.PRINCIPAL_ROLES, principalRoles);

      map.put(ResourceKeys.POLICY_REGISTRATION, authzMgr);
      
      WebResource webResource = new WebResource(Collections.unmodifiableMap(map));
      webResource.setPolicyContextID(PolicyContext.getContextID());
      webResource.setPrincipal(principal);
      webResource.setServletName(servletName);
      
      //Get the authenticated subject
      Subject subject = null;
      try
      {
         subject = SecurityActions.getActiveSubject();
      }
      catch( Exception e)
      {
         log.trace("Exception in getting subject:",e);
         subject = securityContext.getUtil().getSubject();
      }
      webResource.setCallerSubject(subject);
      SecurityContextCallbackHandler sch = new SecurityContextCallbackHandler(this.securityContext); 
      RoleGroup callerRoles = authzMgr.getSubjectRoles(subject, sch);
      
      try
      {
         int permit = authzMgr.authorize(webResource, subject, callerRoles);
         hasTheRole = (permit == AuthorizationContext.PERMIT);
         String level = (hasTheRole ? AuditLevel.SUCCESS : AuditLevel.FAILURE);
         if(this.enableAudit)
           this.authorizationAudit(level,webResource, null);
      }
      catch (AuthorizationException e)
      {
         hasTheRole = false; 
         if(log.isTraceEnabled()) 
            log.trace("hasRole check failed:"+e.getLocalizedMessage()); 
         if(this.enableAudit)
            authorizationAudit(AuditLevel.ERROR,webResource,e); 
      }
      return hasTheRole; 
   }
   
   public boolean hasUserDataPermission(Map<String,Object> contextMap,
         ServletRequest request, ServletResponse response,
         AuthorizationManager authzMgr)
   {
      boolean hasPerm =  false;   
      contextMap.put(ResourceKeys.POLICY_REGISTRATION, authzMgr);
      
      WebResource webResource = new WebResource(Collections.unmodifiableMap(contextMap)); 
      webResource.setPolicyContextID(PolicyContext.getContextID());
      webResource.setServletRequest(request);
      webResource.setServletResponse(response);
      
      //Get the authenticated subject
      Subject subject = null;
      try
      {
         subject = SecurityActions.getActiveSubject();
      }
      catch( Exception e)
      {
         log.trace("Exception in getting subject:",e);
         subject = securityContext.getUtil().getSubject();
      }
      webResource.setCallerSubject(subject);
      SecurityContextCallbackHandler sch = new SecurityContextCallbackHandler(this.securityContext); 
      RoleGroup callerRoles = authzMgr.getSubjectRoles(subject, sch);
      
      try
      {
         int permit = authzMgr.authorize(webResource, subject, callerRoles);
         hasPerm = (permit == AuthorizationContext.PERMIT);
         String level = (hasPerm ? AuditLevel.SUCCESS : AuditLevel.FAILURE);
         if(this.enableAudit)
            this.authorizationAudit(level,webResource, null);
      }
      catch (AuthorizationException e)
      {
         hasPerm = false; 
         if(log.isTraceEnabled()) 
            log.trace("hasRole check failed:"+e.getLocalizedMessage()); 
         if(this.enableAudit)
            authorizationAudit(AuditLevel.ERROR,webResource,e); 
      }
      return hasPerm;
   }
}