/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.plugins;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.security.Principal;
import java.security.acl.Group;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;

import org.jboss.logging.Logger;
import org.jboss.security.AuthorizationManager;
import org.jboss.security.SecurityConstants;
import org.jboss.security.auth.callback.SecurityAssociationHandler;
import org.jboss.security.authorization.AuthorizationException;
import org.jboss.security.authorization.EntitlementHolder;
import org.jboss.security.authorization.Resource;
import org.jboss.security.identity.Identity;
import org.jboss.security.identity.RoleGroup;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.util.CachePolicy;

//$Id: AuthorizationManagerService.java 58710 2006-11-28 17:32:06Z anil.saldhana@jboss.com $

/**
 *  Service that provides Authorization capabilities.
 *  The service defaults to the
 *  org.jboss.security.plugins.AuthorizationManager implementation but
 *  this can be changed via the authorizationManagerClassName property.
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Jan 3, 2006 
 *  @version $Revision: 58710 $
 *  TODO: THIS CLASS NEEDS TO GO ASAP
 */
public class AuthorizationManagerService 
extends ServiceMBeanSupport
implements AuthorizationManagerServiceMBean
{ 
   /** The log4j interface */
   private static Logger log = Logger.getLogger(AuthorizationManagerService.class);
   private static String authorizationMgrClassName = SecurityConstants.DEFAULT_AUTHORIZATION_CLASS;
   private static Class<?> authorizationMgrClass = JBossAuthorizationManager.class;
   
   private static Hashtable<String,AuthorizationManager> authorizationManagersMap 
       = new Hashtable<String,AuthorizationManager>();
    
   /** The JAAS CallbackHandler interface implementation to use */
   private static String callbackHandlerClassName = "org.jboss.security.auth.callback.SecurityAssociationHandler";
   private static Class<?> callbackHandlerClass = SecurityAssociationHandler.class;  
   
   private RuntimeException rte = new RuntimeException("Call the method on the authorization manager");

   /**
    * @see AuthorizationManagerServiceMBean#setAuthorizationManagerClassName(String)
    */
   public void setAuthorizationManagerClassName(String className) 
   throws ClassNotFoundException, ClassCastException
   {  
      authorizationMgrClassName = className;
      ClassLoader loader = Thread.currentThread().getContextClassLoader();
      authorizationMgrClass = loader.loadClass(authorizationMgrClassName);
      if( AuthorizationManager.class.isAssignableFrom(authorizationMgrClass) == false )
         throw new ClassCastException(authorizationMgrClass+" does not implement "+AuthorizationManager.class);
   }
   
   /** Set the default CallbackHandler implementation class name
    * @see javax.security.auth.callback.CallbackHandler
    */
   public void setCallbackHandlerClassName(String className)
      throws ClassNotFoundException
   {
      callbackHandlerClassName = className;
      ClassLoader loader = Thread.currentThread().getContextClassLoader();
      callbackHandlerClass = loader.loadClass(callbackHandlerClassName);
   } 

   /**
    * @see AuthorizationManagerServiceMBean#getAuthorizationManagerClassName()
    */
   public String getAuthorizationManagerClassName()
   {
      return authorizationMgrClassName;
   } 
   
   /**
    * @see AuthorizationManager#doesUserHaveRole(String)
    */
   public boolean doesUserHaveRole(String roleName)
   { 
      String str = "Use getAuthorizationManager method and then call doesUserHaveRole";
      throw new IllegalStateException(str);
   } 
   
   /**
    * @see AuthorizationManager#doesUserHaveRole(Principal, Set)
    */
   public boolean doesUserHaveRole(Principal principal, Set<Principal> roles)
   {
      String str = "Use getAuthorizationManager method and then call doesUserHaveRole";
      throw new IllegalStateException(str);
   } 
   
   //ServiceMBeanSupport methods
   protected void startService() throws Exception
   {  
      super.startService();
   }
   
   protected void stopService() throws Exception
   { 
      super.stopService();
   } 
   
   /** Return the set of domain roles the principal has been assigned.
    @return The Set<Principal> for the application domain roles that the
    principal has been assigned.
    */
   public Set<Principal> getUserRoles(Principal principal)
   {
      String str = "Use getAuthorizationManager method and then call getUserRoles";
      throw new IllegalStateException(str);
   }
   
   /**
    * @see AuthorizationManager#getPrincipal(Principal)
    */
   public Principal getPrincipal(Principal principal)
   { 
      String str = "Use getAuthorizationManager method and then call getPrincipal";
      throw new IllegalStateException(str);
   }
   
   /**
    * @see AuthorizationManagerServiceMBean#getAuthorizationManager(String)
    */
   public AuthorizationManager getAuthorizationManager( String securityDomain)  
   {
      AuthorizationManager amanager = (AuthorizationManager)authorizationManagersMap.get(securityDomain);
      if(amanager == null)
      {
         //create a new Authorization Manager 
         amanager = newAuthorizationManager(securityDomain); 
         authorizationManagersMap.put(securityDomain, amanager);
         log.debug("Added "+securityDomain+", " + amanager + " to map");
         //Add a JNDI binding based on the JaasSecurityManagerService
         //SecurityDomainContext
         try
         {
            Context ctx = new InitialContext();
            
            SecurityDomainContext sdc = (SecurityDomainContext)ctx.lookup("java:jaas/security/domainContext");
            sdc.setAuthorizationManager(amanager);
         }
         catch (NamingException e)
         {
            if(log.isTraceEnabled())
               log.trace("Error in naming", e);
            log.error("Error in getAuthorizationManager",e); 
         }
      }
      return amanager;
   } 
   
   /**
    * @see AuthorizationManager#authorize(Resource)
    */
   public int authorize(Resource resource)
   {
      String str = "Use getAuthorizationManager method and then call authorize";
      throw new IllegalStateException(str);
   }
   
   //PRIVATE METHODS 
   
   /** Create a new AuthorizationManager for securityDomain.
    * @param securityDomain
    * @return
    * @throws NamingException
    */
   static AuthorizationManager newAuthorizationManager(String securityDomain) 
   {
      AuthorizationManager securityMgr = null;
      try
      {
         // Create instance of securityMgrClass
         //Class[] parameterTypes = {String.class, CallbackHandler.class};
         Class<?>[] parameterTypes = {String.class};
         Constructor<?> ctor = authorizationMgrClass.getConstructor(parameterTypes); 
         CallbackHandler handler = (CallbackHandler) callbackHandlerClass.newInstance();
         //Object[] args = {securityDomain, handler};
         Object[] args = {securityDomain};
         securityMgr = (AuthorizationManager) ctor.newInstance(args); 
         log.debug("Created AuthorizationManager="+securityMgr); 
         
         CachePolicy cachePolicy = JaasSecurityManagerService.lookupCachePolicy(securityDomain);
         log.debug("Found Cache Policy="+cachePolicy); 
         // See if the security mgr supports an externalized cache policy
         setSecurityDomainCache(securityMgr, cachePolicy); 
      }
      catch(Exception e2)
      {
         String msg = "Failed to create authz mgr('"+securityDomain+"'), authorizationMgrClass="
         + authorizationMgrClass ;
         log.error(msg, e2); 
      }
      return securityMgr;
   } 
   
   /** Use reflection to attempt to set the authentication cache on the
    * the Authorization Manager argument.
    * @param securityMgr the Authorization Manager
    * @param cachePolicy the cache policy implementation
    */
   private static void setSecurityDomainCache(AuthorizationManager securityMgr,
         CachePolicy cachePolicy)
   {
      try
      {
         Class[] setCachePolicyTypes = {CachePolicy.class};
         Method m = authorizationMgrClass.getMethod("setCachePolicy", setCachePolicyTypes);
         Object[] setCachePolicyArgs = {cachePolicy};
         m.invoke(securityMgr, setCachePolicyArgs);
         log.debug("setCachePolicy, c="+setCachePolicyArgs[0]);
      }
      catch(Exception e2)
      {   // No cache policy support, this is ok
         if(log.isTraceEnabled())
            log.trace("optional setCachePolicy failed:" + e2.getLocalizedMessage());
      }
   }

   public Group getTargetRoles(Principal targetPrincipal, Map<String, Object> contextMap)
   {
      throw rte; 
   }  
   
   public String getSecurityDomain()
   {
      throw rte;
   } 

   public EntitlementHolder<?> entitlements(Resource resource, Identity identity) 
   throws AuthorizationException
   { 
      throw rte;
   }

   public int authorize(Resource arg0, Subject arg1, RoleGroup arg2) throws AuthorizationException
   {
      throw rte;
   }

   public int authorize(Resource arg0, Subject arg1, Group arg2) throws AuthorizationException
   {
      throw rte;
   }

   public RoleGroup getSubjectRoles(Subject arg0, CallbackHandler arg1)
   { 
      throw rte;
   } 
}