/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.acl.AclBinding;
import org.apache.kafka.common.acl.AclBindingFilter;
import org.apache.kafka.common.acl.AclState;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.errors.UnknownServerException;
import org.apache.kafka.common.metadata.AccessControlEntryRecord;
import org.apache.kafka.common.metadata.RemoveAccessControlEntryRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.requests.ApiError;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.metadata.authorizer.ConfluentStandardAcl;
import org.apache.kafka.metadata.authorizer.StandardAclWithId;
import org.apache.kafka.server.authorizer.AclCreateResult;
import org.apache.kafka.server.authorizer.AclDeleteResult;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.mutable.BoundedList;
import org.apache.kafka.server.mutable.BoundedListTooLongException;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.apache.kafka.timeline.TimelineHashMap;
import org.apache.kafka.timeline.TimelineHashSet;
import org.slf4j.Logger;

public class AclControlManager {
    private final Logger log;
    private final TimelineHashMap<Uuid, ConfluentStandardAcl> idToAcl;
    private final TimelineHashSet<ConfluentStandardAcl> existingAcls;
    private final TimelineHashMap<Uuid, TimelineHashSet<Uuid>> linkIdToAcls;
    private final Function<Uuid, Boolean> isValidClusterLink;
    private final SnapshotRegistry snapshotRegistry;

    private AclControlManager(LogContext logContext, SnapshotRegistry snapshotRegistry, Function<Uuid, Boolean> validLinkIdChecker) {
        this.log = logContext.logger(AclControlManager.class);
        this.idToAcl = new TimelineHashMap(snapshotRegistry, 0);
        this.existingAcls = new TimelineHashSet(snapshotRegistry, 0);
        this.linkIdToAcls = new TimelineHashMap(snapshotRegistry, 0);
        this.isValidClusterLink = validLinkIdChecker;
        this.snapshotRegistry = snapshotRegistry;
    }

    ControllerResult<List<AclCreateResult>> createAcls(List<AclBinding> acls) {
        return this.createAcls(acls, AclState.ACTIVE);
    }

    ControllerResult<List<AclCreateResult>> createAcls(List<AclBinding> acls, AclState aclState) {
        HashSet<ConfluentStandardAcl> aclsToCreate = new HashSet<ConfluentStandardAcl>(acls.size());
        ArrayList<AclCreateResult> results = new ArrayList<AclCreateResult>(acls.size());
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        for (AclBinding acl : acls) {
            try {
                AclControlManager.validateNewAcl(acl);
                AclControlManager.validateAclStateForCreate(aclState);
                for (Uuid linkId : acl.entry().clusterLinkIds()) {
                    if (linkId.equals((Object)Uuid.ZERO_UUID) || this.isValidClusterLink.apply(linkId).booleanValue()) continue;
                    throw new InvalidRequestException("Unknown link ID " + String.valueOf(linkId) + " provided to create ACL");
                }
            }
            catch (Throwable t) {
                ApiException e = t instanceof ApiException ? (ApiException)t : new UnknownServerException("Unknown error while trying to create ACL", t);
                results.add(new AclCreateResult(e));
                continue;
            }
            List<ConfluentStandardAcl> confluentStandardAclList = ConfluentStandardAcl.fromAclBindingWithAclState(acl, aclState);
            for (ConfluentStandardAcl confluentStandardAcl : confluentStandardAclList) {
                if (!this.existingAcls.contains((Object)confluentStandardAcl)) {
                    if (aclsToCreate.add(confluentStandardAcl)) {
                        StandardAclWithId standardAclWithId = new StandardAclWithId(this.newAclId(), confluentStandardAcl);
                        records.add(new ApiMessageAndVersion((ApiMessage)standardAclWithId.toRecord(), 0));
                        continue;
                    }
                    this.log.debug("Ignoring duplicate ACL from request: {}", (Object)confluentStandardAcl);
                    continue;
                }
                this.log.error("Not creating ACL since it already exists: {}", (Object)confluentStandardAcl);
            }
            results.add(AclCreateResult.SUCCESS);
        }
        return new ControllerResult<List<AclCreateResult>>((List<ApiMessageAndVersion>)records, results, true);
    }

    Uuid newAclId() {
        Uuid uuid;
        while (this.idToAcl.containsKey((Object)(uuid = Uuid.randomUuid()))) {
        }
        return uuid;
    }

    static void validateNewAcl(AclBinding binding) {
        switch (binding.pattern().resourceType()) {
            case UNKNOWN: 
            case ANY: {
                throw new InvalidRequestException("Invalid resourceType " + String.valueOf(binding.pattern().resourceType()));
            }
        }
        switch (binding.pattern().patternType()) {
            case LITERAL: 
            case PREFIXED: {
                break;
            }
            default: {
                throw new InvalidRequestException("Invalid patternType " + String.valueOf(binding.pattern().patternType()));
            }
        }
        switch (binding.entry().operation()) {
            case UNKNOWN: 
            case ANY: {
                throw new InvalidRequestException("Invalid operation " + String.valueOf(binding.entry().operation()));
            }
        }
        switch (binding.entry().permissionType()) {
            case DENY: 
            case ALLOW: {
                break;
            }
            default: {
                throw new InvalidRequestException("Invalid permissionType " + String.valueOf(binding.entry().permissionType()));
            }
        }
        if (binding.pattern().name() == null || binding.pattern().name().isEmpty()) {
            throw new InvalidRequestException("Resource name should not be empty");
        }
        int colonIndex = binding.entry().principal().indexOf(":");
        if (colonIndex == -1) {
            throw new InvalidRequestException("Could not parse principal from `" + binding.entry().principal() + "` (no colon is present separating the principal type from the principal name)");
        }
    }

    ControllerResult<List<AclDeleteResult>> deleteAcls(List<AclBindingFilter> filters) {
        return this.deleteAcls(filters, AclState.ANY);
    }

    ControllerResult<List<AclDeleteResult>> deleteAcls(List<AclBindingFilter> filters, AclState aclState) {
        ArrayList<AclDeleteResult> results = new ArrayList<AclDeleteResult>();
        HashSet<ApiMessageAndVersion> records = new HashSet<ApiMessageAndVersion>();
        for (AclBindingFilter filter : filters) {
            try {
                AclControlManager.validateFilter(filter);
                AclControlManager.validateAclStateForDelete(aclState);
                AclDeleteResult result = this.deleteAclsForFilter(filter, records, aclState);
                results.add(result);
            }
            catch (BoundedListTooLongException e) {
                throw new InvalidRequestException(e.getMessage(), (Throwable)e);
            }
            catch (Throwable e) {
                results.add(new AclDeleteResult(ApiError.fromThrowable((Throwable)e).exception()));
            }
        }
        return ControllerResult.atomicOf(new ArrayList<ApiMessageAndVersion>(records), results);
    }

    AclDeleteResult deleteAclsForFilter(AclBindingFilter filter, Set<ApiMessageAndVersion> records, AclState aclState) {
        ArrayList<AclDeleteResult.AclBindingDeleteResult> deleted = new ArrayList<AclDeleteResult.AclBindingDeleteResult>();
        for (Map.Entry entry : this.idToAcl.entrySet()) {
            Uuid id = (Uuid)entry.getKey();
            ConfluentStandardAcl acl = (ConfluentStandardAcl)entry.getValue();
            AclBinding binding = acl.toBinding();
            if (!filter.matches(binding) || acl.aclState() != aclState && aclState != AclState.ANY) continue;
            if (records.size() >= 10000) {
                throw new BoundedListTooLongException("Cannot remove more than 10000 acls in a single delete operation.");
            }
            if (aclState != AclState.ANY || acl.aclState() != AclState.DELETED) {
                deleted.add(new AclDeleteResult.AclBindingDeleteResult(binding));
            }
            records.add(new ApiMessageAndVersion((ApiMessage)new RemoveAccessControlEntryRecord().setId(id), 0));
        }
        return new AclDeleteResult(deleted);
    }

    static void validateAclStateForCreate(AclState aclState) {
        switch (aclState) {
            case UNKNOWN: 
            case ANY: {
                throw new InvalidRequestException("Invalid AclState " + String.valueOf(aclState));
            }
        }
    }

    static void validateAclStateForDelete(AclState aclState) {
        if (aclState == AclState.UNKNOWN) {
            throw new InvalidRequestException("Invalid AclState " + String.valueOf(aclState));
        }
    }

    static void validateFilter(AclBindingFilter filter) {
        if (filter.patternFilter().isUnknown()) {
            throw new InvalidRequestException("Unknown patternFilter.");
        }
        if (filter.entryFilter().isUnknown()) {
            throw new InvalidRequestException("Unknown entryFilter.");
        }
    }

    public void replay(AccessControlEntryRecord record) {
        StandardAclWithId aclWithId = StandardAclWithId.fromRecord(record);
        this.idToAcl.put((Object)aclWithId.id(), (Object)aclWithId.acl());
        this.existingAcls.add((Object)aclWithId.acl());
        if (aclWithId.acl().hasLinkId()) {
            Uuid linkId = aclWithId.acl().clusterLinkId().get();
            TimelineHashSet aclsForLink = (TimelineHashSet)this.linkIdToAcls.computeIfAbsent((Object)linkId, __ -> new TimelineHashSet(this.snapshotRegistry, 0));
            aclsForLink.add((Object)aclWithId.id());
        }
        this.log.info("Replayed AccessControlEntryRecord for {}, setting {}", (Object)record.id(), (Object)aclWithId.acl());
    }

    public void replay(RemoveAccessControlEntryRecord record) {
        ConfluentStandardAcl acl = (ConfluentStandardAcl)this.idToAcl.remove((Object)record.id());
        if (acl == null) {
            throw new RuntimeException("Unable to replay " + String.valueOf(record) + ": no acl with that ID found.");
        }
        if (!this.existingAcls.remove((Object)acl)) {
            throw new RuntimeException("Unable to replay " + String.valueOf(record) + " for " + String.valueOf(acl) + ": acl not found in existingAcls.");
        }
        Uuid linkId = null;
        if (acl.hasLinkId()) {
            linkId = acl.clusterLinkId().get();
            this.linkIdToAcls.compute((Object)linkId, (id, aclsForLink) -> {
                if (aclsForLink == null || !aclsForLink.remove((Object)record.id())) {
                    throw new RuntimeException("Unable to replay " + String.valueOf(record) + " for " + String.valueOf(acl) + ": acl not found in ACLs with link id");
                }
                if (aclsForLink.isEmpty()) {
                    return null;
                }
                return aclsForLink;
            });
        }
        this.log.info("Replayed RemoveAccessControlEntryRecord for {}, removing {}{}", new Object[]{record.id(), acl, linkId == null ? "" : ", and updating the ACLs associated with the cluster link {}", linkId});
    }

    Map<Uuid, ConfluentStandardAcl> idToAcl() {
        return Collections.unmodifiableMap(this.idToAcl);
    }

    void unlinkAcls(Uuid linkId) {
        if (linkId.equals((Object)Uuid.ZERO_UUID)) {
            throw new IllegalStateException("Trying to unlink ACLs for invalid link id");
        }
        Set toRemoveLinkAcls = (Set)this.linkIdToAcls.remove((Object)linkId);
        if (toRemoveLinkAcls == null) {
            return;
        }
        HashMap aclChanges = new HashMap();
        AtomicInteger aclsAdded = new AtomicInteger();
        toRemoveLinkAcls.forEach(aclId -> {
            ConfluentStandardAcl acl = (ConfluentStandardAcl)this.idToAcl.remove(aclId);
            if (acl == null || !this.existingAcls.remove((Object)acl)) {
                this.log.error("Found non-existent ACL trying to clear link ID " + String.valueOf(linkId));
            } else {
                ConfluentStandardAcl localAcl = acl.toLocalAcl();
                if (this.existingAcls.add((Object)localAcl)) {
                    this.idToAcl.put(aclId, (Object)localAcl);
                    aclChanges.put(aclId, Optional.of(localAcl));
                    aclsAdded.getAndIncrement();
                } else {
                    aclChanges.put(aclId, Optional.empty());
                }
            }
        });
        this.log.info("Removed " + (aclChanges.size() - aclsAdded.get()) + "acls and added " + aclsAdded.get() + "local acls after unlinking cluster link " + String.valueOf(linkId));
    }

    boolean validateAclsInCache(Set<ConfluentStandardAcl> acls) {
        if (this.idToAcl.size() != this.existingAcls.size()) {
            return false;
        }
        return this.existingAcls.equals(acls);
    }

    static class Builder {
        private LogContext logContext = null;
        private SnapshotRegistry snapshotRegistry = null;
        private Function<Uuid, Boolean> validLinkIdChecker = __ -> true;

        Builder() {
        }

        Builder setLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        Builder setSnapshotRegistry(SnapshotRegistry snapshotRegistry) {
            this.snapshotRegistry = snapshotRegistry;
            return this;
        }

        Builder setValidLinkIdChecker(Function<Uuid, Boolean> validLinkIdChecker) {
            this.validLinkIdChecker = validLinkIdChecker;
            return this;
        }

        AclControlManager build() {
            if (this.logContext == null) {
                this.logContext = new LogContext();
            }
            if (this.snapshotRegistry == null) {
                this.snapshotRegistry = new SnapshotRegistry(this.logContext);
            }
            return new AclControlManager(this.logContext, this.snapshotRegistry, this.validLinkIdChecker);
        }
    }
}

