/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.storage.checksum;

import io.confluent.kafka.availability.FileChannelWrapper;
import io.confluent.kafka.availability.FilesWrapper;
import io.confluent.kafka.storage.checksum.Adler32CheckedFileIO;
import io.confluent.kafka.storage.checksum.Adler32WithInPlaceUpdate;
import io.confluent.kafka.storage.checksum.Algorithm;
import io.confluent.kafka.storage.checksum.ChecksumWithInPlaceUpdate;
import io.confluent.kafka.storage.checksum.NoChecksumFileIO;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.rmi.UnexpectedException;
import java.util.Arrays;
import java.util.Optional;
import org.apache.kafka.common.errors.MultiChecksumTypeException;
import org.apache.kafka.common.errors.OtherChecksumTypeException;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CheckedFileIO
implements AutoCloseable {
    static final int READ_CHUNK_SIZE = 262144;
    public static final ByteOrder BYTE_ORDER = ByteOrder.LITTLE_ENDIAN;
    final byte version = 1;
    final FileChannel fileChannel;
    final Algorithm algorithm;
    final ChecksumWithInPlaceUpdate checksumPackage;
    final short superBlockLength;
    final boolean readOnly;
    static final Logger log = LoggerFactory.getLogger(CheckedFileIO.class);

    CheckedFileIO(Algorithm algorithm, short superBlockLength, Path path, OpenOption ... options) throws IOException {
        CheckedFileIO.verifyArguments(algorithm, superBlockLength, path, options);
        this.readOnly = Arrays.stream(options).noneMatch(option -> option == StandardOpenOption.WRITE);
        this.algorithm = algorithm;
        this.superBlockLength = superBlockLength;
        this.fileChannel = FileChannelWrapper.open((Path)path, (OpenOption[])options);
        this.checksumPackage = algorithm == Algorithm.ADLER ? new Adler32WithInPlaceUpdate() : null;
    }

    public static void migrateFile(Path srcPath, Path destPath, short destSuperBlockLength) throws IOException {
        Algorithm destAlgo;
        if (!FilesWrapper.exists((Path)srcPath, (LinkOption[])new LinkOption[0])) {
            throw new IllegalArgumentException("Source input file path is not present in migrate: " + String.valueOf(srcPath));
        }
        if (FilesWrapper.exists((Path)destPath, (LinkOption[])new LinkOption[0])) {
            throw new IllegalArgumentException("Destination file path is present in migrate: " + String.valueOf(destPath));
        }
        Algorithm srcAlgo = CheckedFileIO.isValidPath(Algorithm.ADLER, srcPath) ? Algorithm.ADLER : Algorithm.NO_CHECKSUM;
        Algorithm algorithm = destAlgo = CheckedFileIO.isValidPath(Algorithm.ADLER, destPath) ? Algorithm.ADLER : Algorithm.NO_CHECKSUM;
        if (srcAlgo == destAlgo) {
            throw new IllegalArgumentException("Source and destination algorithm are same for migrate: " + String.valueOf((Object)srcAlgo));
        }
        log.info("Migrating source file path: " + String.valueOf(srcPath) + " source algorithm: " + String.valueOf((Object)srcAlgo) + " to destination file path: " + String.valueOf(destPath) + " destination algorithm: " + String.valueOf((Object)destAlgo) + " destination super block length: " + destSuperBlockLength);
        CheckedFileIO cfIO = CheckedFileIO.open(srcPath, StandardOpenOption.READ);
        cfIO.migrateHelper(srcPath, destAlgo, destSuperBlockLength, destPath);
    }

    private void migrateHelper(Path srcPath, Algorithm destAlgo, short destSuperBlockLength, Path destPath) throws IOException {
        Path destDir = destPath.normalize().getParent();
        Path tmpPath = destDir == null ? FilesWrapper.createTempFile((String)"CheckedFileIO.tmp", (String)destAlgo.suffix, (FileAttribute[])new FileAttribute[0]) : FilesWrapper.createTempFile((Path)destDir, (String)"CheckedFileIO.tmp", (String)destAlgo.suffix, (FileAttribute[])new FileAttribute[0]);
        CheckedFileIO tmpIO = CheckedFileIO.openOrCreate(tmpPath, destAlgo, destSuperBlockLength, StandardOpenOption.READ, StandardOpenOption.WRITE);
        this.transferTo(0L, this.size(), tmpIO);
        tmpIO.close();
        this.close();
        Utils.atomicMoveWithFallback((Path)tmpPath, (Path)destPath);
        FilesWrapper.delete((Path)srcPath);
        log.info("Migration successful from source file path: " + String.valueOf(srcPath) + " to destination file path: " + String.valueOf(destPath));
    }

    public abstract boolean validate() throws IOException, ReflectiveOperationException;

    public static Path validPath(Algorithm algo, Path path) {
        String pathStr = path.toString();
        if (algo == Algorithm.ADLER) {
            if (CheckedFileIO.isValidPath(algo, path)) {
                return path;
            }
            return Paths.get(pathStr + Algorithm.ADLER.suffix, new String[0]);
        }
        if (algo == Algorithm.NO_CHECKSUM) {
            if (CheckedFileIO.isValidPath(Algorithm.ADLER, path)) {
                return Paths.get(pathStr.substring(0, pathStr.length() - Algorithm.ADLER.suffix.length()), new String[0]);
            }
            return path;
        }
        throw new UnsupportedOperationException("Unhandled algorithm: " + String.valueOf((Object)algo) + " in validPath");
    }

    public static void verifyOnlyValidFileExists(Algorithm algo, Path path) throws FileNotFoundException {
        Path adlerFilePath = CheckedFileIO.validPath(Algorithm.ADLER, path);
        Path noCheckSumPath = CheckedFileIO.validPath(Algorithm.NO_CHECKSUM, path);
        boolean adlerFileExists = FilesWrapper.exists((Path)adlerFilePath, (LinkOption[])new LinkOption[0]);
        boolean noChecksumFileExists = FilesWrapper.exists((Path)noCheckSumPath, (LinkOption[])new LinkOption[0]);
        if (noChecksumFileExists && adlerFileExists) {
            throw new MultiChecksumTypeException("Both non-checksum file " + String.valueOf(noCheckSumPath) + " and adler file " + String.valueOf(adlerFilePath) + " exists");
        }
        if (!noChecksumFileExists && !adlerFileExists) {
            throw new FileNotFoundException("No checksum supported file found " + String.valueOf(noCheckSumPath) + " or " + String.valueOf(adlerFilePath));
        }
        if (algo == Algorithm.ADLER) {
            if (noChecksumFileExists) {
                throw new OtherChecksumTypeException("Non-checksum file exists " + String.valueOf(noCheckSumPath) + " for algo " + String.valueOf((Object)algo));
            }
        } else if (algo == Algorithm.NO_CHECKSUM && adlerFileExists) {
            throw new OtherChecksumTypeException("Adler file exists " + String.valueOf(adlerFilePath) + " for algo " + String.valueOf((Object)algo));
        }
    }

    public static boolean isValidPath(Algorithm algo, Path path) {
        boolean isAdlerPath = path.toString().endsWith(Algorithm.ADLER.suffix);
        if (algo == Algorithm.ADLER) {
            return isAdlerPath;
        }
        if (algo == Algorithm.NO_CHECKSUM) {
            return !isAdlerPath;
        }
        throw new UnsupportedOperationException("Unhandled algorithm: " + String.valueOf((Object)algo) + " in isValidPath");
    }

    private static void verifyArguments(Algorithm algo, short superBlockLength, Path path, OpenOption ... options) {
        if (algo == Algorithm.ADLER) {
            if (superBlockLength > 4096) {
                throw new IllegalArgumentException("Incorrect super block length argument for algorithm: " + String.valueOf((Object)algo) + " superBlockLength " + superBlockLength + " should be less than 4096");
            }
        } else if (algo == Algorithm.NO_CHECKSUM && superBlockLength != 0) {
            throw new IllegalArgumentException("Incorrect super block length argument for algorithm: " + String.valueOf((Object)algo) + " superBlockLength " + superBlockLength + " should be 0");
        }
        if (!CheckedFileIO.isValidPath(algo, path)) {
            throw new IllegalArgumentException("Incorrect path argument for algorithm: " + String.valueOf((Object)algo) + "  path: " + String.valueOf(path));
        }
        if (Arrays.stream(options).anyMatch(option -> option == StandardOpenOption.APPEND)) {
            throw new IllegalArgumentException("Incorrect options argument for algorithm: " + String.valueOf((Object)algo) + " path: " + String.valueOf(path) + " options cannot contain APPEND");
        }
        if (Arrays.stream(options).noneMatch(option -> option == StandardOpenOption.READ) && Arrays.stream(options).noneMatch(option -> option == StandardOpenOption.WRITE)) {
            throw new IllegalArgumentException("Incorrect options argument for algorithm: " + String.valueOf((Object)algo) + " path: " + String.valueOf(path) + " options missing READ and WRITE");
        }
        if (Arrays.stream(options).noneMatch(option -> option == StandardOpenOption.READ) && Arrays.stream(options).anyMatch(option -> option == StandardOpenOption.WRITE)) {
            throw new IllegalArgumentException("Incorrect options argument for algorithm: " + String.valueOf((Object)algo) + " path: " + String.valueOf(path) + " options has WRITE but missing READ");
        }
    }

    public Optional<Long> checksum() {
        if (this.algorithm == Algorithm.ADLER) {
            return Optional.of(this.checksumPackage.getValue());
        }
        return Optional.empty();
    }

    public static CheckedFileIO openOrCreate(Path path, Algorithm algo, short superBlockLength, boolean skipMultiChecksumValidation, OpenOption ... options) throws IOException {
        try {
            if (!skipMultiChecksumValidation) {
                CheckedFileIO.verifyOnlyValidFileExists(algo, path);
            }
            if (algo == Algorithm.ADLER) {
                return new Adler32CheckedFileIO(algo, superBlockLength, path, options);
            }
            if (algo == Algorithm.NO_CHECKSUM) {
                return new NoChecksumFileIO(algo, superBlockLength, path, options);
            }
        }
        catch (FileNotFoundException e) {
            return CheckedFileIO.createHelper(path, algo, superBlockLength, options);
        }
        throw new UnsupportedOperationException("Unhandled algorithm: " + String.valueOf((Object)algo) + " in open");
    }

    public static CheckedFileIO openOrCreate(Path path, Algorithm algo, short superBlockLength, OpenOption ... options) throws IOException {
        return CheckedFileIO.openOrCreate(path, algo, superBlockLength, false, options);
    }

    public static CheckedFileIO open(Path path, boolean skipMultiChecksumValidation, OpenOption ... options) throws IOException {
        short superBlockLength;
        Algorithm algo;
        if (!FilesWrapper.exists((Path)path, (LinkOption[])new LinkOption[0])) {
            throw new FileNotFoundException();
        }
        if (Arrays.stream(options).anyMatch(option -> option == StandardOpenOption.CREATE) || Arrays.stream(options).anyMatch(option -> option == StandardOpenOption.CREATE_NEW)) {
            throw new IllegalArgumentException("Incorrect options for path: " + String.valueOf(path) + " options contains CREATE or CREATE_NEW");
        }
        if (CheckedFileIO.isValidPath(Algorithm.ADLER, path)) {
            algo = Algorithm.ADLER;
            FileChannel fc = FileChannelWrapper.open((Path)path, (OpenOption[])new OpenOption[]{StandardOpenOption.READ});
            ByteBuffer superBlockParams = Adler32CheckedFileIO.readSuperBlockParams(fc);
            fc.close();
            superBlockLength = superBlockParams.getShort();
        } else if (CheckedFileIO.isValidPath(Algorithm.NO_CHECKSUM, path)) {
            algo = Algorithm.NO_CHECKSUM;
            superBlockLength = 0;
        } else {
            throw new UnexpectedException("Input path: " + String.valueOf(path) + "is not valid");
        }
        return CheckedFileIO.openOrCreate(path, algo, superBlockLength, skipMultiChecksumValidation, options);
    }

    public static CheckedFileIO open(Path path, OpenOption ... options) throws IOException {
        return CheckedFileIO.open(path, false, options);
    }

    public static void create(Path path, Algorithm algo, short superBlockLength) throws IOException {
        CheckedFileIO cfIO = CheckedFileIO.createHelper(path, algo, superBlockLength, StandardOpenOption.CREATE_NEW, StandardOpenOption.READ, StandardOpenOption.WRITE);
        cfIO.close();
    }

    private static CheckedFileIO createHelper(Path path, Algorithm algo, short superBlockLength, OpenOption ... options) throws IOException {
        if (Arrays.stream(options).noneMatch(option -> option == StandardOpenOption.CREATE) && Arrays.stream(options).noneMatch(option -> option == StandardOpenOption.CREATE_NEW)) {
            throw new IllegalArgumentException("Couldn't open file for path: " + String.valueOf(path) + " as CREATE or CREATE_NEW option is missing");
        }
        if (algo == Algorithm.ADLER) {
            return new Adler32CheckedFileIO(algo, superBlockLength, path, options);
        }
        if (algo == Algorithm.NO_CHECKSUM) {
            return new NoChecksumFileIO(algo, superBlockLength, path, options);
        }
        throw new UnsupportedOperationException("Unhandled algorithm: " + String.valueOf((Object)algo) + " in create");
    }

    public void transferTo(long position, long count, CheckedFileIO target) throws IOException {
        ByteBuffer bb = ByteBuffer.allocate(262144).order(BYTE_ORDER);
        long totalBytesRead = 0L;
        long pos = position;
        while (totalBytesRead < count) {
            bb.clear();
            this.read(bb, pos);
            bb.flip();
            long bytesRead = bb.remaining();
            if (bb.remaining() == 0) break;
            if (totalBytesRead + bytesRead >= count) {
                bytesRead = count - totalBytesRead;
                bb.limit((int)bytesRead);
            }
            totalBytesRead += bytesRead;
            pos += bytesRead;
            target.write(bb);
        }
    }

    public void transferFrom(CheckedFileIO source, long position, long count) throws IOException {
        if (this.readOnly) {
            throw new IllegalArgumentException("Cannot transfer from as file is read-only");
        }
        ByteBuffer bb = ByteBuffer.allocate(262144).order(BYTE_ORDER);
        long totalBytesRead = 0L;
        long pos = position;
        while (totalBytesRead < count) {
            bb.clear();
            source.read(bb, pos);
            bb.flip();
            long bytesRead = bb.remaining();
            if (bb.remaining() == 0) break;
            if (totalBytesRead + bytesRead >= count) {
                bytesRead = count - totalBytesRead;
                bb.limit((int)bytesRead);
            }
            totalBytesRead += bytesRead;
            pos += bytesRead;
            this.write(bb);
        }
    }

    public abstract void read(ByteBuffer var1, long var2) throws IOException;

    public abstract void write(ByteBuffer var1) throws IOException;

    public abstract void write(ByteBuffer var1, long var2) throws IOException;

    public abstract long position() throws IOException;

    public abstract void position(long var1) throws IOException;

    public abstract long size() throws IOException;

    public abstract void truncate(long var1) throws IOException;

    public abstract void flush() throws IOException;

    @Override
    public abstract void close() throws IOException;
}

