/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.runtime.mongodb;

import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.ReadConcern;
import com.mongodb.ReadConcernLevel;
import com.mongodb.ReadPreference;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.MongoIterable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.StreamSupport;
import javax.net.ssl.SSLContext;
import org.apache.commons.beanutils.BeanIntrospector;
import org.apache.commons.beanutils.BeanUtilsBean;
import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.beanutils.FluentPropertyBeanIntrospector;
import org.apache.commons.beanutils.PropertyUtilsBean;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.SSLContexts;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nuxeo.runtime.RuntimeServiceException;
import org.nuxeo.runtime.mongodb.MongoDBConnectionConfig;

public class MongoDBConnectionHelper {
    private static final Logger log = LogManager.getLogger(MongoDBConnectionHelper.class);
    private static final String DB_DEFAULT = "nuxeo";
    private static final int MONGODB_OPTION_CONNECT_TIMEOUT_MS = 30000;
    private static final int MONGODB_OPTION_READ_TIMEOUT_MS = 60000;

    private MongoDBConnectionHelper() {
    }

    public static MongoClient newMongoClient(String server) {
        MongoDBConnectionConfig config = new MongoDBConnectionConfig();
        config.server = server;
        return MongoDBConnectionHelper.newMongoClient(config);
    }

    public static MongoClient newMongoClient(MongoDBConnectionConfig config) {
        return MongoDBConnectionHelper.newMongoClient(config, null);
    }

    public static MongoClient newMongoClient(MongoDBConnectionConfig config, Consumer<MongoClientSettings.Builder> settingsConsumer) {
        String server = config.server;
        if (StringUtils.isBlank((CharSequence)server)) {
            throw new RuntimeServiceException("Missing <server> in MongoDB descriptor");
        }
        MongoClientSettings.Builder settingsBuilder = MongoClientSettings.builder().applicationName("Nuxeo");
        SSLContext sslContext = MongoDBConnectionHelper.getSSLContext(config);
        if (sslContext == null) {
            if (config.ssl != null) {
                settingsBuilder.applyToSslSettings(s -> s.enabled(config.ssl.booleanValue()));
            }
        } else {
            settingsBuilder.applyToSslSettings(s -> s.enabled(true).context(sslContext));
        }
        settingsBuilder.applyToSocketSettings(s -> s.connectTimeout(30000, TimeUnit.MILLISECONDS).readTimeout(60000, TimeUnit.MILLISECONDS));
        MongoDBConnectionHelper.populateProperties(config, settingsBuilder);
        if (settingsConsumer != null) {
            settingsConsumer.accept(settingsBuilder);
        }
        if (server.startsWith("mongodb://") || server.startsWith("mongodb+srv://")) {
            settingsBuilder.applyConnectionString(new ConnectionString(server));
        } else {
            settingsBuilder.applyToClusterSettings(b -> b.hosts(List.of(new ServerAddress(server))));
        }
        MongoClientSettings settings = settingsBuilder.build();
        MongoClient client = MongoClients.create((MongoClientSettings)settings);
        log.debug("MongoClient initialized with settings: {}", (Object)settings);
        return client;
    }

    protected static void populateProperties(MongoDBConnectionConfig config, MongoClientSettings.Builder settingsBuilder) {
        ConvertUtilsBean convertUtils = new ConvertUtilsBean();
        convertUtils.register((Converter)ReadPreferenceConverter.INSTANCE, ReadPreference.class);
        convertUtils.register((Converter)ReadConcernConverter.INSTANCE, ReadConcern.class);
        convertUtils.register((Converter)WriteConcernConverter.INSTANCE, WriteConcern.class);
        PropertyUtilsBean propertyUtils = new PropertyUtilsBean();
        propertyUtils.addBeanIntrospector((BeanIntrospector)new FluentPropertyBeanIntrospector(""));
        BeanUtilsBean beanUtils = new BeanUtilsBean(convertUtils, propertyUtils);
        try {
            beanUtils.populate((Object)settingsBuilder, config.properties);
        }
        catch (ReflectiveOperationException e) {
            throw new RuntimeServiceException((Throwable)e);
        }
    }

    protected static SSLContext getSSLContext(MongoDBConnectionConfig config) {
        try {
            KeyStore trustStore = MongoDBConnectionHelper.loadKeyStore(config.trustStorePath, config.trustStorePassword, config.trustStoreType);
            KeyStore keyStore = MongoDBConnectionHelper.loadKeyStore(config.keyStorePath, config.keyStorePassword, config.keyStoreType);
            if (trustStore == null && keyStore == null) {
                return null;
            }
            SSLContextBuilder sslContextBuilder = SSLContexts.custom();
            if (trustStore != null) {
                sslContextBuilder.loadTrustMaterial(trustStore, null);
            }
            if (keyStore != null) {
                sslContextBuilder.loadKeyMaterial(keyStore, StringUtils.isBlank((CharSequence)config.keyStorePassword) ? null : config.keyStorePassword.toCharArray());
            }
            return sslContextBuilder.build();
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeServiceException("Cannot setup SSL context: " + config, (Throwable)e);
        }
    }

    protected static KeyStore loadKeyStore(String path, String password, String type) throws GeneralSecurityException, IOException {
        if (StringUtils.isBlank((CharSequence)path)) {
            return null;
        }
        String keyStoreType = (String)StringUtils.defaultIfBlank((CharSequence)type, (CharSequence)KeyStore.getDefaultType());
        KeyStore keyStore = KeyStore.getInstance(keyStoreType);
        char[] passwordChars = StringUtils.isBlank((CharSequence)password) ? null : password.toCharArray();
        try (InputStream is = Files.newInputStream(Paths.get(path, new String[0]), new OpenOption[0]);){
            keyStore.load(is, passwordChars);
        }
        return keyStore;
    }

    public static MongoDatabase getDatabase(MongoClient mongoClient, String dbname) {
        if (StringUtils.isBlank((CharSequence)dbname)) {
            dbname = DB_DEFAULT;
        }
        return mongoClient.getDatabase(dbname);
    }

    public static boolean hasCollection(MongoDatabase mongoDatabase, String collection) {
        MongoIterable collections = mongoDatabase.listCollectionNames();
        boolean found = StreamSupport.stream(collections.spliterator(), false).anyMatch(collection::equals);
        return found && mongoDatabase.getCollection(collection).estimatedDocumentCount() > 0L;
    }

    public static class WriteConcernConverter
    implements Converter {
        public static final WriteConcernConverter INSTANCE = new WriteConcernConverter();

        public <T> T convert(Class<T> type, Object value) {
            return (T)WriteConcern.valueOf((String)((String)value));
        }
    }

    public static class ReadConcernConverter
    implements Converter {
        public static final ReadConcernConverter INSTANCE = new ReadConcernConverter();

        public <T> T convert(Class<T> type, Object value) {
            ReadConcern readConcern;
            if ("default".equalsIgnoreCase((String)value)) {
                readConcern = ReadConcern.DEFAULT;
            } else {
                ReadConcernLevel level = ReadConcernLevel.fromString((String)((String)value));
                readConcern = new ReadConcern(level);
            }
            return (T)readConcern;
        }
    }

    public static class ReadPreferenceConverter
    implements Converter {
        public static final ReadPreferenceConverter INSTANCE = new ReadPreferenceConverter();

        public <T> T convert(Class<T> type, Object value) {
            return (T)ReadPreference.valueOf((String)((String)value));
        }
    }
}

