package com.cebpubservice.remotebidevaluation.component.sdk.expertlibraryshareextract.result;

import com.cebpubservice.remotebidevaluation.DefaultCebsClient;
import com.cebpubservice.remotebidevaluation.ICebsClient;
import com.cebpubservice.remotebidevaluation.enums.InitType;
import com.cebpubservice.remotebidevaluation.exceptions.ClientException;

/**
 * 远程专家库共享抽取单例
 *
 * @author 柏晨瑶
 * ===============================
 * Date：2025/01/14
 * Time：17:46
 * ================================
 */
public class ComponentExpertLibraryShareExtractSingleton {

    /**
     * 用于存储 ICebsClient 实例的静态变量，使用 volatile 关键字确保线程可见性。
     */
    private static volatile ICebsClient instance;

    /**
     * 私有构造函数，防止外部实例化对象。
     */
    private ComponentExpertLibraryShareExtractSingleton() {
    }

    /**
     * 使用平台代码、服务URL、客户端ID、客户端密钥和访问密钥初始化单例。
     * 这是初始化单例的主要方法，使用了双重检查锁定来确保线程安全。
     *
     * @param tradingSystemCode 交易系统编码，用于区分不同交易系统的客户端。
     * @param serviceUrl        服务URL，指定前置机的服务地址。
     * @param clientId          客户端ID，用于标识调用方。
     * @param clientSecret      客户端密钥，用于验证调用方的身份。
     * @param signatureSecret   访问密钥 secret，用于签名请求。
     * @return 返回初始化后的 ICebsClient 实例。
     */
    public static ICebsClient init(String tradingSystemCode, String serviceUrl, String clientId, String clientSecret, String signatureSecret) throws ClientException {
        if (instance == null) {
            synchronized (ComponentExpertLibraryShareExtractSingleton.class) {
                if (instance == null) {
                    instance = new DefaultCebsClient(InitType.TRADING_SYSTEM_CODE, tradingSystemCode, serviceUrl, clientId, clientSecret, signatureSecret);
                }
            }
        }
        return instance;
    }

    /**
     * 获取 ICebsClient 的单例实例。
     * 如果实例尚未初始化，则抛出运行时异常。
     *
     * @return 返回 ICebsClient 的单例实例。
     * @throws RuntimeException 如果 ICebsClient 尚未初始化，则抛出此异常。
     */
    public static ICebsClient getInstance() {
        if (instance == null) {
            throw new RuntimeException("CebsClient not init,please execute expertlibraryshareextract ComponentExpertLibraryShareExtractSingleton.init().");
        }
        return instance;
    }
}

