/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.xades.reference;

import eu.europa.esig.dss.DomUtils;
import eu.europa.esig.dss.definition.xmldsig.XMLDSigAttribute;
import eu.europa.esig.dss.definition.xmldsig.XMLDSigElement;
import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.exception.IllegalInputException;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.xades.DSSXMLUtils;
import eu.europa.esig.dss.xades.XAdESSignatureParameters;
import eu.europa.esig.dss.xades.reference.Base64Transform;
import eu.europa.esig.dss.xades.reference.CanonicalizationTransform;
import eu.europa.esig.dss.xades.reference.DSSReference;
import eu.europa.esig.dss.xades.reference.DSSTransform;
import eu.europa.esig.dss.xades.reference.XPath2FilterEnvelopedSignatureTransform;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class ReferenceBuilder {
    private final List<DSSDocument> documents;
    private final DigestAlgorithm digestAlgorithm;
    private XAdESSignatureParameters signatureParameters;
    private static final String OBJECT_ID_PREFIX = "o-";
    private String referenceIdPrefix = "r-";

    public ReferenceBuilder(List<DSSDocument> documents, XAdESSignatureParameters xadesSignatureParameters) {
        Objects.requireNonNull(documents, "List of documents shall be provided!");
        Objects.requireNonNull(xadesSignatureParameters, "Signature parameters shall be provided!");
        this.documents = documents;
        this.signatureParameters = xadesSignatureParameters;
        this.digestAlgorithm = this.getReferenceDigestAlgorithmOrDefault(xadesSignatureParameters);
    }

    public ReferenceBuilder(List<DSSDocument> documents, DigestAlgorithm digestAlgorithm) {
        Objects.requireNonNull(documents, "List of documents shall be provided!");
        Objects.requireNonNull(digestAlgorithm, "Digest Algorithm shall be provided!");
        this.documents = documents;
        this.digestAlgorithm = digestAlgorithm;
    }

    public void setReferenceIdPrefix(String referenceIdPrefix) {
        if (Utils.isStringBlank((String)referenceIdPrefix)) {
            throw new IllegalArgumentException("The reference id prefix cannot be blank!");
        }
        this.referenceIdPrefix = referenceIdPrefix;
    }

    public List<DSSReference> build() {
        ArrayList<DSSReference> references = new ArrayList<DSSReference>();
        int referenceIndex = 1;
        for (DSSDocument dssDocument : this.documents) {
            references.add(this.createDSSReferenceForDocument(dssDocument, referenceIndex));
            ++referenceIndex;
        }
        return references;
    }

    private DSSReference createDSSReferenceForDocument(DSSDocument document, int index) {
        if (this.signatureParameters != null) {
            Objects.requireNonNull(this.signatureParameters.getSignaturePackaging(), "SignaturePackaging must be defined!");
            switch (this.signatureParameters.getSignaturePackaging()) {
                case ENVELOPED: {
                    return this.envelopedDSSReference(document, index);
                }
                case ENVELOPING: {
                    return this.envelopingDSSReference(document, index);
                }
                case DETACHED: {
                    return this.detachedDSSReference(document, index);
                }
                case INTERNALLY_DETACHED: {
                    return this.internallyDetachedDSSReference(document, index);
                }
            }
            throw new DSSException(String.format("The given signature packaging %s is not supported!", this.signatureParameters.getSignaturePackaging()));
        }
        return this.detachedDSSReference(document, index);
    }

    private DSSReference envelopedDSSReference(DSSDocument document, int index) {
        this.assertEnvelopedSignaturePossible(document);
        DSSReference dssReference = new DSSReference();
        dssReference.setId(this.referenceIdPrefix + this.getReferenceId(index));
        dssReference.setUri("");
        dssReference.setContents(document);
        dssReference.setDigestMethodAlgorithm(this.digestAlgorithm);
        ArrayList<DSSTransform> dssTransformList = new ArrayList<DSSTransform>();
        XPath2FilterEnvelopedSignatureTransform xPathTransform = new XPath2FilterEnvelopedSignatureTransform(this.signatureParameters.getXmldsigNamespace());
        dssTransformList.add(xPathTransform);
        CanonicalizationTransform canonicalizationTransform = new CanonicalizationTransform(this.signatureParameters.getXmldsigNamespace(), "http://www.w3.org/2001/10/xml-exc-c14n#");
        dssTransformList.add(canonicalizationTransform);
        dssReference.setTransforms(dssTransformList);
        return dssReference;
    }

    private void assertEnvelopedSignaturePossible(DSSDocument document) {
        if (!DomUtils.isDOM((DSSDocument)document)) {
            throw new IllegalInputException("Enveloped signature cannot be created. Reason : the provided document is not XML!");
        }
        Document dom = DomUtils.buildDOM((DSSDocument)document);
        Element documentElement = dom.getDocumentElement();
        if (XMLDSigElement.SIGNATURE.isSameTagName(documentElement.getLocalName())) {
            throw new IllegalInputException("Unable to create an enveloped signature for another XML signature document!");
        }
    }

    private void assertEnvelopingSignatureWithEmbeddedXMLPossible(DSSDocument document) {
        if (!DomUtils.isDOM((DSSDocument)document)) {
            throw new IllegalInputException("Enveloping signature with embedded XML cannot be created. Reason : the provided document is not XML!");
        }
    }

    private DSSReference envelopingDSSReference(DSSDocument document, int index) {
        String refId = this.getReferenceId(index);
        DSSReference reference = new DSSReference();
        reference.setId(this.referenceIdPrefix + refId);
        reference.setContents(document);
        reference.setDigestMethodAlgorithm(this.digestAlgorithm);
        if (this.signatureParameters.isManifestSignature()) {
            this.assertEnvelopingSignatureWithEmbeddedXMLPossible(document);
            Document manifestDoc = DomUtils.buildDOM((DSSDocument)document);
            Element manifestElement = manifestDoc.getDocumentElement();
            this.assertXmlManifestSignaturePossible(manifestElement);
            reference.setType("http://www.w3.org/2000/09/xmldsig#Manifest");
            reference.setUri(DomUtils.toElementReference((String)manifestElement.getAttribute(XMLDSigAttribute.ID.getAttributeName())));
            CanonicalizationTransform xmlTransform = new CanonicalizationTransform(this.signatureParameters.getXmldsigNamespace(), "http://www.w3.org/2001/10/xml-exc-c14n#");
            reference.setTransforms(Collections.singletonList(xmlTransform));
        } else if (this.signatureParameters.isEmbedXML()) {
            this.assertEnvelopingSignatureWithEmbeddedXMLPossible(document);
            reference.setType("http://www.w3.org/2000/09/xmldsig#Object");
            reference.setUri(DomUtils.toElementReference((String)(OBJECT_ID_PREFIX + refId)));
            CanonicalizationTransform xmlTransform = new CanonicalizationTransform(this.signatureParameters.getXmldsigNamespace(), "http://www.w3.org/2001/10/xml-exc-c14n#");
            reference.setTransforms(Collections.singletonList(xmlTransform));
        } else {
            reference.setType("http://www.w3.org/2000/09/xmldsig#Object");
            reference.setUri(DomUtils.toElementReference((String)(OBJECT_ID_PREFIX + refId)));
            Base64Transform base64Transform = new Base64Transform(this.signatureParameters.getXmldsigNamespace());
            reference.setTransforms(Collections.singletonList(base64Transform));
        }
        return reference;
    }

    private void assertXmlManifestSignaturePossible(Element manifestElement) {
        String idAttr = manifestElement.getAttribute(XMLDSigAttribute.ID.getAttributeName());
        if (Utils.isStringBlank((String)idAttr)) {
            throw new IllegalInputException("Manifest signature is not possible for an XML file without Id attribute in the root element!");
        }
    }

    private DSSReference detachedDSSReference(DSSDocument document, int index) {
        DSSReference reference = new DSSReference();
        reference.setId(this.referenceIdPrefix + this.getReferenceId(index));
        if (Utils.isStringNotEmpty((String)document.getName())) {
            reference.setUri(DSSUtils.encodeURI((String)document.getName()));
        }
        reference.setContents(document);
        reference.setDigestMethodAlgorithm(this.digestAlgorithm);
        return reference;
    }

    private DSSReference internallyDetachedDSSReference(DSSDocument document, int index) {
        DSSReference reference = new DSSReference();
        reference.setId(this.referenceIdPrefix + this.getReferenceId(index));
        Document dom = DomUtils.buildDOM((DSSDocument)document);
        String identifier = DSSXMLUtils.getIDIdentifier(dom.getDocumentElement());
        Objects.requireNonNull(identifier, "ID not defined on the root xml element");
        reference.setUri(DomUtils.toElementReference((String)identifier));
        reference.setContents(document);
        reference.setDigestMethodAlgorithm(this.digestAlgorithm);
        ArrayList<DSSTransform> dssTransformList = new ArrayList<DSSTransform>();
        CanonicalizationTransform canonicalization = new CanonicalizationTransform(this.signatureParameters.getXmldsigNamespace(), "http://www.w3.org/2001/10/xml-exc-c14n#");
        dssTransformList.add(canonicalization);
        reference.setTransforms(dssTransformList);
        return reference;
    }

    private String getReferenceId(int index) {
        StringBuilder referenceId = new StringBuilder();
        if (this.signatureParameters != null) {
            referenceId.append(this.signatureParameters.getDeterministicId());
            referenceId.append("-");
        }
        referenceId.append(index);
        return referenceId.toString();
    }

    private DigestAlgorithm getReferenceDigestAlgorithmOrDefault(XAdESSignatureParameters params) {
        return params.getReferenceDigestAlgorithm() != null ? params.getReferenceDigestAlgorithm() : params.getDigestAlgorithm();
    }
}

