/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.scheduler.startup;

import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.startup.StartupProcessException;
import io.camunda.zeebe.scheduler.startup.StartupProcessShutdownException;
import io.camunda.zeebe.scheduler.startup.StartupProcessStepException;
import io.camunda.zeebe.scheduler.startup.StartupStep;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public final class StartupProcess<CONTEXT> {
    private final Logger logger;
    private final Queue<StartupStep<CONTEXT>> steps;
    private final Deque<StartupStep<CONTEXT>> startedSteps = new ArrayDeque<StartupStep<CONTEXT>>();
    private boolean startupCalled = false;
    private ActorFuture<CONTEXT> shutdownFuture;
    private ActorFuture<CONTEXT> startupFuture;
    private Map<String, String> loggingContext = Map.of();

    public StartupProcess(List<StartupStep<CONTEXT>> steps) {
        this(LoggerFactory.getLogger(StartupProcess.class), steps);
    }

    public StartupProcess(Logger logger, List<? extends StartupStep<CONTEXT>> steps) {
        this.steps = new ArrayDeque<StartupStep<CONTEXT>>((Collection)Objects.requireNonNull(steps));
        this.logger = Objects.requireNonNull(logger);
    }

    public StartupProcess(Map<String, String> loggingContext, Logger logger, List<? extends StartupStep<CONTEXT>> steps) {
        this.steps = new ArrayDeque<StartupStep<CONTEXT>>((Collection)Objects.requireNonNull(steps));
        this.logger = Objects.requireNonNull(logger);
        this.loggingContext = Objects.requireNonNull(loggingContext);
    }

    public ActorFuture<CONTEXT> startup(ConcurrencyControl concurrencyControl, CONTEXT context) {
        ActorFuture result = concurrencyControl.createFuture();
        concurrencyControl.run(() -> this.startupSynchronized(concurrencyControl, context, result));
        return result;
    }

    public ActorFuture<CONTEXT> shutdown(ConcurrencyControl concurrencyControl, CONTEXT context) {
        ActorFuture result = concurrencyControl.createFuture();
        concurrencyControl.run(() -> this.shutdownSynchronized(concurrencyControl, context, result));
        return result;
    }

    private void startupSynchronized(ConcurrencyControl concurrencyControl, CONTEXT context, ActorFuture<CONTEXT> startupFuture) {
        this.setCustomMDC();
        this.logger.debug("Startup was called with context: {}", context);
        this.clearCustomMDC();
        if (this.startupCalled) {
            throw new IllegalStateException("startup(...) must only be called once");
        }
        this.startupCalled = true;
        this.startupFuture = startupFuture;
        concurrencyControl.runOnCompletion(startupFuture, (result, error) -> {
            this.startupFuture = null;
        });
        ArrayDeque<StartupStep<CONTEXT>> stepsToStart = new ArrayDeque<StartupStep<CONTEXT>>(this.steps);
        this.proceedWithStartupSynchronized(concurrencyControl, stepsToStart, context, startupFuture);
    }

    private void proceedWithStartupSynchronized(ConcurrencyControl concurrencyControl, Queue<StartupStep<CONTEXT>> stepsToStart, CONTEXT context, ActorFuture<CONTEXT> startupFuture) {
        if (stepsToStart.isEmpty()) {
            startupFuture.complete(context);
            this.setCustomMDC();
            this.logger.debug("Finished startup process");
            this.clearCustomMDC();
        } else if (this.shutdownFuture != null) {
            this.setCustomMDC();
            this.logger.info("Aborting startup process because shutdown was called");
            this.clearCustomMDC();
            startupFuture.completeExceptionally(new StartupProcessShutdownException("Aborting startup process because shutdown was called"));
        } else {
            StartupStep<CONTEXT> stepToStart = stepsToStart.poll();
            this.startedSteps.push(stepToStart);
            this.logCurrentStepSynchronized("Startup", stepToStart);
            long before = System.nanoTime();
            ActorFuture<CONTEXT> stepStartupFuture = stepToStart.startup(context);
            concurrencyControl.runOnCompletion(stepStartupFuture, (contextReturnedByStep, error) -> {
                long completedAt = System.nanoTime();
                this.logger.debug("StartupStep {} startup completed (error={}) took {} millis ", new Object[]{stepToStart.getName(), error != null, (double)(completedAt - before) / 1000000.0});
                if (error != null) {
                    this.completeStartupFutureExceptionallySynchronized(startupFuture, stepToStart, (Throwable)error);
                } else {
                    this.proceedWithStartupSynchronized(concurrencyControl, stepsToStart, contextReturnedByStep, startupFuture);
                }
            });
        }
    }

    private void completeStartupFutureExceptionallySynchronized(ActorFuture<CONTEXT> startupFuture, StartupStep<CONTEXT> stepToStart, Throwable error) {
        this.setCustomMDC();
        this.logger.warn("Aborting startup process due to exception during step " + stepToStart.getName(), error);
        this.clearCustomMDC();
        startupFuture.completeExceptionally(this.aggregateExceptionsSynchronized("Startup", Collections.singletonList(new StartupProcessStepException(stepToStart.getName(), error))));
    }

    private void shutdownSynchronized(ConcurrencyControl concurrencyControl, CONTEXT context, ActorFuture<CONTEXT> resultFuture) {
        this.setCustomMDC();
        this.logger.debug("Shutdown was called with context: {}", context);
        this.clearCustomMDC();
        if (this.shutdownFuture == null) {
            this.shutdownFuture = resultFuture;
            if (this.startupFuture != null) {
                concurrencyControl.runOnCompletion(this.startupFuture, (contextReturnedByStartup, error) -> {
                    Object contextForShutdown = error == null ? contextReturnedByStartup : context;
                    this.proceedWithShutdownSynchronized(concurrencyControl, contextForShutdown, this.shutdownFuture, new ArrayList<StartupProcessStepException>());
                });
            } else {
                this.proceedWithShutdownSynchronized(concurrencyControl, context, this.shutdownFuture, new ArrayList<StartupProcessStepException>());
            }
        } else {
            this.setCustomMDC();
            this.logger.info("Shutdown already in progress");
            this.clearCustomMDC();
            concurrencyControl.runOnCompletion(this.shutdownFuture, (contextReturnedByShutdown, error) -> {
                if (error != null) {
                    resultFuture.completeExceptionally((Throwable)error);
                } else {
                    resultFuture.complete(contextReturnedByShutdown);
                }
            });
        }
    }

    private void proceedWithShutdownSynchronized(ConcurrencyControl concurrencyControl, CONTEXT context, ActorFuture<CONTEXT> shutdownFuture, List<StartupProcessStepException> collectedExceptions) {
        if (this.startedSteps.isEmpty()) {
            this.completeShutdownFutureSynchronized(context, shutdownFuture, collectedExceptions);
        } else {
            StartupStep<CONTEXT> stepToShutdown = this.startedSteps.pop();
            long before = System.nanoTime();
            this.logCurrentStepSynchronized("Shutdown", stepToShutdown);
            ActorFuture<CONTEXT> shutdownStepFuture = stepToShutdown.shutdown(context);
            concurrencyControl.runOnCompletion(shutdownStepFuture, (contextReturnedByShutdown, error) -> {
                Object contextToUse;
                long completedAt = System.nanoTime();
                if (error != null) {
                    collectedExceptions.add(new StartupProcessStepException(stepToShutdown.getName(), (Throwable)error));
                    contextToUse = context;
                } else {
                    contextToUse = contextReturnedByShutdown;
                }
                this.logger.debug("StartupStep {} shutdown completed (error={}) took {} millis ", new Object[]{stepToShutdown.getName(), error != null, (double)(completedAt - before) / 1000000.0});
                this.proceedWithShutdownSynchronized(concurrencyControl, contextToUse, shutdownFuture, collectedExceptions);
            });
        }
    }

    private void completeShutdownFutureSynchronized(CONTEXT context, ActorFuture<CONTEXT> shutdownFuture, List<StartupProcessStepException> collectedExceptions) {
        if (collectedExceptions.isEmpty()) {
            shutdownFuture.complete(context);
            this.setCustomMDC();
            this.logger.debug("Finished shutdown process");
            this.clearCustomMDC();
        } else {
            Throwable umbrellaException = this.aggregateExceptionsSynchronized("Shutdown", collectedExceptions);
            shutdownFuture.completeExceptionally(umbrellaException);
            this.setCustomMDC();
            this.logger.warn(umbrellaException.getMessage(), umbrellaException);
        }
    }

    private Throwable aggregateExceptionsSynchronized(String operation, List<StartupProcessStepException> exceptions) {
        List failedSteps = exceptions.stream().map(StartupProcessStepException::getStepName).collect(Collectors.toList());
        String message = String.format("%s failed in the following steps: %s. See suppressed exceptions for details.", operation, failedSteps);
        StartupProcessException exception = new StartupProcessException(message);
        exceptions.forEach(exception::addSuppressed);
        return exception;
    }

    private void logCurrentStepSynchronized(String process, StartupStep<CONTEXT> step) {
        this.setCustomMDC();
        this.logger.info(process + " " + step.getName());
        this.clearCustomMDC();
    }

    private void setCustomMDC() {
        this.loggingContext.forEach((key, value) -> {
            if (key != null && value != null) {
                MDC.put((String)key, (String)value);
            }
        });
    }

    private void clearCustomMDC() {
        this.loggingContext.keySet().forEach(key -> {
            if (key != null) {
                MDC.remove((String)key);
            }
        });
    }
}

