/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.controlcenter.streams.alert;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.inject.Provider;
import io.confluent.command.record.Command;
import io.confluent.controlcenter.alert.TriggerEventUtil;
import io.confluent.controlcenter.record.Controlcenter;
import io.confluent.controlcenter.streams.internals.KeyValueStoreFacade;
import io.confluent.controlcenter.util.KvQuery;
import io.confluent.monitoring.common.Clock;
import io.confluent.monitoring.common.TimeBucket;
import io.confluent.serializers.OrderedKeyUberSerde;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.TimeUnit;
import org.apache.kafka.common.utils.Bytes;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.kstream.Transformer;
import org.apache.kafka.streams.kstream.TransformerSupplier;
import org.apache.kafka.streams.processor.ProcessorContext;
import org.apache.kafka.streams.processor.StateStore;
import org.apache.kafka.streams.state.KeyValueIterator;
import org.apache.kafka.streams.state.KeyValueStore;
import org.apache.kafka.streams.state.ReadOnlyKeyValueStore;
import org.apache.kafka.streams.state.TimestampedKeyValueStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MonitoringTriggerTransformerSupplier
implements TransformerSupplier<Controlcenter.WindowedClusterGroup, Controlcenter.TriggerMeasurement, KeyValue<Void, Iterable<KeyValue<Bytes, Controlcenter.TriggerEvent>>>> {
    private static final Logger log = LoggerFactory.getLogger(MonitoringTriggerTransformerSupplier.class);
    private static final long PUNCTUATE_INTERVAL_MS = TimeBucket.SIZE;
    private static final long DEFAULT_MAX_PUNCTUATE_MS = TimeUnit.SECONDS.toMillis(1L);
    private static final long MAX_WAIT_TO_MOVE_C3_CLOCK = 8L * TimeBucket.SIZE;
    private static final long DEFAULT_MAX_PUNCTUATE_ITEMS = 8192L;
    private final String storeName;
    private final Clock clock;
    private final OrderedKeyUberSerde<Controlcenter.WindowedClusterGroup> keySerde;
    private final Provider<? extends ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage>> triggerConfigsProvider;
    private final int numInternalPartitions;

    public MonitoringTriggerTransformerSupplier(String storeName, Clock clock, OrderedKeyUberSerde<Controlcenter.WindowedClusterGroup> keySerde, Provider<? extends ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage>> triggerConfigsProvider, int numInternalPartitions) {
        this.storeName = storeName;
        this.clock = clock;
        this.keySerde = keySerde;
        this.triggerConfigsProvider = triggerConfigsProvider;
        this.numInternalPartitions = numInternalPartitions;
    }

    public Transformer<Controlcenter.WindowedClusterGroup, Controlcenter.TriggerMeasurement, KeyValue<Void, Iterable<KeyValue<Bytes, Controlcenter.TriggerEvent>>>> get() {
        return new MonitoringTriggerTransformer(PUNCTUATE_INTERVAL_MS, DEFAULT_MAX_PUNCTUATE_MS, 8192L, this.storeName, this.triggerConfigsProvider);
    }

    private class MonitoringTriggerTransformer
    implements Transformer<Controlcenter.WindowedClusterGroup, Controlcenter.TriggerMeasurement, KeyValue<Void, Iterable<KeyValue<Bytes, Controlcenter.TriggerEvent>>>> {
        private final Bytes lastKey = Bytes.wrap((byte[])new byte[]{-1});
        private final String storeName;
        private final long maxPunctuateMs;
        private final long punctuateIntervalMs;
        private final Provider<? extends ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage>> triggerConfigsProvider;
        private final long maxPunctuateItems;
        private ProcessorContext context;
        private KeyValueStore<Bytes, Controlcenter.TriggerMeasurement> kvStore;
        private long lastPunctuateTimeMs = -1L;
        private long approxC3Time = 0L;
        private long lastTimeMoveC3Clock = -1L;
        private Bytes lastPunctuateKey = null;

        public MonitoringTriggerTransformer(long punctuateIntervalMs, long maxPunctuateMs, long maxPunctuateItems, String storeName, Provider<? extends ReadOnlyKeyValueStore<Command.CommandKey, Command.CommandMessage>> triggerConfigsProvider) {
            this.punctuateIntervalMs = punctuateIntervalMs;
            this.maxPunctuateMs = maxPunctuateMs;
            this.maxPunctuateItems = maxPunctuateItems;
            this.storeName = storeName;
            this.triggerConfigsProvider = triggerConfigsProvider;
        }

        public void init(ProcessorContext context) {
            this.context = context;
            StateStore rawkvStore = this.context.getStateStore(this.storeName);
            this.kvStore = rawkvStore instanceof TimestampedKeyValueStore ? new KeyValueStoreFacade<Bytes, Controlcenter.TriggerMeasurement>((TimestampedKeyValueStore)rawkvStore) : (KeyValueStore)rawkvStore;
        }

        public KeyValue<Void, Iterable<KeyValue<Bytes, Controlcenter.TriggerEvent>>> transform(Controlcenter.WindowedClusterGroup key, Controlcenter.TriggerMeasurement triggerMeasurement) {
            if (triggerMeasurement == null || triggerMeasurement.getMeasurementCase() == Controlcenter.TriggerMeasurement.MeasurementCase.MEASUREMENT_NOT_SET && !triggerMeasurement.getHeartbeat()) {
                log.trace("Ignoring null trigger measurement ");
                return null;
            }
            Bytes storeKey = MonitoringTriggerTransformerSupplier.this.keySerde.key((Object)key);
            if (triggerMeasurement.getHeartbeat()) {
                log.trace("Monitoring measurement C3 heartbeat");
                this.updateC3Clock(null, MonitoringTriggerTransformerSupplier.this.clock.currentTimeMillis());
                ArrayList<KeyValue> ret = new ArrayList<KeyValue>();
                for (int partition = 0; partition < MonitoringTriggerTransformerSupplier.this.numInternalPartitions; ++partition) {
                    Controlcenter.TriggerEvent msg = Controlcenter.TriggerEvent.newBuilder().setHeartbeat(true).setInternalTopicPartition(partition).setTimestamp(this.approxC3Time).setWindow(triggerMeasurement.getWindow()).build();
                    ret.add(KeyValue.pair((Object)Bytes.wrap((byte[])new byte[0]), (Object)msg));
                }
                KeyValue<Void, Iterable<KeyValue<Bytes, Controlcenter.TriggerEvent>>> pair = this.maybePunctuate();
                if (pair != null) {
                    Iterables.addAll(ret, (Iterable)((Iterable)pair.value));
                }
                return KeyValue.pair(null, ret);
            }
            if (log.isTraceEnabled()) {
                log.trace("trigger transform storeKey={} measurement={}", (Object)storeKey, (Object)TriggerEventUtil.logMeasurement(triggerMeasurement));
            }
            this.updateC3Clock(triggerMeasurement.getArrivalTime(), MonitoringTriggerTransformerSupplier.this.clock.currentTimeMillis());
            try {
                Controlcenter.TriggerMeasurement storedMeasurement = (Controlcenter.TriggerMeasurement)this.kvStore.get((Object)storeKey);
                long arrivalTime = Math.min(triggerMeasurement.getArrivalTime(), storedMeasurement == null ? Long.MAX_VALUE : storedMeasurement.getArrivalTime());
                triggerMeasurement = Controlcenter.TriggerMeasurement.newBuilder((Controlcenter.TriggerMeasurement)triggerMeasurement).setComponent(key.getComponent()).setClusterId(key.getClusterId()).setWindow(key.getWindow()).setArrivalTime(arrivalTime).build();
                boolean isTriggerable = false;
                try (KeyValueIterator iter = ((ReadOnlyKeyValueStore)this.triggerConfigsProvider.get()).all();){
                    isTriggerable = TriggerEventUtil.isTriggerable(triggerMeasurement, (Iterator<KeyValue<Command.CommandKey, Command.CommandMessage>>)iter);
                }
                catch (Exception e) {
                    log.error("Failure while checking if event is triggerable.", (Throwable)e);
                }
                if (isTriggerable) {
                    if (log.isTraceEnabled()) {
                        log.trace("Measurement for key={} is in triggerable state: {}", (Object)storeKey, (Object)TriggerEventUtil.logMeasurement(triggerMeasurement));
                    }
                    this.kvStore.put((Object)storeKey, (Object)triggerMeasurement);
                } else if (storedMeasurement != null) {
                    this.kvStore.delete((Object)storeKey);
                }
            }
            catch (Exception e) {
                log.warn("Failed to update trigger store. May not fire a trigger when needed.", (Throwable)e);
            }
            return this.maybePunctuate();
        }

        private KeyValue<Void, Iterable<KeyValue<Bytes, Controlcenter.TriggerEvent>>> maybePunctuate() {
            if (this.lastPunctuateTimeMs < 0L) {
                this.lastPunctuateTimeMs = this.approxC3Time;
            }
            if (this.approxC3Time - this.lastPunctuateTimeMs < this.punctuateIntervalMs) {
                return null;
            }
            this.lastPunctuateTimeMs = this.approxC3Time;
            long punctuateTimeout = MonitoringTriggerTransformerSupplier.this.clock.currentTimeMillis() + this.maxPunctuateMs;
            ArrayList<KeyValue<Bytes, Controlcenter.TriggerEvent>> triggerList = new ArrayList<KeyValue<Bytes, Controlcenter.TriggerEvent>>();
            try (KeyValueIterator<Bytes, Controlcenter.TriggerMeasurement> iter = KvQuery.rangeFrom(this.kvStore, this.lastPunctuateKey);){
                long count = 0L;
                while (iter.hasNext()) {
                    KeyValue entry = (KeyValue)iter.next();
                    if (entry == null) continue;
                    if (entry.value == null) {
                        log.debug("removing null value key={} from store={}", entry.key, (Object)this.storeName);
                        this.kvStore.delete(entry.key);
                    } else {
                        ArrayList triggerConfigs;
                        if (log.isTraceEnabled()) {
                            log.trace("checking key={} measurement={} store={}", new Object[]{entry.key, TriggerEventUtil.logMeasurement((Controlcenter.TriggerMeasurement)entry.value), this.storeName});
                        }
                        Bytes triggerKey = MonitoringTriggerTransformerSupplier.this.keySerde.key((Object)Controlcenter.WindowedClusterGroup.newBuilder((Controlcenter.WindowedClusterGroup)((Controlcenter.WindowedClusterGroup)MonitoringTriggerTransformerSupplier.this.keySerde.toProto((Bytes)entry.key))).setWindow(0L).build());
                        boolean createTrigger = false;
                        try (KeyValueIterator configIterator = ((ReadOnlyKeyValueStore)this.triggerConfigsProvider.get()).all();){
                            triggerConfigs = Lists.newArrayList((Iterator)configIterator);
                        }
                        createTrigger = TriggerEventUtil.createTriggerEventsIfTriggerable(triggerKey, (Controlcenter.TriggerMeasurement)entry.value, triggerList, this.approxC3Time, triggerConfigs);
                        if (createTrigger) {
                            this.kvStore.delete(entry.key);
                        } else {
                            this.kvStore.put(entry.key, (Object)Controlcenter.TriggerMeasurement.newBuilder((Controlcenter.TriggerMeasurement)((Controlcenter.TriggerMeasurement)entry.value)).setLastTriggerEventTimestamp(this.approxC3Time).build());
                        }
                    }
                    this.lastPunctuateKey = (Bytes)entry.key;
                    if (++count > this.maxPunctuateItems || MonitoringTriggerTransformerSupplier.this.clock.currentTimeMillis() > punctuateTimeout) {
                        log.debug("stopping punctuate at key={} store={} count={}", new Object[]{this.lastPunctuateKey, this.storeName, count});
                        break;
                    }
                    if (!iter.hasNext()) {
                        log.debug("finished all punctuation for store={}", (Object)this.storeName);
                        this.lastPunctuateKey = null;
                    }
                    log.debug("punctuated on count={} records", (Object)count);
                }
            }
            catch (Exception e) {
                log.error("failure during punctuate for store={} e={}", (Object)this.storeName, (Object)e.getMessage());
            }
            if (triggerList.isEmpty()) {
                return null;
            }
            return KeyValue.pair(null, triggerList);
        }

        public void close() {
            log.info("closing store={}", (Object)this.storeName);
        }

        private void updateC3Clock(Long arrivalTime, Long currentTime) {
            if (arrivalTime != null && arrivalTime > this.approxC3Time) {
                this.approxC3Time = arrivalTime;
                this.lastTimeMoveC3Clock = currentTime;
                log.trace("Moved C3 clock based on arrival time: {}", (Object)this.approxC3Time);
            }
            if (this.lastTimeMoveC3Clock < 0L) {
                this.lastTimeMoveC3Clock = currentTime;
            }
            if (this.lastTimeMoveC3Clock + MAX_WAIT_TO_MOVE_C3_CLOCK < currentTime) {
                this.approxC3Time += currentTime - this.lastTimeMoveC3Clock;
                this.lastTimeMoveC3Clock = currentTime;
                log.trace("Moved C3 clock based on wallclock time: {}", (Object)this.approxC3Time);
            }
        }
    }
}

