/*
 * Copyright 2017-2024 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.context.conditions;

import io.micronaut.context.condition.Condition;
import io.micronaut.context.condition.ConditionContext;
import io.micronaut.core.annotation.AnnotationClassValue;
import io.micronaut.core.annotation.Internal;
import io.micronaut.core.annotation.UsedByGeneratedCode;

import java.util.Arrays;
import java.util.Objects;

/**
 * Missing classes condition.
 *
 * @param classes The classes
 * @author Denis Stepanov
 * @since 4.6
 */
@UsedByGeneratedCode
@Internal
public record MatchesAbsenceOfClassesCondition(AnnotationClassValue<?>[] classes) implements Condition {
    @Override
    public boolean matches(ConditionContext context) {
        for (AnnotationClassValue<?> classValue : classes) {
            if (classValue.getType().isPresent()) {
                context.fail("Class [" + classValue.getName() + "] is not absent");
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        MatchesAbsenceOfClassesCondition that = (MatchesAbsenceOfClassesCondition) o;
        return Objects.deepEquals(classes, that.classes);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(classes);
    }

    @Override
    public String toString() {
        return "MatchesAbsenceOfClassesCondition{" +
            "classes=" + Arrays.toString(classes) +
            '}';
    }
}
