/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.kafka.common.Cell;
import org.apache.kafka.common.CellLoad;
import org.apache.kafka.common.CellMigrationState;
import org.apache.kafka.common.CellState;
import org.apache.kafka.common.Tenant;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.message.AlterCellMigrationResponseData;
import org.apache.kafka.common.message.AlterCellResponseData;
import org.apache.kafka.common.message.AssignBrokersToCellResponseData;
import org.apache.kafka.common.message.CreateCellResponseData;
import org.apache.kafka.common.message.DeleteCellResponseData;
import org.apache.kafka.common.message.DescribeCellMigrationResponseData;
import org.apache.kafka.common.message.DescribeCellsResponseData;
import org.apache.kafka.common.message.UnAssignBrokersFromCellResponseData;
import org.apache.kafka.common.metadata.CellMigrationRecord;
import org.apache.kafka.common.metadata.CellRecord;
import org.apache.kafka.common.metadata.MetadataRecordType;
import org.apache.kafka.common.metadata.RemoveCellRecord;
import org.apache.kafka.common.metadata.RemoveTenantRecord;
import org.apache.kafka.common.metadata.TenantRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.AlterCellMigrationRequest;
import org.apache.kafka.common.requests.AlterCellRequest;
import org.apache.kafka.common.requests.ApiError;
import org.apache.kafka.common.requests.AssignBrokersToCellRequest;
import org.apache.kafka.common.requests.CreateCellRequest;
import org.apache.kafka.common.requests.DeleteCellRequest;
import org.apache.kafka.common.requests.DescribeCellMigrationRequest;
import org.apache.kafka.common.requests.DescribeCellsRequest;
import org.apache.kafka.common.requests.UnAssignBrokersFromCellRequest;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.controller.FeatureControlManager;
import org.apache.kafka.metadata.placement.CellAssignor;
import org.apache.kafka.metadata.placement.CellDescriber;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.mutable.BoundedList;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.apache.kafka.timeline.TimelineHashMap;
import org.slf4j.Logger;

public class CellControlManager
implements CellDescriber {
    public static final Set<CellState> PROHIBITED_TARGET_STATES = Collections.unmodifiableSet(new HashSet<CellState>(List.of(CellState.QUARANTINED, CellState.EXCLUDED)));
    public static final Set<CellState> PROHIBITED_SOURCE_STATES = Set.of(CellState.QUARANTINED);
    public static final CellState INITIAL_CELL_STATE = CellState.READY;
    public static final short K2_DEFAULT_CELL_MIN_SIZE = 3;
    public static final short K2_DEFAULT_CELL_MAX_SIZE = 300;
    private final Logger log;
    private final TimelineHashMap<Integer, Cell> cellIdToCell;
    private final TimelineHashMap<String, Tenant> tenantIdToTenant;
    private final TimelineHashMap<Integer, Integer> cellIdToNumTenants;
    private final FeatureControlManager featureControl;
    private final short cellSize;
    private final short defaultCellMinSize;
    private final short defaultCellMaxSize;
    private final boolean isImplicitCellCreationEnabled;
    private final boolean cellsEnabled;
    private final CellAssignor cellAssignor;
    private final short replicationFactor;
    private final int k2CellBaseBrokerId;
    private volatile CellMigrationState cellMigrationState;

    public CellControlManager(LogContext logContext, SnapshotRegistry snapshotRegistry, FeatureControlManager featureControl, CellAssignor cellAssignor, short cellSize, short defaultCellMinSize, short defaultCellMaxSize, short replicationFactor, boolean isImplicitCellCreationEnabled, boolean cellsEnabled, int k2CellBaseBrokerId) {
        CellAssignor.confirmInitialCellStateValid(defaultCellMinSize, defaultCellMaxSize, cellSize);
        this.log = logContext.logger(CellControlManager.class);
        this.cellIdToCell = new TimelineHashMap(snapshotRegistry, 0);
        this.tenantIdToTenant = new TimelineHashMap(snapshotRegistry, 0);
        this.cellIdToNumTenants = new TimelineHashMap(snapshotRegistry, 0);
        this.featureControl = featureControl;
        this.cellSize = cellSize;
        this.defaultCellMinSize = defaultCellMinSize;
        this.defaultCellMaxSize = defaultCellMaxSize;
        this.replicationFactor = replicationFactor;
        this.isImplicitCellCreationEnabled = isImplicitCellCreationEnabled;
        this.cellsEnabled = cellsEnabled;
        this.cellAssignor = cellAssignor;
        this.cellMigrationState = CellMigrationState.INACTIVE;
        this.k2CellBaseBrokerId = k2CellBaseBrokerId;
    }

    CellControlManager(LogContext logContext, Time time, SnapshotRegistry snapshotRegistry, FeatureControlManager featureControl, Random random, short cellSize, short defaultCellMinSize, short defaultCellMaxSize, boolean isImplicitCellCreationEnabled, int k2CellBaseBrokerId) {
        this(logContext, snapshotRegistry, featureControl, new CellAssignor(random, time), cellSize, defaultCellMinSize, defaultCellMaxSize, 3, isImplicitCellCreationEnabled, isImplicitCellCreationEnabled, k2CellBaseBrokerId);
    }

    public ControllerResult<DescribeCellsResponseData> describeCells(DescribeCellsRequest request, Set<Integer> usableBrokers) {
        this.confirmCellsSupported();
        DescribeCellsResponseData data = new DescribeCellsResponseData();
        ArrayList cellIds = request.cellIds().isEmpty() ? new ArrayList(this.cellIdToCell.keySet()) : new ArrayList(request.cellIds());
        Collections.sort(cellIds);
        ArrayList<DescribeCellsResponseData.Cell> cells = new ArrayList<DescribeCellsResponseData.Cell>();
        HashSet assignedBrokers = new HashSet();
        for (Integer cellId : cellIds) {
            Optional<Cell> cellOpt = this.getCell(cellId);
            if (!cellOpt.isPresent()) {
                return ControllerResult.atomicOf(List.of(), data.setErrorCode(Errors.CELL_NOT_FOUND.code()).setErrorMessage(String.format("Target cell %s does not exist", cellId)));
            }
            Cell cell = cellOpt.get();
            cells.add(new DescribeCellsResponseData.Cell().setCellId(cell.cellId()).setState(cell.state().code()).setBrokers(cell.brokersList()).setOpenForTenantAssignment(CellAssignor.isCellOpenForAssignment(cell, usableBrokers, this.replicationFactor)));
            assignedBrokers.addAll(cell.brokersList());
        }
        if (this.isCellMigrationEnabled() && request.cellIds().isEmpty()) {
            usableBrokers.removeAll(assignedBrokers);
            if (!usableBrokers.isEmpty()) {
                cells.add(new DescribeCellsResponseData.Cell().setCellId(-1).setState(CellState.READY.code()).setBrokers(new ArrayList<Integer>(usableBrokers)).setOpenForTenantAssignment(CellAssignor.isDefaultCellOpenForAssignment(usableBrokers, this.replicationFactor)));
            }
        }
        return ControllerResult.atomicOf(List.of(), data.setCells(cells).setCellsEnabled(this.isCellsEnabled()));
    }

    public ControllerResult<CreateCellResponseData> createCell(CreateCellRequest request) {
        this.confirmCellsSupported();
        CreateCellResponseData data = new CreateCellResponseData();
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        ApiError err = this.createCell(request.cellId(), CellState.toEnum((byte)request.state()), this.defaultCellMinSize, this.defaultCellMaxSize, ((List)records)::add);
        if (!ApiError.NONE.equals((Object)err)) {
            return ControllerResult.atomicOf(List.of(), data.setErrorCode(err.error().code()).setErrorMessage(err.message()));
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data);
    }

    public ControllerResult<AlterCellResponseData> alterCell(AlterCellRequest request, Set<Integer> usableBrokers) {
        this.confirmCellsSupported();
        AlterCellResponseData data = new AlterCellResponseData();
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        Optional<Cell> cellOpt = this.getCell(request.cellId());
        if (!cellOpt.isPresent()) {
            return ControllerResult.atomicOf(List.of(), data.setErrorCode(Errors.CELL_NOT_FOUND.code()).setErrorMessage(String.format("Cell %s does not exist", request.cellId())));
        }
        Cell cell = cellOpt.get();
        ApiError err = this.updateCellMetadata(cell.cellId(), CellState.toEnum((byte)request.state()), cell.minSize(), cell.maxSize(), usableBrokers, ((List)records)::add);
        if (!ApiError.NONE.equals((Object)err)) {
            return ControllerResult.atomicOf(List.of(), data.setErrorCode(err.error().code()).setErrorMessage(err.message()));
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data);
    }

    public ControllerResult<AssignBrokersToCellResponseData> assignBrokersToCell(AssignBrokersToCellRequest request, Set<Integer> usableBrokers) {
        this.confirmCellsSupported();
        AssignBrokersToCellResponseData data = new AssignBrokersToCellResponseData();
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        ApiError err = this.cellIdToCell.containsKey((Object)request.cellId()) ? this.assignBrokersToCell(new HashSet<Integer>(request.brokers()), request.cellId(), request.force(), usableBrokers, ((List)records)::add) : new ApiError(Errors.CELL_NOT_FOUND, String.format("Target cell %s does not exist", request.cellId()));
        if (!ApiError.NONE.equals((Object)err)) {
            return ControllerResult.atomicOf(List.of(), data.setErrorCode(err.error().code()).setErrorMessage(err.message()));
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data);
    }

    public ControllerResult<UnAssignBrokersFromCellResponseData> unassignBrokersFromCell(UnAssignBrokersFromCellRequest request, Set<Integer> usableBrokers) {
        this.confirmCellsSupported();
        UnAssignBrokersFromCellResponseData data = new UnAssignBrokersFromCellResponseData();
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        ApiError err = this.unassignBrokersFromCell(new HashSet<Integer>(request.brokers()), request.force(), usableBrokers, ((List)records)::add);
        if (!ApiError.NONE.equals((Object)err)) {
            return ControllerResult.atomicOf(List.of(), data.setErrorCode(err.error().code()).setErrorMessage(err.message()));
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data);
    }

    public ControllerResult<DeleteCellResponseData> deleteCell(DeleteCellRequest request) {
        this.confirmCellsSupported();
        DeleteCellResponseData data = new DeleteCellResponseData();
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        int numTenantsAssignedToCell = (Integer)this.cellIdToNumTenants.getOrDefault((Object)request.cellId(), (Object)0);
        if (numTenantsAssignedToCell > 0) {
            String errorMsg = String.format("Did not delete cell %s since there are %s tenants that are still assigned", request.cellId(), numTenantsAssignedToCell);
            return ControllerResult.atomicOf(List.of(), data.setErrorCode(Errors.INVALID_REQUEST.code()).setErrorMessage(errorMsg));
        }
        ApiError err = this.deleteCell(request.cellId(), ((List)records)::add);
        if (!ApiError.NONE.equals((Object)err)) {
            return ControllerResult.atomicOf(List.of(), data.setErrorCode(err.error().code()).setErrorMessage(err.message()));
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data);
    }

    public ControllerResult<AlterCellMigrationResponseData> alterCellMigration(AlterCellMigrationRequest request) {
        this.confirmCellMigrationSupported();
        AlterCellMigrationResponseData data = new AlterCellMigrationResponseData();
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        ApiError err = this.updateCellMigrationState(this.cellMigrationState, CellMigrationState.toEnum((byte)request.state()), ((List)records)::add);
        if (!ApiError.NONE.equals((Object)err)) {
            return ControllerResult.atomicOf(List.of(), data.setErrorCode(err.error().code()).setErrorMessage(err.message()));
        }
        this.cellMigrationState = CellMigrationState.toEnum((byte)request.state());
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data);
    }

    public ControllerResult<DescribeCellMigrationResponseData> describeCellMigration(DescribeCellMigrationRequest request) {
        this.confirmCellMigrationSupported();
        DescribeCellMigrationResponseData data = new DescribeCellMigrationResponseData();
        return ControllerResult.atomicOf(List.of(), data.setCellMigrationState(this.cellMigrationState.toString()));
    }

    @Override
    public int getBrokerCellId(int brokerId) {
        return this.cellIdToCell.values().stream().filter(cell -> cell.brokers().contains(brokerId)).findFirst().map(Cell::cellId).orElse(-1);
    }

    void replay(CellRecord cellRecord) {
        Cell cell = new Cell(cellRecord.cellId(), new HashSet<Integer>(cellRecord.brokers()), CellState.toEnum((byte)cellRecord.state()), cellRecord.minSize(), cellRecord.maxSize());
        this.cellIdToCell.put((Object)cell.cellId(), (Object)cell);
        this.log.info("Replayed CellRecord for {}", (Object)cell);
    }

    void replay(RemoveCellRecord removeCellRecord) {
        this.cellIdToCell.remove((Object)removeCellRecord.cellId());
        this.log.info("Replayed RemoveCellRecord removing cell id {}", (Object)removeCellRecord.cellId());
    }

    void replay(TenantRecord tenantRecord) {
        String tenantId = tenantRecord.tenantId();
        Tenant tenant = (Tenant)this.tenantIdToTenant.get((Object)tenantId);
        if (tenant != null) {
            tenant.cellIds().stream().filter(arg_0 -> this.cellIdToNumTenants.containsKey(arg_0)).forEach(cellId -> this.cellIdToNumTenants.put(cellId, (Object)((Integer)this.cellIdToNumTenants.get(cellId) - 1)));
        }
        this.log.info("Replayed TenantRecord for {}", (Object)tenant);
        List<Integer> cellIdsToUse = CellControlManager.getCellIdsFromTenantRecord(tenantRecord);
        for (Integer cellId2 : cellIdsToUse) {
            int newNumTenants = (Integer)this.cellIdToNumTenants.getOrDefault((Object)cellId2, (Object)0) + 1;
            this.log.info("There are now {} tenant(s) in this cell {}.", (Object)newNumTenants, (Object)cellId2);
            this.cellIdToNumTenants.put((Object)cellId2, (Object)newNumTenants);
        }
        this.tenantIdToTenant.put((Object)tenantRecord.tenantId(), (Object)new Tenant(tenantRecord.tenantId(), cellIdsToUse));
    }

    static List<Integer> getCellIdsFromTenantRecord(TenantRecord record) {
        return record.cellIds().isEmpty() ? List.of(Integer.valueOf(record.cellId())) : record.cellIds();
    }

    void replay(RemoveTenantRecord removeTenantRecord) {
        String tenantId = removeTenantRecord.tenantId();
        Tenant tenant = (Tenant)this.tenantIdToTenant.get((Object)tenantId);
        if (tenant != null) {
            tenant.cellIds().stream().filter(arg_0 -> this.cellIdToNumTenants.containsKey(arg_0)).forEach(cellId -> this.cellIdToNumTenants.put(cellId, (Object)((Integer)this.cellIdToNumTenants.get(cellId) - 1)));
            this.tenantIdToTenant.remove((Object)removeTenantRecord.tenantId());
        }
        this.log.info("Replayed RemoveTenantRecord for {}.", (Object)tenantId);
        for (Integer cellId2 : tenant.cellIds()) {
            this.log.info("There are now {} tenant(s) in this cell {}.", this.cellIdToNumTenants.get((Object)cellId2), (Object)cellId2);
        }
    }

    List<Integer> brokersInQuarantinedAndExcludedCells() {
        if (!this.isCellsEnabled()) {
            return List.of();
        }
        return this.cellIdToCell.values().stream().filter(cell -> cell.state() == CellState.EXCLUDED || cell.state() == CellState.QUARANTINED).flatMap(cell -> cell.brokers().stream()).collect(Collectors.toList());
    }

    void replay(CellMigrationRecord cellMigrationRecord) {
        this.cellMigrationState = CellMigrationState.toEnum((byte)cellMigrationRecord.state());
        this.log.info("Replayed CellMigration Record for {}", (Object)cellMigrationRecord.state());
    }

    int createCellForBroker(int brokerId, Consumer<ApiMessageAndVersion> recordConsumer) {
        Optional<Cell> cellOpt = this.getBrokerCell(brokerId);
        if (cellOpt.isPresent()) {
            return cellOpt.get().cellId();
        }
        int targetCellId = Cell.getImplicitBrokerCellId((int)brokerId, (int)this.cellSize, (int)this.k2CellBaseBrokerId, (int)1000, (int)0);
        Cell cell = (Cell)this.cellIdToCell.computeIfAbsent((Object)targetCellId, id -> {
            if (id == 1000) {
                return new Cell(id.intValue(), new HashSet(), CellState.QUARANTINED, 3, 300);
            }
            return new Cell(id.intValue(), new HashSet(), INITIAL_CELL_STATE, this.defaultCellMinSize, this.defaultCellMaxSize);
        });
        if (cell.brokers().size() < cell.maxSize()) {
            if (!this.cellIdToCell.containsKey((Object)targetCellId)) {
                this.log.info("Creating cell {}", (Object)targetCellId);
            }
            this.log.info("Assigning broker {} to cell {}", (Object)brokerId, (Object)targetCellId);
            CellRecord record = new CellRecord().setCellId(cell.cellId()).setState(cell.state().code()).setMinSize(cell.minSize()).setMaxSize(cell.maxSize()).setBrokers(new ArrayList<Integer>(CellControlManager.union(Set.of(Integer.valueOf(brokerId)), cell.brokers())));
            recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.CELL_RECORD.highestSupportedVersion()));
            return cell.cellId();
        }
        this.log.warn("Broker {} is not assignable to cell {} since cell is at max size {}", new Object[]{brokerId, cell.cellId(), cell.maxSize()});
        return -1;
    }

    void unregisterBroker(int brokerId, Consumer<ApiMessageAndVersion> recordConsumer) {
        if (!this.isImplicitCellCreationEnabled()) {
            return;
        }
        int cellId = this.getBrokerCellId(brokerId);
        if (cellId == -1) {
            return;
        }
        this.log.info("Removing broker {} from cell {}", (Object)brokerId, (Object)cellId);
        Cell cell = (Cell)this.cellIdToCell.get((Object)cellId);
        HashSet cellBrokers = new HashSet(cell.brokers());
        cellBrokers.remove(brokerId);
        CellRecord record = new CellRecord().setCellId(cell.cellId()).setState(cell.state().code()).setMinSize(cell.minSize()).setMaxSize(cell.maxSize()).setBrokers(new ArrayList<Integer>(cellBrokers));
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.CELL_RECORD.highestSupportedVersion()));
    }

    List<Cell> sortedCells() {
        return this.cellIdToCell.values().stream().sorted(Comparator.comparingInt(Cell::cellId)).collect(Collectors.toList());
    }

    Optional<Cell> getCell(int cellId) {
        return Optional.ofNullable((Cell)this.cellIdToCell.get((Object)cellId));
    }

    List<Cell> computeUsableCells(Set<Integer> usableBrokers, int minimumBrokerCount, int tenantStripeFactor) {
        return this.cellAssignor.computeUsableCells(usableBrokers, this.cellIdToCell.values(), minimumBrokerCount, tenantStripeFactor);
    }

    void fullUpdateCellLoadCache(Set<CellLoad> cellIdToCellLoad, long updateTimeMs) {
        this.cellAssignor.fullUpdateCellLoadCache(cellIdToCellLoad, updateTimeMs);
    }

    boolean isImplicitCellCreationEnabled() {
        return this.isImplicitCellCreationEnabled && this.isCellsEnabled() && !this.isCellMigrationEnabled();
    }

    boolean isCellMigrationSupported() {
        return this.featureControl.metadataVersionOrThrow().isCellMigrationSupported();
    }

    void confirmCellMigrationSupported() {
        this.confirmCellsSupported();
        if (!this.featureControl.metadataVersionOrThrow().isCellMigrationSupported()) {
            throw new UnsupportedVersionException("Attempted to use Cell Migration information at unsupported version.");
        }
    }

    boolean isCellMigrationEnabled() {
        return this.isCellMigrationSupported() && this.cellMigrationState != CellMigrationState.INACTIVE;
    }

    void confirmCellsSupported() {
        if (!this.featureControl.metadataVersionOrThrow().isCellsSupported()) {
            throw new UnsupportedVersionException("Attempted to use Cells information at unsupported version.");
        }
    }

    List<String> tenantIds() {
        return new ArrayList<String>(this.tenantIdToTenant.keySet());
    }

    boolean containsTenant(String tenant) {
        return this.tenantIdToTenant.containsKey((Object)tenant);
    }

    Iterator<Map.Entry<String, Tenant>> tenantIdToTenantIterator(long epoch) {
        return this.tenantIdToTenant.entrySet(epoch).iterator();
    }

    Tenant getTenant(String tenant) {
        return (Tenant)this.tenantIdToTenant.get((Object)tenant);
    }

    private ApiError updateCell(Cell cell, Set<Integer> newBrokerIds, CellState state, short minSize, short maxSize, Set<Integer> usableBrokers, Consumer<ApiMessageAndVersion> recordConsumer) {
        Cell newCell = new Cell(cell.cellId(), newBrokerIds, state, minSize, maxSize);
        try {
            CellAssignor.checkCellMetadata(newCell, cell, usableBrokers, this.replicationFactor, (Integer)this.cellIdToNumTenants.getOrDefault((Object)cell.cellId(), (Object)0));
        }
        catch (InvalidRequestException e) {
            return ApiError.fromThrowable((Throwable)e);
        }
        if (!newCell.equals((Object)cell)) {
            CellRecord record = new CellRecord().setCellId(newCell.cellId()).setState(newCell.state().code()).setMinSize(newCell.minSize()).setMaxSize(newCell.maxSize()).setBrokers(newCell.brokersList());
            recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.CELL_RECORD.highestSupportedVersion()));
        }
        return ApiError.NONE;
    }

    private ApiError updateCellMigrationState(CellMigrationState currentMigrationState, CellMigrationState requestedMigrationState, Consumer<ApiMessageAndVersion> recordConsumer) {
        if (currentMigrationState == CellMigrationState.INACTIVE && requestedMigrationState == CellMigrationState.PAUSED) {
            return new ApiError(Errors.INVALID_REQUEST, String.format("Invalid migration state transition. current state: %s; requested state: %s", currentMigrationState, requestedMigrationState));
        }
        if (requestedMigrationState != currentMigrationState) {
            CellMigrationRecord record = new CellMigrationRecord().setState(requestedMigrationState.code());
            recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.CELL_MIGRATION_RECORD.highestSupportedVersion()));
        }
        return ApiError.NONE;
    }

    private Optional<Cell> getBrokerCell(int brokerId) {
        return Optional.ofNullable((Cell)this.cellIdToCell.get((Object)this.getBrokerCellId(brokerId)));
    }

    private ApiError assignBrokersToCell(Set<Integer> brokerIds, int targetCellId, boolean force, Set<Integer> usableBrokers, Consumer<ApiMessageAndVersion> recordConsumer) {
        try {
            CellAssignor.checkBrokerAssignment(brokerIds, this.cellSize, force);
        }
        catch (InvalidRequestException e) {
            return ApiError.fromThrowable((Throwable)e);
        }
        ArrayList cells = new ArrayList(this.cellIdToCell.values());
        Collections.sort(cells, (c0, c1) -> Boolean.compare(c0.cellId() != targetCellId, c1.cellId() != targetCellId));
        for (Cell cell : cells) {
            ApiError err;
            Set<Integer> newBrokers;
            Set<Integer> set = newBrokers = cell.cellId() == targetCellId ? CellControlManager.union(cell.brokers(), brokerIds) : CellControlManager.difference(cell.brokers(), brokerIds);
            if (cell.brokers().equals(newBrokers) || ApiError.NONE.equals((Object)(err = this.updateCell(cell, newBrokers, cell.state(), cell.minSize(), cell.maxSize(), usableBrokers, recordConsumer)))) continue;
            return err;
        }
        return ApiError.NONE;
    }

    private ApiError unassignBrokersFromCell(Set<Integer> brokerIds, boolean force, Set<Integer> usableBrokers, Consumer<ApiMessageAndVersion> recordConsumer) {
        return this.assignBrokersToCell(brokerIds, -1, force, usableBrokers, recordConsumer);
    }

    private ApiError updateCellMetadata(int cellId, CellState state, short minSize, short maxSize, Set<Integer> usableBrokers, Consumer<ApiMessageAndVersion> recordConsumer) {
        Optional<Cell> cellOpt = this.getCell(cellId);
        if (!cellOpt.isPresent()) {
            return new ApiError(Errors.CELL_NOT_FOUND, String.format("Target cell %s does not exist", cellId));
        }
        Cell cell = cellOpt.get();
        return this.updateCell(cell, cell.brokers(), state, minSize, maxSize, usableBrokers, recordConsumer);
    }

    private ApiError createCell(int cellId, CellState state, short minSize, short maxSize, Consumer<ApiMessageAndVersion> recordConsumer) {
        if (this.cellIdToCell.containsKey((Object)cellId)) {
            String msg = String.format("Cell %s won't be created since it already exists", cellId);
            this.log.error(msg);
            return new ApiError(Errors.INVALID_REQUEST, msg);
        }
        Cell newCell = new Cell(cellId, new HashSet(), state, minSize, maxSize);
        try {
            CellAssignor.checkCellMetadata(newCell, newCell, new HashSet<Integer>(), this.replicationFactor, 0);
        }
        catch (InvalidRequestException e) {
            return ApiError.fromThrowable((Throwable)e);
        }
        this.log.info("Creating cell {}", (Object)cellId);
        CellRecord record = new CellRecord().setCellId(cellId).setState(state.code()).setMinSize(minSize).setMaxSize(maxSize).setBrokers(new ArrayList<Integer>());
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.CELL_RECORD.highestSupportedVersion()));
        return ApiError.NONE;
    }

    private ApiError deleteCell(int cellId, Consumer<ApiMessageAndVersion> recordConsumer) {
        Optional<Cell> cell = this.getCell(cellId);
        if (!cell.isPresent()) {
            return new ApiError(Errors.CELL_NOT_FOUND, String.format("Did not delete cell %s since it does not exist", cellId));
        }
        if (!cell.get().brokers().isEmpty()) {
            return new ApiError(Errors.INVALID_REQUEST, String.format("Did not delete cell %s since brokers %s are still assigned", cellId, cell.get().brokers()));
        }
        RemoveCellRecord record = new RemoveCellRecord().setCellId(cellId);
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.REMOVE_CELL_RECORD.highestSupportedVersion()));
        this.log.info("Deleted cell " + cellId);
        return ApiError.NONE;
    }

    public boolean isCellsEnabled() {
        return this.cellsEnabled && this.featureControl.metadataVersionOrThrow().isCellsSupported();
    }

    private static Set<Integer> union(Set<Integer> a, Set<Integer> b) {
        HashSet<Integer> result = new HashSet<Integer>(a);
        result.addAll(b);
        return result;
    }

    private static Set<Integer> difference(Set<Integer> a, Set<Integer> b) {
        HashSet<Integer> result = new HashSet<Integer>(a);
        result.removeAll(b);
        return result;
    }
}

