/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.storage.checksum;

import com.google.flatbuffers.FlatBufferBuilder;
import io.confluent.kafka.availability.FileChannelWrapper;
import io.confluent.kafka.storage.checksum.Algorithm;
import io.confluent.kafka.storage.checksum.CheckedFileIO;
import io.confluent.kafka.storage.checksum.ChecksumWithInPlaceUpdate;
import io.confluent.kafka.storage.tier.serdes.CheckedFileIOHeader;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Arrays;
import org.apache.kafka.common.utils.Utils;

public class Adler32CheckedFileIO
extends CheckedFileIO {
    public static final short ADLER_SUPER_BLOCK_LENGTH_MAX = 4096;
    public static final short SUPER_BLOCK_PARAMS_LENGTH = 4;

    Adler32CheckedFileIO(Algorithm algo, short superBlockLength, Path path, OpenOption ... options) throws IOException {
        super(algo, superBlockLength, path, options);
        if (this.fileChannel.size() == 0L) {
            log.debug("Writing super block header for empty file path: " + String.valueOf(path));
            if (this.readOnly) {
                throw new IllegalArgumentException("Cannot write super block as file is read-only");
            }
            this.writeSuperBlock();
        } else {
            log.debug("Reading super block header for non-empty file path: " + String.valueOf(path));
            CheckedFileIOHeader header = this.readSuperBlock();
            this.checksumPackage.restore(header.checksum(), header.endPosition());
        }
        this.fileChannel.position(superBlockLength);
        log.info("Open successfully for file path: " + String.valueOf(path) + " " + String.valueOf(this));
    }

    @Override
    public boolean validate() throws IOException, ReflectiveOperationException {
        ChecksumWithInPlaceUpdate checksumPackageValidate = (ChecksumWithInPlaceUpdate)this.checksumPackage.getClass().getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        this.reCalculateChecksum(checksumPackageValidate);
        if (this.checksumPackage.getValue() != checksumPackageValidate.getValue()) {
            log.error("Checksum mismatch: checksum({}) != recalculated checksum({})", (Object)this.checksumPackage.getValue(), (Object)checksumPackageValidate.getValue());
            return false;
        }
        return true;
    }

    @Override
    public void read(ByteBuffer dst, long position) throws IOException {
        if (position < 0L) {
            throw new IllegalArgumentException("Cannot read as position: " + position + " is < 0");
        }
        long currentPosition = this.actualPosition(position);
        Utils.readFully((FileChannel)this.fileChannel, (ByteBuffer)dst, (long)currentPosition);
    }

    @Override
    public void write(ByteBuffer src) throws IOException {
        int sz = src.remaining();
        long pos = this.position();
        this.write(src, pos);
        this.position(pos + (long)sz);
    }

    @Override
    public void write(ByteBuffer src, long position) throws IOException {
        if (this.readOnly) {
            throw new IllegalArgumentException("Cannot write as file is read-only");
        }
        if (position < 0L) {
            throw new IllegalArgumentException("Cannot write as position: " + position + " is < 0");
        }
        this.maybeFillZeros(position);
        int srcPos = src.position();
        int srcLimit = src.limit();
        this.readAndUpdateChecksum(src, position);
        src.position(srcPos);
        src.limit(srcLimit);
        Utils.writeFully((FileChannel)this.fileChannel, (long)this.actualPosition(position), (ByteBuffer)src);
    }

    @Override
    public long position() throws IOException {
        return this.relativePosition(this.fileChannel.position());
    }

    @Override
    public void position(long newPosition) throws IOException {
        if (newPosition < 0L) {
            throw new IllegalArgumentException("Cannot set as position: " + newPosition + " is < 0");
        }
        this.fileChannel.position(this.actualPosition(newPosition));
    }

    @Override
    public long size() throws IOException {
        return this.relativePosition(this.fileChannel.size());
    }

    @Override
    public void truncate(long size) throws IOException {
        if (this.readOnly) {
            throw new IllegalArgumentException("Cannot truncate as file is read-only");
        }
        if (size < 0L) {
            throw new IllegalArgumentException("Cannot truncate as size: " + size + " is < 0");
        }
        FileChannelWrapper.truncate((FileChannel)this.fileChannel, (long)this.actualPosition(size));
        this.reCalculateChecksum(this.checksumPackage);
    }

    @Override
    public void flush() throws IOException {
        if (this.readOnly) {
            throw new IllegalArgumentException("Cannot flush as file is read-only");
        }
        this.updateHeader();
        FileChannelWrapper.force((FileChannel)this.fileChannel, (boolean)true);
    }

    @Override
    public void close() throws IOException {
        if (!this.readOnly) {
            this.flush();
        }
        FileChannelWrapper.close((FileChannel)this.fileChannel);
    }

    private long actualPosition(long position) {
        return position + (long)this.superBlockLength;
    }

    private long relativePosition(long position) {
        return position - (long)this.superBlockLength;
    }

    static ByteBuffer readSuperBlockParams(FileChannel fc) throws IOException {
        long fileSize = fc.size();
        if (4L > fileSize) {
            throw new IOException("Unable to read super block parameters as SUPER_BLOCK_PARAMS_LENGTH: 4 is greater than actual file size: " + fileSize);
        }
        ByteBuffer paramsLength = ByteBuffer.allocate(4).order(BYTE_ORDER);
        Utils.readFully((FileChannel)fc, (ByteBuffer)paramsLength, (long)0L);
        if (paramsLength.hasRemaining()) {
            throw new IOException("Unable to read Header params length buffer as it has more data remaining");
        }
        paramsLength.flip();
        return paramsLength;
    }

    private CheckedFileIOHeader readSuperBlock() throws IOException {
        ByteBuffer superBlockParams = Adler32CheckedFileIO.readSuperBlockParams(this.fileChannel);
        long fileSize = this.fileChannel.size();
        short superBlockLength = superBlockParams.getShort();
        if (this.superBlockLength != superBlockLength) {
            throw new IllegalArgumentException("Incorrect superBlockLength actual: " + superBlockLength + " expected: " + this.superBlockLength);
        }
        short headerSize = superBlockParams.getShort();
        log.debug("Reading super block params superBlockLength: " + superBlockLength + ", headerSize: " + headerSize);
        if (superBlockLength < headerSize) {
            throw new IllegalArgumentException("Incorrect value read as super block length: " + superBlockLength + " <  header size: " + headerSize);
        }
        if (fileSize < (long)(4 + headerSize)) {
            throw new IOException("Unable to read header as headerSize: " + (4 + headerSize) + " is greater than file size");
        }
        ByteBuffer headerBuf = ByteBuffer.allocate(headerSize).order(BYTE_ORDER);
        Utils.readFully((FileChannel)this.fileChannel, (ByteBuffer)headerBuf, (long)4L);
        if (headerBuf.hasRemaining()) {
            throw new IOException("Unable to read Header buffer as it has more data remaining");
        }
        headerBuf.flip();
        CheckedFileIOHeader chkHeader = CheckedFileIOHeader.getRootAsCheckedFileIOHeader(headerBuf);
        log.debug("Reading header checksum: " + chkHeader.checksum() + " length: " + chkHeader.endPosition() + " file sz: " + this.size());
        if (this.size() != chkHeader.endPosition()) {
            throw new IOException("File couldn't be opened because file sz has changed to: " + this.size() + " but checksum was calculated till: " + chkHeader.endPosition());
        }
        return chkHeader;
    }

    private void writeSuperBlock() throws IOException {
        Utils.writeFully((FileChannel)this.fileChannel, (long)0L, (ByteBuffer)this.zeros(this.superBlockLength));
        ByteBuffer headerBuffer = this.serializeHeader();
        ByteBuffer headerLength = ByteBuffer.allocate(4).order(BYTE_ORDER);
        headerLength.putShort(this.superBlockLength);
        headerLength.putShort((short)headerBuffer.remaining());
        headerLength.flip();
        if (headerLength.remaining() + headerBuffer.remaining() > this.superBlockLength) {
            throw new IOException("Header size: " + (headerLength.remaining() + headerBuffer.remaining()) + " exceeds super block length: " + this.superBlockLength);
        }
        Utils.writeFully((FileChannel)this.fileChannel, (long)0L, (ByteBuffer)headerLength);
        Utils.writeFully((FileChannel)this.fileChannel, (long)4L, (ByteBuffer)headerBuffer);
    }

    private ByteBuffer serializeHeader() {
        FlatBufferBuilder flatBuffer = new FlatBufferBuilder().forceDefaults(true);
        int headerId = CheckedFileIOHeader.createCheckedFileIOHeader(flatBuffer, (byte)1, this.checksumPackage.getValue(), this.checksumPackage.length());
        CheckedFileIOHeader.finishCheckedFileIOHeaderBuffer(flatBuffer, headerId);
        return flatBuffer.dataBuffer();
    }

    private void updateHeader() throws IOException {
        ByteBuffer headerBuffer = this.serializeHeader();
        if (headerBuffer.remaining() > this.superBlockLength) {
            throw new IOException("Header size: " + headerBuffer.remaining() + " exceeds super block length: " + this.superBlockLength);
        }
        Utils.writeFully((FileChannel)this.fileChannel, (long)4L, (ByteBuffer)headerBuffer);
    }

    void reCalculateChecksum(ChecksumWithInPlaceUpdate csUpdate) throws IOException {
        csUpdate.reset();
        long fileSize = this.size();
        ByteBuffer bb = ByteBuffer.allocate(262144).order(BYTE_ORDER);
        for (long pos = 0L; pos < fileSize; pos += (long)bb.remaining()) {
            bb.clear();
            this.read(bb, pos);
            bb.flip();
            csUpdate.append(bb);
        }
    }

    private void maybeFillZeros(long position) throws IOException {
        long writeAheadGap = position - this.size();
        if (writeAheadGap > 0L) {
            ByteBuffer zerosBb = this.zeros((int)writeAheadGap);
            this.checksumPackage.append(zerosBb);
            zerosBb.flip();
            Utils.writeFully((FileChannel)this.fileChannel, (long)this.actualPosition(this.size()), (ByteBuffer)zerosBb);
        }
    }

    void readAndUpdateChecksum(ByteBuffer src, long pos) throws IOException {
        long fileSz = this.size();
        int dataSz = src.remaining();
        long endPos = pos + (long)dataSz;
        if (pos == fileSz) {
            this.checksumPackage.append(src);
        } else if (endPos > fileSz) {
            int updSz = (int)(fileSz - pos);
            ByteBuffer previous = ByteBuffer.allocate(updSz).order(BYTE_ORDER);
            this.read(previous, pos);
            previous.flip();
            int srcPos = src.position();
            src.position(srcPos + updSz);
            this.checksumPackage.append(src);
            src.position(srcPos);
            src.limit(srcPos + updSz);
            this.checksumPackage.update(src, previous, pos);
        } else {
            int updSz = (int)(endPos - pos);
            ByteBuffer previous = ByteBuffer.allocate(updSz).order(BYTE_ORDER);
            this.read(previous, pos);
            previous.flip();
            this.checksumPackage.update(src, previous, pos);
        }
    }

    private ByteBuffer zeros(int n) {
        byte[] ar = new byte[n];
        Arrays.fill(ar, (byte)0);
        return ByteBuffer.wrap(ar).order(BYTE_ORDER);
    }

    public String toString() {
        return "CheckedFileIO(version=1, algorithm=" + String.valueOf((Object)this.algorithm) + ", superBlockLength=" + this.superBlockLength + ", checksumPackage=" + String.valueOf(this.checksumPackage) + ")";
    }
}

