/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import io.confluent.kafka.availability.FilesWrapper;
import io.confluent.kafka.storage.checksum.ChecksumInfo;
import io.confluent.kafka.storage.checksum.ChecksumParams;
import io.confluent.kafka.storage.checksum.ChecksumStore;
import io.confluent.kafka.storage.checksum.E2EChecksumProtectedObjectType;
import io.confluent.kafka.storage.checksum.E2EChecksumStore;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.function.Supplier;
import java.util.zip.Checksum;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.utils.Checksums;
import org.apache.kafka.common.utils.PrimitiveRef;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.storage.internals.log.AbortedTxn;
import org.apache.kafka.storage.internals.log.CorruptIndexException;
import org.apache.kafka.storage.internals.log.TxnIndexSearchResult;

public class TransactionIndex
implements Closeable {
    private final long startOffset;
    private final boolean fileAlreadyExists;
    private volatile File file;
    private Optional<FileChannel> maybeChannel = Optional.empty();
    private OptionalLong lastOffset = OptionalLong.empty();
    private final Optional<E2EChecksumStore> checksumStoreOpt;
    private final boolean e2eChecksumEnabledForTopic;
    private final boolean shouldPersistChecksum;
    private volatile boolean shouldInitializeChecksum;

    public TransactionIndex(long startOffset, File file, boolean fileAlreadyExists, ChecksumParams checksumParams) throws IOException {
        this.startOffset = startOffset;
        this.file = file;
        this.fileAlreadyExists = fileAlreadyExists;
        this.checksumStoreOpt = checksumParams.checksumStoreOpt();
        this.e2eChecksumEnabledForTopic = checksumParams.e2eChecksumEnabledForTopic();
        this.shouldPersistChecksum = checksumParams.shouldPersistChecksum();
        boolean bl = this.shouldInitializeChecksum = file.length() == 0L;
        if (file.exists()) {
            this.openChannel();
        }
    }

    public TransactionIndex(long startOffset, File file, boolean fileAlreadyExists) throws IOException {
        this(startOffset, file, fileAlreadyExists, ChecksumParams.EMPTY);
    }

    public TransactionIndex(long startOffset, File file) throws IOException {
        this(startOffset, file, false, ChecksumParams.EMPTY);
    }

    public File file() {
        return this.file;
    }

    public void updateParentDir(File parentDir) {
        this.file = new File(parentDir, this.file.getName());
    }

    public void append(AbortedTxn abortedTxn) throws IOException {
        this.append(abortedTxn, System.currentTimeMillis());
    }

    public void append(AbortedTxn abortedTxn, long appendTimeMs) throws IOException {
        this.lastOffset.ifPresent(offset -> {
            if (offset >= abortedTxn.lastOffset()) {
                throw new IllegalArgumentException("The last offset of appended transactions must increase sequentially, but " + abortedTxn.lastOffset() + " is not greater than current last offset " + offset + " of index " + this.file.getAbsolutePath());
            }
        });
        this.lastOffset = OptionalLong.of(abortedTxn.lastOffset());
        Utils.writeFully((FileChannel)this.channel(), (ByteBuffer)abortedTxn.buffer.duplicate());
        this.checksumStoreOpt.ifPresent(checksumStore -> this.mayUpdateChecksum((E2EChecksumStore)checksumStore, abortedTxn.buffer.duplicate(), appendTimeMs));
        this.shouldInitializeChecksum = false;
    }

    private void mayUpdateChecksum(E2EChecksumStore checksumStore, ByteBuffer buffer, long appendTimeMs) {
        if (this.e2eChecksumEnabledForTopic && checksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.TRANSACTION_INDEX)) {
            if (this.shouldInitializeChecksum) {
                checksumStore.store().initializeEntry(this.file.getAbsolutePath(), this.shouldPersistChecksum);
            }
            checksumStore.store().update(this.file.getAbsolutePath(), buffer, appendTimeMs);
        }
    }

    private void mayRemoveChecksum(E2EChecksumStore checksumStore) {
        if (this.e2eChecksumEnabledForTopic && checksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.TRANSACTION_INDEX)) {
            checksumStore.store().remove(this.file.getAbsolutePath());
            this.shouldInitializeChecksum = true;
        }
    }

    public void flush() throws IOException {
        FileChannel channel = this.channelOrNull();
        if (channel != null) {
            channel.force(true);
        }
    }

    public void reset() throws IOException {
        FileChannel channel = this.channelOrNull();
        if (channel != null) {
            channel.truncate(0L);
            this.checksumStoreOpt.ifPresent(this::mayRemoveChecksum);
        }
        this.lastOffset = OptionalLong.empty();
    }

    @Override
    public void close() throws IOException {
        FileChannel channel = this.channelOrNull();
        if (channel != null && channel.isOpen()) {
            channel.close();
        }
        this.maybeChannel = Optional.empty();
    }

    public boolean deleteIfExists() throws IOException {
        this.close();
        return FilesWrapper.deleteIfExists((Path)this.file.toPath());
    }

    public void renameTo(File f) throws IOException {
        try {
            if (this.file.exists()) {
                Utils.atomicMoveWithFallback((Path)this.file.toPath(), (Path)f.toPath(), (boolean)false);
            }
        }
        finally {
            this.file = f;
        }
    }

    public void truncateTo(long offset) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(34);
        OptionalLong newLastOffset = OptionalLong.empty();
        long currentTimeMs = System.currentTimeMillis();
        this.checksumStoreOpt.ifPresent(this::mayRemoveChecksum);
        for (AbortedTxnWithPosition txnWithPosition : this.iterable(() -> buffer)) {
            AbortedTxn abortedTxn = txnWithPosition.txn;
            long position = txnWithPosition.position;
            if (abortedTxn.lastOffset() >= offset) {
                this.channel().truncate(position);
                this.lastOffset = newLastOffset;
                return;
            }
            this.checksumStoreOpt.ifPresent(checksumStore -> this.mayUpdateChecksum((E2EChecksumStore)checksumStore, abortedTxn.buffer.duplicate(), currentTimeMs));
            this.shouldInitializeChecksum = false;
            newLastOffset = OptionalLong.of(abortedTxn.lastOffset());
        }
    }

    public List<AbortedTxn> allAbortedTxns() {
        ArrayList<AbortedTxn> result = new ArrayList<AbortedTxn>();
        for (AbortedTxnWithPosition txnWithPosition : this.iterable()) {
            result.add(txnWithPosition.txn);
        }
        return result;
    }

    public TxnIndexSearchResult collectAbortedTxns(long fetchOffset, long upperBoundOffset) {
        return this.collectAbortedTxns(fetchOffset, upperBoundOffset, false);
    }

    public TxnIndexSearchResult collectAbortedTxns(long fetchOffset, long upperBoundOffset, boolean shouldValidateChecksum) {
        ArrayList<AbortedTxn> abortedTransactions = new ArrayList<AbortedTxn>();
        boolean isComplete = false;
        Checksum actualChecksum = null;
        Optional<Object> expectedChecksum = Optional.empty();
        if (shouldValidateChecksum && this.checksumStoreOpt.isPresent()) {
            E2EChecksumStore e2eChecksumStore = this.checksumStoreOpt.get();
            ChecksumStore store = e2eChecksumStore.store();
            if (e2eChecksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.TRANSACTION_INDEX)) {
                expectedChecksum = store.get(this.file.getAbsolutePath());
                actualChecksum = store.emptyChecksum();
            }
        }
        shouldValidateChecksum = shouldValidateChecksum && expectedChecksum.isPresent();
        for (AbortedTxnWithPosition txnWithPosition : this.iterable()) {
            AbortedTxn abortedTxn = txnWithPosition.txn;
            if (isComplete && !shouldValidateChecksum) break;
            if (abortedTxn.lastOffset() >= fetchOffset && abortedTxn.firstOffset() < upperBoundOffset) {
                abortedTransactions.add(abortedTxn);
            }
            if (shouldValidateChecksum) {
                Checksums.update((Checksum)actualChecksum, (ByteBuffer)abortedTxn.buffer.duplicate(), (int)34);
            }
            isComplete = abortedTxn.lastStableOffset() >= upperBoundOffset;
        }
        if (shouldValidateChecksum) {
            this.validateChecksum((ChecksumInfo)expectedChecksum.get(), actualChecksum);
        }
        return new TxnIndexSearchResult(abortedTransactions, isComplete);
    }

    private void validateChecksum(ChecksumInfo expectedChecksum, Checksum actualChecksum) {
        if (expectedChecksum.checksum().getValue() != actualChecksum.getValue()) {
            throw new CorruptIndexException("Transaction Index " + this.file.getAbsolutePath() + " is corrupted. Corruption is detected while collecting aborted transactions.");
        }
    }

    public void sanityCheck() {
        ByteBuffer buffer = ByteBuffer.allocate(34);
        for (AbortedTxnWithPosition txnWithPosition : this.iterable(() -> buffer)) {
            AbortedTxn abortedTxn = txnWithPosition.txn;
            if (abortedTxn.lastOffset() >= this.startOffset) continue;
            throw new CorruptIndexException("Last offset of aborted transaction " + String.valueOf(abortedTxn) + " in index " + this.file.getAbsolutePath() + " is less than start offset " + this.startOffset);
        }
    }

    public boolean isEmpty() {
        return !this.iterable().iterator().hasNext();
    }

    private FileChannel openChannel() throws IOException {
        FileChannel channel = FileChannel.open(this.file.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        if (!this.fileAlreadyExists && channel.size() > 0L) {
            channel.close();
            throw new IllegalStateException("Non-empty transaction index file " + String.valueOf(this.file) + " already exists while it shouldn't.");
        }
        this.maybeChannel = Optional.of(channel);
        channel.position(channel.size());
        return channel;
    }

    private FileChannel channel() throws IOException {
        FileChannel channel = this.channelOrNull();
        if (channel == null) {
            return this.openChannel();
        }
        return channel;
    }

    private FileChannel channelOrNull() {
        return this.maybeChannel.orElse(null);
    }

    private Iterable<AbortedTxnWithPosition> iterable() {
        return this.iterable(() -> ByteBuffer.allocate(34));
    }

    private Iterable<AbortedTxnWithPosition> iterable(final Supplier<ByteBuffer> allocate) {
        final FileChannel channel = this.channelOrNull();
        if (channel == null) {
            return Collections.emptyList();
        }
        final PrimitiveRef.IntRef position = PrimitiveRef.ofInt((int)0);
        return () -> new Iterator<AbortedTxnWithPosition>(){

            @Override
            public boolean hasNext() {
                try {
                    return channel.position() - (long)position.value >= 34L;
                }
                catch (IOException e) {
                    throw new KafkaException("Failed read position from the transaction index " + TransactionIndex.this.file.getAbsolutePath(), (Throwable)e);
                }
            }

            @Override
            public AbortedTxnWithPosition next() {
                try {
                    ByteBuffer buffer = (ByteBuffer)allocate.get();
                    Utils.readFully((FileChannel)channel, (ByteBuffer)buffer, (long)position.value);
                    buffer.flip();
                    AbortedTxn abortedTxn = new AbortedTxn(buffer);
                    if (abortedTxn.version() > 0) {
                        throw new KafkaException("Unexpected aborted transaction version " + abortedTxn.version() + " in transaction index " + TransactionIndex.this.file.getAbsolutePath() + ", current version is 0");
                    }
                    AbortedTxnWithPosition nextEntry = new AbortedTxnWithPosition(abortedTxn, position.value);
                    position.value += 34;
                    return nextEntry;
                }
                catch (IOException e) {
                    throw new KafkaException("Failed to read from the transaction index " + TransactionIndex.this.file.getAbsolutePath(), (Throwable)e);
                }
            }
        };
    }

    private static class AbortedTxnWithPosition {
        final AbortedTxn txn;
        final int position;

        AbortedTxnWithPosition(AbortedTxn txn, int position) {
            this.txn = txn;
            this.position = position;
        }
    }
}

