/*
 * ==========================================================================================
 * =                   JAHIA'S DUAL LICENSING - IMPORTANT INFORMATION                       =
 * ==========================================================================================
 *
 *                                 http://www.jahia.com
 *
 *     Copyright (C) 2002-2025 Jahia Solutions Group SA. All rights reserved.
 *
 *     THIS FILE IS AVAILABLE UNDER TWO DIFFERENT LICENSES:
 *     1/Apache2 OR 2/JSEL
 *
 *     1/ Apache2
 *     ==================================================================================
 *
 *     Copyright (C) 2002-2025 Jahia Solutions Group SA. All rights reserved.
 *
 *     Licensed under the Apache License, Version 2.0 (the "License");
 *     you may not use this file except in compliance with the License.
 *     You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 *
 *
 *     2/ JSEL - Commercial and Supported Versions of the program
 *     ===================================================================================
 *
 *     IF YOU DECIDE TO CHOOSE THE JSEL LICENSE, YOU MUST COMPLY WITH THE FOLLOWING TERMS:
 *
 *     Alternatively, commercial and supported versions of the program - also known as
 *     Enterprise Distributions - must be used in accordance with the terms and conditions
 *     contained in a separate written agreement between you and Jahia Solutions Group SA.
 *
 *     If you are unsure which license is appropriate for your use,
 *     please contact the sales department at sales@jahia.com.
 */
package org.jahia.ajax.gwt.helper;

import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.atmosphere.cpr.Broadcaster;
import org.atmosphere.cpr.BroadcasterFactory;
import org.jahia.ajax.gwt.client.data.definition.GWTJahiaNodeProperty;
import org.jahia.ajax.gwt.client.data.job.GWTJahiaJobDetail;
import org.jahia.ajax.gwt.client.service.GWTJahiaServiceException;
import org.jahia.ajax.gwt.client.widget.poller.ProcessPollingEvent;
import org.jahia.ajax.gwt.commons.server.ChannelHolder;
import org.jahia.ajax.gwt.commons.server.JGroupsChannel;
import org.jahia.ajax.gwt.commons.server.ManagedGWTResource;
import org.jahia.osgi.BundleUtils;
import org.jahia.osgi.FrameworkService;
import org.jahia.services.SpringContextSingleton;
import org.jahia.services.atmosphere.AtmosphereServlet;
import org.jahia.services.content.JCRCallback;
import org.jahia.services.content.JCRSessionWrapper;
import org.jahia.services.content.JCRTemplate;
import org.jahia.services.content.PublicationJob;
import org.jahia.services.content.rules.ActionJob;
import org.jahia.services.content.rules.RuleJob;
import org.jahia.services.content.textextraction.TextExtractorJob;
import org.jahia.services.events.JournalEventReader;
import org.jahia.services.importexport.ImportJob;
import org.jahia.services.scheduler.BackgroundJob;
import org.jahia.services.scheduler.SchedulerService;
import org.quartz.*;
import org.quartz.listeners.JobListenerSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.ItemNotFoundException;
import javax.jcr.RepositoryException;
import java.util.*;

import static org.jahia.api.Constants.CLUSTER_BROADCAST_TOPIC_PREFIX;

/**
 * User: toto
 * Date: Sep 17, 2010
 * Time: 2:04:17 PM
 */
public class SchedulerHelper {
    private Logger logger = LoggerFactory.getLogger(SchedulerHelper.class);

    private SchedulerService scheduler;

    public void setScheduler(SchedulerService scheduler) {
        this.scheduler = scheduler;
    }

    public void start() {
        try {
            scheduler.getScheduler().addGlobalJobListener(new PollingSchedulerListener());
        } catch (SchedulerException e) {
            logger.error("Cannot register job listener", e);
        }
    }

    private Long getLong(JobDataMap jobDataMap, String key) {
        if (jobDataMap.get(key) == null) {
            return null;
        }
        return Long.parseLong(jobDataMap.getString(key));
    }

    private List<GWTJahiaJobDetail> convertToGWTJobs(List<JobDetail> jobDetails) {
        List<GWTJahiaJobDetail> jobs = new ArrayList<GWTJahiaJobDetail>();
        for (JobDetail jobDetail : jobDetails) {
            JobDataMap jobDataMap = jobDetail.getJobDataMap();
            Date created = (Date) jobDataMap.get(BackgroundJob.JOB_CREATED);
            final String status = jobDataMap.getString(BackgroundJob.JOB_STATUS);
            final String user = StringUtils.substringAfterLast(jobDataMap.getString(BackgroundJob.JOB_USERKEY), "/");
            final String message = jobDataMap.getString(BackgroundJob.JOB_MESSAGE);
            final Long beginTime = getLong(jobDataMap, BackgroundJob.JOB_BEGIN);
            final Long endTime = getLong(jobDataMap, BackgroundJob.JOB_END);
            final String site = jobDataMap.getString(BackgroundJob.JOB_SITEKEY);
            if (created == null && beginTime != null) {
                // this can happen for cron scheduler jobs.
                created = new Date(beginTime);
            }
            Long duration = getLong(jobDataMap, BackgroundJob.JOB_DURATION);
            if ((duration == null) && (beginTime != null) && (endTime == null) && BackgroundJob.STATUS_EXECUTING.equals(status)) {
                // here we have a currently running job, let's calculate the duration until now.
                duration = System.currentTimeMillis() - beginTime.longValue();
            }
            final String jobLocale = jobDataMap.getString(BackgroundJob.JOB_CURRENT_LOCALE);
            String targetNodeIdentifier = null;
            String targetAction = null;
            String targetWorkspace = null;

//            if ((jahiaUser != null) && (!jahiaUser.getUserKey().equals(user))) {
            // we must check whether the user has the permission to view other users's jobs
//                if (!jahiaUser.isPermitted(new PermissionIdentity("view-all-jobs"))) {
//                    // he doesn't we skip this entry.
//                    continue;
//                }
//            }

            String description = jobDetail.getDescription();
            final List<String> targetPaths = new ArrayList<String>();
            String fileName = jobDataMap.getString(ImportJob.FILENAME);
            if (BackgroundJob.getGroupName(PublicationJob.class).equals(jobDetail.getGroup())) {
                @SuppressWarnings("unchecked")
                List<GWTJahiaNodeProperty> publicationInfos = (List<GWTJahiaNodeProperty>) jobDataMap.get(PublicationJob.PUBLICATION_PROPERTIES);
                String publicationTitle = (String) jobDataMap.get(PublicationJob.PUBLICATION_TITLE);
                if (publicationInfos != null && publicationInfos.size() > 0) {
                    description += " " + publicationInfos.get(0).getValues();
                } else if (StringUtils.isNotEmpty(publicationTitle)) {
                    description += " [" + publicationTitle + "]";
                }

                getPublicationPaths(jobDataMap, targetPaths);
            } else if (BackgroundJob.getGroupName(ImportJob.class).equals(jobDetail.getGroup())) {
                String uri = (String) jobDataMap.get(ImportJob.URI);
                if (uri != null) {
                    targetPaths.add(uri);
                    description += " " + uri;
                } else {
                    String destinationParentPath = jobDataMap.getString(ImportJob.DESTINATION_PARENT_PATH);
                    targetPaths.add(destinationParentPath);
                }
            } else if (BackgroundJob.getGroupName(ActionJob.class).equals(jobDetail.getGroup())) {
                String actionToExecute = jobDataMap.getString(ActionJob.JOB_ACTION_TO_EXECUTE);
                targetAction = actionToExecute;
                String nodeUUID = jobDataMap.getString(ActionJob.JOB_NODE_UUID);
                targetNodeIdentifier = nodeUUID;
            } else if (BackgroundJob.getGroupName(RuleJob.class).equals(jobDetail.getGroup())) {
                String ruleToExecute = jobDataMap.getString(RuleJob.JOB_RULE_TO_EXECUTE);
                targetAction = ruleToExecute;
                String nodeUUID = jobDataMap.getString(RuleJob.JOB_NODE_UUID);
                targetNodeIdentifier = nodeUUID;
                String workspace = jobDataMap.getString(RuleJob.JOB_WORKSPACE);
                targetWorkspace = workspace;
            } else if (BackgroundJob.getGroupName(TextExtractorJob.class).equals(jobDetail.getGroup())) {
                String path = jobDataMap.getString(TextExtractorJob.JOB_PATH);
                String extractNodePath = jobDataMap.getString(TextExtractorJob.JOB_EXTRACTNODE_PATH);
                targetPaths.add(path);
                targetPaths.add(extractNodePath);
            }
            GWTJahiaJobDetail job = new GWTJahiaJobDetail(jobDetail.getName(), created, user, site, description,
                    status, message, targetPaths,
                    jobDetail.getGroup(), jobDetail.getJobClass().getName(), beginTime, endTime, duration, jobLocale, fileName, targetNodeIdentifier, targetAction, targetWorkspace);
            job.setLabelKey("label." + jobDetail.getGroup() + ".task");
            jobs.add(job);
        }
        return jobs;
    }

    private void getPublicationPaths(JobDataMap jobDataMap, List<String> targetPaths) {
        List<String> publicationPathsFromJob = (List<String>) jobDataMap.get(PublicationJob.PUBLICATION_PATHS);
        // get target paths from job if specified, if not, use uuids to get the nodes
        if (publicationPathsFromJob != null && publicationPathsFromJob.size() > 0) {
            targetPaths.addAll(publicationPathsFromJob);
        } else {
            final List<String> uuids = (List<String>) jobDataMap.get("publicationInfos");
            try {
                JCRTemplate.getInstance().doExecuteWithSystemSession(new JCRCallback<Object>() {
                    @Override
                    public Object doInJCR(JCRSessionWrapper session) throws RepositoryException {
                        for (String uuid : uuids) {
                            try {
                                targetPaths.add(session.getNodeByIdentifier(uuid).getPath());
                            } catch (ItemNotFoundException e) {
                                logger.debug("Cannot get item " + uuid, e);
                            }
                        }
                        return null;
                    }
                });
            } catch (RepositoryException e) {
                logger.error("Cannot get publication details", e);
            }
        }
    }

    public List<GWTJahiaJobDetail> getActiveJobs(Locale locale) throws GWTJahiaServiceException {
        try {
            List<JobDetail> l = scheduler.getAllActiveJobs();
            return convertToGWTJobs(l);
        } catch (Exception e) {
            throw new GWTJahiaServiceException("Error retrieving active jobs", e);
        }
    }

    public List<GWTJahiaJobDetail> getAllJobs(Set<String> groupNames, String sortField, String sortDir, String groupBy) throws GWTJahiaServiceException {
        try {
            List<JobDetail> jobDetails;
            if (groupNames == null) {
                jobDetails = scheduler.getAllJobs();
            } else {
                jobDetails = new ArrayList<>();
                for (String groupName : groupNames) {
                    jobDetails.addAll(scheduler.getAllJobs(groupName));
                }
            }
            List<GWTJahiaJobDetail> gwtJobList = convertToGWTJobs(jobDetails);
            final String fSortField = sortField == null ? "creationTime" : sortField;
            final int i = "ASC".equals(sortDir) ? 1 : -1;
            gwtJobList.sort((o1, o2) -> ObjectUtils.compare(o1.get(fSortField), o2.get(fSortField)) * i);
            if (groupBy != null) {
                gwtJobList.sort((o1, o2) -> ObjectUtils.compare(o1.get(groupBy), o2.get(groupBy)));
            }

            return gwtJobList;
        } catch (Exception e) {
            throw new GWTJahiaServiceException("Cannot retrieve jobs. Cause: " + e.getLocalizedMessage(), e);
        }

    }

    public Boolean deleteJob(String jobName, String groupName) throws GWTJahiaServiceException {
        try {
            return scheduler.getScheduler().deleteJob(jobName, groupName);
        } catch (SchedulerException e) {
            throw new GWTJahiaServiceException("Cannot delete job " + jobName + ". Cause: " + e.getLocalizedMessage(), e);
        }
    }

    public List<String> getAllJobGroupNames() throws GWTJahiaServiceException {
        try {
            return Arrays.asList(scheduler.getScheduler().getJobGroupNames());
        } catch (SchedulerException e) {
            throw new GWTJahiaServiceException("Cannot get all job group names. Cause: " + e.getLocalizedMessage(), e);
        }
    }

    public Integer deleteAllCompletedJobs() throws GWTJahiaServiceException {
        try {
            return scheduler.deleteAllCompletedJobs();
        } catch (SchedulerException e) {
            throw new GWTJahiaServiceException("Cannot delete completed jobs. Cause: " + e.getLocalizedMessage(), e);
        }
    }


    class PollingSchedulerListener extends JobListenerSupport {
        int totalCount = -1;

        public String getName() {
            return "PollingSchedulerListener";
        }

        @Override
        public void jobToBeExecuted(JobExecutionContext context) {
            JobDetail contextJobDetail = context.getJobDetail();
            updateJobs(Collections.singletonList(contextJobDetail), Collections.<JobDetail>emptyList());
            String topic = CLUSTER_BROADCAST_TOPIC_PREFIX + "/publication/start";
            sendEventAcrossCluster(contextJobDetail, topic);
        }

        @Override
        public void jobWasExecuted(JobExecutionContext context, JobExecutionException jobException) {
            JobDetail contextJobDetail = context.getJobDetail();
            updateJobs(Collections.<JobDetail>emptyList(), Collections.singletonList(contextJobDetail));
            String topic = CLUSTER_BROADCAST_TOPIC_PREFIX + "/publication/done";
            sendEventAcrossCluster(contextJobDetail, topic);
        }

        private void sendEventAcrossCluster(JobDetail contextJobDetail, String topic) {
            if (BackgroundJob.getGroupName(PublicationJob.class).equals(contextJobDetail.getGroup())) {
                List<String> publicationPaths = new ArrayList<>();
                JobDataMap jobDataMap = contextJobDetail.getJobDataMap();
                getPublicationPaths(jobDataMap, publicationPaths);
                Map<String, List<String>> publicationInfos = new HashMap<>();
                publicationInfos.put("paths", publicationPaths);
                final String jobLocale = jobDataMap.getString(BackgroundJob.JOB_CURRENT_LOCALE);
                if (jobLocale != null) {
                    publicationInfos.put("language", Collections.singletonList(jobLocale));
                }
                final String site = jobDataMap.getString(BackgroundJob.JOB_SITEKEY);
                if (site != null) {
                    publicationInfos.put("siteKey", Collections.singletonList(site));
                }
                final String user = StringUtils.substringAfterLast(jobDataMap.getString(BackgroundJob.JOB_USERKEY), "/");
                if (user != null) {
                    publicationInfos.put("user", Collections.singletonList(user));
                }
                JournalEventReader reader = BundleUtils.getOsgiService(org.jahia.services.events.JournalEventReader.class, null);
                if (reader != null) {
                    publicationInfos.put("revision", Collections.singletonList(String.valueOf(reader.getGlobalRevision())));
                }

                logger.debug("Sending {} event for publication paths: {}", topic, publicationPaths);
                FrameworkService.sendEvent(topic, publicationInfos, true);
            }
        }

        private void updateJobs(List<JobDetail> startedJob, List<JobDetail> endedJob) {
            try {
                if (totalCount == -1) {
                    totalCount = getActiveJobs(Locale.ENGLISH).size();
                } else {
                    totalCount += startedJob.size();
                }
                totalCount -= endedJob.size();
                if (totalCount < 0) {
                    //In case job is ended only and there is no active nor started job
                    totalCount = 0;
                }
                final BroadcasterFactory broadcasterFactory = AtmosphereServlet.getBroadcasterFactory();
                if (broadcasterFactory != null) {
                    ProcessPollingEvent pollingEvent = new ProcessPollingEvent();
                    if (startedJob != null) {
                        pollingEvent.setStartedJob(convertToGWTJobs(startedJob));
                    }
                    if (endedJob != null) {
                        pollingEvent.setEndedJob(convertToGWTJobs(endedJob));
                    }
                    pollingEvent.setTotalCount(totalCount);
                    Broadcaster broadcaster = broadcasterFactory.lookup(ManagedGWTResource.GWT_BROADCASTER_ID);
                    if (broadcaster != null) {
                        broadcaster.broadcast(pollingEvent);
                    } else {
                        try {
                            ChannelHolder bean = (ChannelHolder) SpringContextSingleton.getBean("org.jahia.ajax.gwt.commons.server.ChannelHolderImpl");
                            JGroupsChannel jc = bean.getChannel();
                            if (jc != null) {
                                jc.send(ManagedGWTResource.GWT_BROADCASTER_ID, pollingEvent);
                            }
                        } catch (Exception e) {
                            logger.debug(e.getMessage(), e);
                        }
                    }
                }
            } catch (GWTJahiaServiceException e) {
                logger.error("Cannot parse jobs", e);
            }

        }
    }

}
