/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.portal.identity.db;

import org.jboss.portal.identity.User;
import org.jboss.portal.identity.IdentityException;
import org.jboss.portal.identity.NoSuchUserException;
import org.jboss.portal.common.util.Tools;
import org.jboss.portal.identity.service.UserModuleService;
import org.hibernate.SessionFactory;
import org.hibernate.Session;
import org.hibernate.Query;
import org.hibernate.HibernateException;

import javax.naming.InitialContext;
import java.io.Serializable;
import java.util.Set;
import java.util.Iterator;


/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet </a>
 * @version $Revision: 5448 $
 * @portal.core
 */
public class HibernateUserModuleImpl extends UserModuleService
{

   /** . */
   private static final org.jboss.logging.Logger log = org.jboss.logging.Logger.getLogger(HibernateUserModuleImpl.class);

   /** . */
   protected SessionFactory sessionFactory;

   /** . */
   protected String sessionFactoryJNDIName;

   public void start() throws Exception
   {
      //
      sessionFactory = (SessionFactory)new InitialContext().lookup(sessionFactoryJNDIName);

      super.start();
   }

   public void stop() throws Exception
   {

      //
      sessionFactory = null;

      super.stop();
   }

   public String getSessionFactoryJNDIName()
   {
      return sessionFactoryJNDIName;
   }

   public void setSessionFactoryJNDIName(String sessionFactoryJNDIName)
   {
      this.sessionFactoryJNDIName = sessionFactoryJNDIName;
   }

   public User findUserByUserName(String userName) throws IdentityException
   {
      if (userName != null)
      {
         try
         {
            Session session = getCurrentSession();
            Query query = session.createQuery("from HibernateUserImpl where userName=:userName");
            query.setParameter("userName", userName);
            query.setCacheable(true);
            HibernateUserImpl user = (HibernateUserImpl)query.uniqueResult();
            if (user == null)
            {
               throw new NoSuchUserException("No such user " + userName);
            }
            return user;
         }
         catch (HibernateException e)
         {
            String message = "Cannot find user by name " + userName;
            log.error(message, e);
            throw new IdentityException(message, e);
         }
      }
      else
      {
         throw new IllegalArgumentException("user name cannot be null");
      }
   }

   public User findUserById(String id) throws IllegalArgumentException, IdentityException, NoSuchUserException
   {
      if (id == null)
      {
         throw new IllegalArgumentException("The id is null");
      }
      try
      {
         return findUserById(new Long(id));
      }
      catch (NumberFormatException e)
      {
         throw new IllegalArgumentException("Cannot parse id into an long " + id);
      }
   }

   public User findUserById(Object id) throws IllegalArgumentException, IdentityException, NoSuchUserException
   {
      if (id instanceof Long)
      {
         try
         {
            Session session = getCurrentSession();
            HibernateUserImpl user = (HibernateUserImpl)session.get(HibernateUserImpl.class, (Long)id);
            if (user == null)
            {
               throw new NoSuchUserException("No user found for " + id);
            }
            return user;
         }
         catch (HibernateException e)
         {
            String message = "Cannot find user by id " + id;
            log.error(message, e);
            throw new IdentityException(message, e);
         }
      }
      else
      {
         throw new IllegalArgumentException("The id is not an long : " + id);
      }
   }

   public User createUser(String userName, String password) throws IdentityException
   {
      if (userName != null)
      {
         try
         {
            HibernateUserImpl user = new HibernateUserImpl(userName);
            user.updatePassword(password);
            //user.setRealEmail(realEmail);
            Session session = getCurrentSession();
            session.save(user);

            //fire events
            fireUserCreatedEvent(user.getId(), user.getUserName());

            return user;
         }
         catch (HibernateException e)
         {
            String message = "Cannot create user " + userName;
            log.error(message, e);
            throw new IdentityException(message, e);
         }
      }
      else
      {
         throw new IllegalArgumentException("name cannot be null");
      }
   }

   public void removeUser(Object id) throws IdentityException
   {
      if (id == null)
      {
         throw new IllegalArgumentException("User id cannot be null");
      }

      if (id instanceof Long)
      {
         try
         {
            Session session = getCurrentSession();
            HibernateUserImpl user = (HibernateUserImpl)session.load(HibernateUserImpl.class, (Serializable)id);


            if (user == null)
            {
               throw new NoSuchUserException("No such user " + id);
            }

            String userName = user.getUserName();

            session.delete(user);
            session.flush();

            //fire events
            fireUserDestroyedEvent(id, userName);
         }
         catch (HibernateException e)
         {
            String message = "Cannot remove user " + id;
            log.error(message, e);
            throw new IdentityException(message, e);
         }
      }
      else
      {
         throw new IllegalArgumentException("The id is not an long : " + id);
      }
   }

   public Set findUsers(int offset, int limit) throws IdentityException
   {
      try
      {
         Session session = getCurrentSession();
         Query query = session.createQuery("from HibernateUserImpl as u order by u.userName");
         query.setFirstResult(offset);
         query.setMaxResults(limit);
         Iterator iterator = query.iterate();
         return Tools.toSet(iterator, true);
      }
      catch (HibernateException e)
      {
         String message = "Cannot find user range [" + offset + "," + limit + "]";
         log.error(message, e);
         throw new IdentityException(message, e);
      }
   }

   public Set findUsersFilteredByUserName(String filter, int offset, int limit) throws IdentityException
   {
      try
      {
         // Remove all occurences of % and add ours
         filter = "%" + filter.replaceAll("%", "") + "%";

         //
         Session session = getCurrentSession();
         Query query = session.createQuery("from HibernateUserImpl as u where u.userName like :filter order by u.userName");
         query.setString("filter", filter);
         query.setFirstResult(offset);
         query.setMaxResults(limit);
         Iterator iterator = query.iterate();
         return Tools.toSet(iterator, true);
      }
      catch (HibernateException e)
      {
         String message = "Cannot find user range [" + offset + "," + limit + "]";
         log.error(message, e);
         throw new IdentityException(message, e);
      }
   }

   public int getUserCount() throws IdentityException
   {
      try
      {
         Session session = getCurrentSession();
         Query query = session.createQuery("select count(u.key) from HibernateUserImpl as u");
         return ((Number)query.uniqueResult()).intValue();
      }
      catch (HibernateException e)
      {
         String message = "Cannot count users";
         log.error(message, e);
         throw new IdentityException(message, e);
      }
   }

   /**
    * Can be subclasses to provide testing in a non JTA environement.
    *
    * @throws IllegalStateException if no session factory is present
    */
   protected Session getCurrentSession() throws IllegalStateException
   {
      if (sessionFactory == null)
      {
         throw new IllegalStateException("No session factory");
      }
      return sessionFactory.getCurrentSession();
   }
}
