/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */

package org.jboss.test.visitor;

import java.io.InputStream;
import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.Field;
import java.net.URL;
import java.util.HashMap;
import java.util.Properties;

import org.jboss.logging.Logger;

/**
 * A TypeVisitor which builds a map of type, method and field properties
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 56504 $
 */
public class PropertiesVisitorImpl implements TypeVisitor
{
   private static Logger log = Logger.getLogger(PropertiesVisitorImpl.class);
   /** HashMap<Class, Properties> */
   private HashMap<Class, Properties> typeProperties = new HashMap<Class, Properties>();
   /** HashMap<String, Properties> */
   private HashMap<String, Properties> methodProperties = new HashMap<String, Properties>();
   /** HashMap<String, Properties> */
   private HashMap<String, Properties> fieldProperties = new HashMap<String, Properties>();

   /**
    * @return HashMap<Class, Properties>
    */
   public HashMap<Class, Properties> getTypeProperties()
   {
      return typeProperties;
   }

   /**
    * @return HashMap<String, Properties>
    */
   public HashMap<String, Properties> getMethodProperties()
   {
      return methodProperties;
   }
   
   /**
    * @return HashMap<String, Properties>
    */
   public HashMap<String, Properties> getFieldProperties()
   {
      return fieldProperties;
   }

   // Begin TypeVisitor interface

   /**
    * Look for a properties file for the type using 
    * @param type
    */
   public void visitClass(Class type)
   {
      String className = type.getName();
      int dot = className.lastIndexOf('.');
      if( dot >= 0 )
         className = className.substring(dot+1);
      String name = className + ".properties";
      loadProperties(type, type, name, typeProperties);
   }

   /**
    * Calls visitClass(i) for each ifaces[n]
    * @param ifaces
    */
   public void visitInterfaces(Class[] ifaces)
   {
      for(int n = 0; n < ifaces.length; n ++)
      {
         Class i = ifaces[n];
         visitClass(i);
      }
   }

   public void visitMethods(Method[] methods)
   {
      for(int n = 0; n < methods.length; n ++)
      {
         Method m = methods[n];
         String name = m.getName();
         Class type = m.getDeclaringClass();
         loadProperties(type, name, name, methodProperties);
      }
   }

   public void visitFields(Field[] fields)
   {
      for(int n = 0; n < fields.length; n ++)
      {
         Field f = fields[n];
         String name = f.getName();
         Class type = f.getDeclaringClass();
         loadProperties(type, name, name, fieldProperties);
      }
   }
   // End TypeVisitor interface

   /**
    * 
    * @param <T> the key type
    * @param type - the class used to load the name resource via
    *    Class.getResource(name)
    * @param key - the key into map
    * @param name - the resource name to search for
    * @param map - Map<key, Properties> to populate
    */
   protected <T> void loadProperties(Class type, T key, String name, HashMap<T, Properties> map)
   {
      URL res = type.getResource(name);
      if( res != null )
      {
         Properties props = map.get(key);
         if( props == null )
         {
            props = new Properties();
            map.put(key, props);
         }
         try
         {
            InputStream is = res.openStream();
            props.load(is);
            is.close();
         }
         catch (IOException e)
         {
            log.warn("Failed to load properties for name: "+name, e);
         }
         // FIXME - this is to ignore problems when it tries to load directories as property files?
         catch (NullPointerException e)
         {
            log.warn("Failed to load properties for name: "+name, e);
         }
      }
   }
}
