/*
 * The MIT License
 * 
 * Copyright (c) 2004-2009, Sun Microsystems, Inc., Kohsuke Kawaguchi
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package hudson.remoting.jnlp;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.remoting.EngineListener;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.logging.Logger;

import static java.util.logging.Level.INFO;
import static java.util.logging.Level.SEVERE;

/**
 * {@link EngineListener} implementation that shows GUI.
 */
public final class GuiListener implements EngineListener {
    public final MainDialog frame;

    public GuiListener() {
        GUI.setUILookAndFeel();
        frame = new MainDialog();
        frame.setVisible(true);
    }

    @Override
    public void status(final String msg) {
        status(msg,null);
    }

    @Override
    public void status(final String msg, final Throwable t) {
        SwingUtilities.invokeLater(() -> {
            frame.status(msg);
            if(t!=null)
                LOGGER.log(INFO, msg, t);
        });
    }

    @Override
    public void error(final Throwable t) {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            @SuppressFBWarnings(value = {"DM_EXIT", "INFORMATION_EXPOSURE_THROUGH_AN_ERROR_MESSAGE"}, justification = "This is an error handler for GUI, exit is valid. Used to show errors to administrator when run with GUI.")
            public void run() {
                LOGGER.log(SEVERE, t.getMessage(), t);
                StringWriter sw = new StringWriter();
                t.printStackTrace(new PrintWriter(sw));
                JOptionPane.showMessageDialog(
                    frame,sw.toString(),"Error",
                    JOptionPane.ERROR_MESSAGE);
                System.exit(-1);
            }
        });
    }

    @Override
    public void onDisconnect() {
        // discard all the menu items that might have been added by the controller.
        SwingUtilities.invokeLater(frame::resetMenuBar);
    }

    @Override
    public void onReconnect() {
    }

    private static final Logger LOGGER = Logger.getLogger(GuiListener.class.getName());
}
