/*
 * (C) Copyright 2006-2010 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     bstefanescu
 */
package org.nuxeo.connect.update.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import org.nuxeo.common.utils.FileUtils;
import org.nuxeo.common.xmap.XMap;
import org.nuxeo.connect.update.LocalPackage;
import org.nuxeo.connect.update.PackageData;
import org.nuxeo.connect.update.PackageDependency;
import org.nuxeo.connect.update.PackageException;
import org.nuxeo.connect.update.PackageType;
import org.nuxeo.connect.update.Validator;
import org.nuxeo.connect.update.Version;
import org.nuxeo.connect.update.forms.Form;
import org.nuxeo.connect.update.impl.task.InstallTask;
import org.nuxeo.connect.update.impl.task.UninstallTask;
import org.nuxeo.connect.update.impl.xml.FormsDefinition;
import org.nuxeo.connect.update.impl.xml.PackageDefinition;
import org.nuxeo.connect.update.impl.xml.TaskDefinition;
import org.nuxeo.connect.update.task.Task;

/**
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 * 
 */
public class LocalPackageImpl implements LocalPackage {

    protected String id;

    protected int state;

    protected LocalPackageData data;

    protected PackageDefinition def;

    public LocalPackageImpl(File file, int state) throws PackageException {
        this(null, file, state);
    }

    public LocalPackageImpl(ClassLoader parent, File file, int state)
            throws PackageException {
        this.state = state;
        XMap xmap = UpdateServiceImpl.getXmap();
        InputStream in = null;
        try {
            this.data = new LocalPackageData(parent, file);
            in = new FileInputStream(data.getManifest());
            def = (PackageDefinition) xmap.load(in);
        } catch (FileNotFoundException e) {
            throw new PackageException(
                    "Invalid package - no package.xml file found in package "
                            + file.getName());
        } catch (Exception e) {
            throw new PackageException(
                    "Failed to load package.xml descriptor for package "
                            + file.getName(), e);
        }
        id = def.getId();
    }

    public void setState(int state) {
        this.state = state;
    }

    public PackageData getData() {
        return data;
    }

    public File getInstallFile() throws PackageException {
        File file = data.getEntry(LocalPackage.INSTALL);
        return file.isFile() ? file : null;
    }

    public File getUninstallFile() throws PackageException {
        File file = data.getEntry(LocalPackage.UNINSTALL);
        return file.isFile() ? file : null;
    }

    public File getContentPage() {
        File file = data.getEntry(LocalPackage.CONTENT);
        return file.isFile() ? file : null;
    }

    public String getLicense() throws PackageException {
        File file = data.getEntry(LocalPackage.CONTENT);
        if (file.isFile()) {
            try {
                return FileUtils.readFile(file);
            } catch (Exception e) {
                throw new PackageException(
                        "Failed to rad license.txt file for package: "
                                + getId());
            }
        }
        return null;
    }

    public String getClassifier() {
        return def.classifier;
    }

    public String getDescription() {
        return def.description;
    }

    public String getId() {
        return id;
    }

    public String getName() {
        return def.name;
    }

    public String getTitle() {
        return def.title;
    }

    public int getState() {
        return state;
    }

    public String[] getTargetPlatforms() {
        return def.platforms;
    }

    public PackageDependency[] getDependencies() {
        return def.dependencies;
    }

    public PackageType getType() {
        return def.type;
    }

    public String getHomePage() {
        return def.homePage;
    }

    public Version getVersion() {
        return def.version;
    }

    public String getVendor() {
        return def.vendor;
    }

    public boolean isLocal() {
        return true;
    }

    public Task getInstallTask() throws PackageException {
        if (def.installer == null) {
            def.installer = new TaskDefinition(InstallTask.class.getName(),
                    false);
        } else if (def.installer.type == null) {
            def.installer.type = InstallTask.class.getName();
        }
        return getTask(def.installer);
    }

    public Task getUninstallTask() throws PackageException {
        if (def.uninstaller == null) {
            def.uninstaller = new TaskDefinition(UninstallTask.class.getName(),
                    false);
        } else if (def.uninstaller.type == null) {
            def.uninstaller.type = UninstallTask.class.getName();
        }
        return getTask(def.uninstaller);
    }

    protected Task getTask(TaskDefinition tdef) throws PackageException {
        Task task = null;
        try {
            task = (Task) data.loadClass(tdef.type).newInstance();
        } catch (Exception e) {
            throw new PackageException("Could not instatiate custom task "
                    + tdef.type + " for package " + getId(), e);
        }
        task.initialize(this, tdef.restart);
        return task;
    }

    public Validator getValidator() throws PackageException {
        if (def.validator != null) {
            try {
                return (Validator) data.loadClass(def.validator).newInstance();
            } catch (Exception e) {
                throw new PackageException(
                        "Could not instatiate custom validator "
                                + def.installer + " for package " + getId(), e);
            }
        }
        return null;
    }

    public Form[] getForms(String path) throws PackageException {
        File file = data.getEntry(path);
        if (file.isFile()) {
            FileInputStream in = null;
            try {
                in = new FileInputStream(file);
                FormsDefinition forms = (FormsDefinition) UpdateServiceImpl.getXmap().load(
                        in);
                return forms.forms;
            } catch (Exception e) {
                throw new PackageException("Failed to load forms file: " + file);
            } finally {
                if (in != null) {
                    try {
                        in.close();
                    } catch (IOException e) {
                    }
                }
            }
        }
        return null;
    }

    public Form[] getValidationForms() throws PackageException {
        return getForms(LocalPackage.VALIDATION_FORMS);
    }

    public Form[] getInstallForms() throws PackageException {
        return getForms(LocalPackage.INSTALL_FORMS);
    }

    public Form[] getUninstallForms() throws PackageException {
        return getForms(LocalPackage.UNINSTALL_FORMS);
    }
}
