/*
 * (C) Copyright 2006-2010 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     bstefanescu
 */
package org.nuxeo.connect.update.impl.task;

import java.util.Map;

import org.nuxeo.connect.update.PackageException;
import org.nuxeo.connect.update.ValidationStatus;
import org.nuxeo.connect.update.impl.xml.XmlWriter;
import org.nuxeo.connect.update.task.Task;
import org.w3c.dom.Element;

/**
 * An <b>atomic</b> command that can be executed as part of a task. Commands are
 * logged so that they can be rollbacked if needed.
 * 
 * A command is assumed to be atomic - so that if anything goes wrong inside the
 * execute method the command will not be rollbacked - only previous commands
 * are rollbacked. If you need to implement a non atomic command then you should
 * take care yourself of the partial rollback if an error occurs during the
 * execution.
 * 
 * The generic T represent the task type where this command can be executed
 * 
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 * 
 */
public interface Command {

    /**
     * The command id
     * 
     * @return
     */
    String getId();

    /**
     * Execute the command and return the inverse command that can be used to
     * rollback. Can return null if no rollback command is needed.
     * 
     * @param task
     * @pram prefs user preferences - can be null if no preferences are given
     * 
     * @throws PackageException
     */
    Command run(Task task, Map<String, String> prefs) throws PackageException;

    /**
     * Validate if the command can be safely executed on the running platform.
     * The command should check if the artifacts it needs to copy or replace are
     * consistent with the command configuration. If inconsistency is detected
     * errors or warnings must be added to the status.
     * 
     * @param status
     */
    void validate(Task task, ValidationStatus status) throws PackageException;

    /**
     * Initialize a command parameters from a DOM element. This method will be
     * called only once by the task when the commands are loaded from an XML
     * file.
     * 
     * @param element
     * @throws PackageException
     */
    public void initialize(Element element) throws PackageException;

    /**
     * Write the command as XML using the given {@link XmlWriter}. This method
     * will be called by the task to create an uninstall.xml file.
     * 
     * @param writer
     */
    public void writeTo(XmlWriter writer);

}
