/*
 * Copyright 2017-2025 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *          https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.springframework.batch.infrastructure.item.ldif.builder;

import org.jspecify.annotations.Nullable;

import org.springframework.batch.infrastructure.item.ExecutionContext;
import org.springframework.batch.infrastructure.item.ItemStreamSupport;
import org.springframework.batch.infrastructure.item.support.AbstractItemCountingItemStreamItemReader;
import org.springframework.batch.infrastructure.item.ldif.LdifReader;
import org.springframework.batch.infrastructure.item.ldif.RecordCallbackHandler;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

/**
 * Creates a fully qualified LdifReader.
 *
 * @author Glenn Renfro
 * @since 4.0
 */
public class LdifReaderBuilder {

	private @Nullable Resource resource;

	private int recordsToSkip = 0;

	private boolean strict = true;

	private @Nullable RecordCallbackHandler skippedRecordsCallback;

	private boolean saveState = true;

	private @Nullable String name;

	private int maxItemCount = Integer.MAX_VALUE;

	private int currentItemCount;

	/**
	 * Configure if the state of the {@link ItemStreamSupport} should be persisted within
	 * the {@link ExecutionContext} for restart purposes.
	 * @param saveState defaults to true
	 * @return The current instance of the builder.
	 */
	public LdifReaderBuilder saveState(boolean saveState) {
		this.saveState = saveState;

		return this;
	}

	/**
	 * The name used to calculate the key within the {@link ExecutionContext}. Required if
	 * {@link #saveState(boolean)} is set to true.
	 * @param name name of the reader instance
	 * @return The current instance of the builder.
	 * @see ItemStreamSupport#setName(String)
	 */
	public LdifReaderBuilder name(String name) {
		this.name = name;

		return this;
	}

	/**
	 * Configure the max number of items to be read.
	 * @param maxItemCount the max items to be read
	 * @return The current instance of the builder.
	 * @see AbstractItemCountingItemStreamItemReader#setMaxItemCount(int)
	 */
	public LdifReaderBuilder maxItemCount(int maxItemCount) {
		this.maxItemCount = maxItemCount;

		return this;
	}

	/**
	 * Index for the current item. Used on restarts to indicate where to start from.
	 * @param currentItemCount current index
	 * @return this instance for method chaining
	 * @see AbstractItemCountingItemStreamItemReader#setCurrentItemCount(int)
	 */
	public LdifReaderBuilder currentItemCount(int currentItemCount) {
		this.currentItemCount = currentItemCount;

		return this;
	}

	/**
	 * In strict mode the reader will throw an exception on
	 * {@link LdifReader#open(ExecutionContext)} if the input resource does not exist.
	 * @param strict true by default
	 * @return this instance for method chaining.
	 * @see LdifReader#setStrict(boolean)
	 */
	public LdifReaderBuilder strict(boolean strict) {
		this.strict = strict;

		return this;
	}

	/**
	 * {@link RecordCallbackHandler RecordCallbackHandler} implementations can be used to
	 * take action on skipped records.
	 * @param skippedRecordsCallback will be called for each one of the initial skipped
	 * lines before any items are read.
	 * @return this instance for method chaining.
	 * @see LdifReader#setSkippedRecordsCallback(RecordCallbackHandler)
	 */
	public LdifReaderBuilder skippedRecordsCallback(RecordCallbackHandler skippedRecordsCallback) {
		this.skippedRecordsCallback = skippedRecordsCallback;

		return this;
	}

	/**
	 * Public setter for the number of lines to skip at the start of a file. Can be used
	 * if the file contains a header without useful (column name) information, and without
	 * a comment delimiter at the beginning of the lines.
	 * @param recordsToSkip the number of lines to skip
	 * @return this instance for method chaining.
	 * @see LdifReader#setRecordsToSkip(int)
	 */
	public LdifReaderBuilder recordsToSkip(int recordsToSkip) {
		this.recordsToSkip = recordsToSkip;

		return this;
	}

	/**
	 * Establishes the resource that will be used as the input for the LdifReader.
	 * @param resource the resource that will be read.
	 * @return this instance for method chaining.
	 * @see LdifReader#setResource(Resource)
	 */
	public LdifReaderBuilder resource(Resource resource) {
		this.resource = resource;

		return this;
	}

	/**
	 * Returns a fully constructed {@link LdifReader}.
	 * @return a new {@link LdifReader}
	 */
	public LdifReader build() throws Exception {
		Assert.notNull(this.resource, "Resource is required.");
		if (this.saveState) {
			Assert.hasText(this.name, "A name is required when saveState is set to true");
		}
		LdifReader reader = new LdifReader(this.resource);
		reader.setRecordsToSkip(this.recordsToSkip);
		reader.setSaveState(this.saveState);
		if (this.name != null) {
			reader.setName(this.name);
		}
		reader.setCurrentItemCount(this.currentItemCount);
		reader.setMaxItemCount(this.maxItemCount);
		if (this.skippedRecordsCallback != null) {
			reader.setSkippedRecordsCallback(this.skippedRecordsCallback);
		}
		reader.setStrict(this.strict);
		reader.afterPropertiesSet();

		return reader;
	}

}
