/*
 * Copyright 2010-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package software.amazon.awssdk.http.apache.internal.impl;

import java.io.IOException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.HttpContext;
import software.amazon.awssdk.annotations.SdkInternalApi;

/**
 * An instance of {@link ConnectionManagerAwareHttpClient} that delegates all the requests to the given http client.
 */
@SdkInternalApi
public class ApacheSdkHttpClient implements ConnectionManagerAwareHttpClient {

    private final HttpClient delegate;

    private final HttpClientConnectionManager cm;

    public ApacheSdkHttpClient(final HttpClient delegate,
                               final HttpClientConnectionManager cm) {
        if (delegate == null) {
            throw new IllegalArgumentException("delegate " +
                                               "cannot be null");
        }
        if (cm == null) {
            throw new IllegalArgumentException("connection manager " +
                                               "cannot be null");
        }
        this.delegate = delegate;
        this.cm = cm;
    }

    @Override
    public HttpParams getParams() {
        return delegate.getParams();
    }

    @Override
    public ClientConnectionManager getConnectionManager() {
        return delegate.getConnectionManager();
    }

    @Override
    public HttpResponse execute(HttpUriRequest request) throws IOException {
        return delegate.execute(request);
    }

    @Override
    public HttpResponse execute(HttpUriRequest request, HttpContext context) throws IOException {
        return delegate.execute(request, context);
    }

    @Override
    public HttpResponse execute(HttpHost target, HttpRequest request) throws IOException {
        return delegate.execute(target, request);
    }

    @Override
    public HttpResponse execute(HttpHost target, HttpRequest request, HttpContext context) throws IOException {
        return delegate.execute(target, request, context);
    }

    @Override
    public <T> T execute(HttpUriRequest request, ResponseHandler<? extends T> responseHandler) throws IOException {
        return delegate.execute(request, responseHandler);
    }

    @Override
    public <T> T execute(HttpUriRequest request,
                         ResponseHandler<? extends T> responseHandler,
                         HttpContext context) throws IOException {
        return delegate.execute(request, responseHandler, context);
    }

    @Override
    public <T> T execute(HttpHost target,
                         HttpRequest request,
                         ResponseHandler<? extends T> responseHandler) throws IOException {
        return delegate.execute(target, request, responseHandler);
    }

    @Override
    public <T> T execute(HttpHost target, HttpRequest request, ResponseHandler<? extends T> responseHandler,
                         HttpContext context) throws IOException {
        return delegate.execute(target, request, responseHandler, context);
    }

    @Override
    public HttpClientConnectionManager getHttpClientConnectionManager() {
        return cm;
    }
}
