/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartQueryExecutionRequest extends AthenaRequest implements
        ToCopyableBuilder<StartQueryExecutionRequest.Builder, StartQueryExecutionRequest> {
    private static final SdkField<String> QUERY_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QueryString").getter(getter(StartQueryExecutionRequest::queryString))
            .setter(setter(Builder::queryString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryString").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(StartQueryExecutionRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<QueryExecutionContext> QUERY_EXECUTION_CONTEXT_FIELD = SdkField
            .<QueryExecutionContext> builder(MarshallingType.SDK_POJO).memberName("QueryExecutionContext")
            .getter(getter(StartQueryExecutionRequest::queryExecutionContext)).setter(setter(Builder::queryExecutionContext))
            .constructor(QueryExecutionContext::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryExecutionContext").build())
            .build();

    private static final SdkField<ResultConfiguration> RESULT_CONFIGURATION_FIELD = SdkField
            .<ResultConfiguration> builder(MarshallingType.SDK_POJO).memberName("ResultConfiguration")
            .getter(getter(StartQueryExecutionRequest::resultConfiguration)).setter(setter(Builder::resultConfiguration))
            .constructor(ResultConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResultConfiguration").build())
            .build();

    private static final SdkField<String> WORK_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkGroup").getter(getter(StartQueryExecutionRequest::workGroup)).setter(setter(Builder::workGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkGroup").build()).build();

    private static final SdkField<List<String>> EXECUTION_PARAMETERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExecutionParameters")
            .getter(getter(StartQueryExecutionRequest::executionParameters))
            .setter(setter(Builder::executionParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ResultReuseConfiguration> RESULT_REUSE_CONFIGURATION_FIELD = SdkField
            .<ResultReuseConfiguration> builder(MarshallingType.SDK_POJO).memberName("ResultReuseConfiguration")
            .getter(getter(StartQueryExecutionRequest::resultReuseConfiguration))
            .setter(setter(Builder::resultReuseConfiguration)).constructor(ResultReuseConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResultReuseConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUERY_STRING_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, QUERY_EXECUTION_CONTEXT_FIELD, RESULT_CONFIGURATION_FIELD, WORK_GROUP_FIELD,
            EXECUTION_PARAMETERS_FIELD, RESULT_REUSE_CONFIGURATION_FIELD));

    private final String queryString;

    private final String clientRequestToken;

    private final QueryExecutionContext queryExecutionContext;

    private final ResultConfiguration resultConfiguration;

    private final String workGroup;

    private final List<String> executionParameters;

    private final ResultReuseConfiguration resultReuseConfiguration;

    private StartQueryExecutionRequest(BuilderImpl builder) {
        super(builder);
        this.queryString = builder.queryString;
        this.clientRequestToken = builder.clientRequestToken;
        this.queryExecutionContext = builder.queryExecutionContext;
        this.resultConfiguration = builder.resultConfiguration;
        this.workGroup = builder.workGroup;
        this.executionParameters = builder.executionParameters;
        this.resultReuseConfiguration = builder.resultReuseConfiguration;
    }

    /**
     * <p>
     * The SQL query statements to be executed.
     * </p>
     * 
     * @return The SQL query statements to be executed.
     */
    public final String queryString() {
        return queryString;
    }

    /**
     * <p>
     * A unique case-sensitive string used to ensure the request to create the query is idempotent (executes only once).
     * If another <code>StartQueryExecution</code> request is received, the same response is returned and another query
     * is not created. An error is returned if a parameter, such as <code>QueryString</code>, has changed. A call to
     * <code>StartQueryExecution</code> that uses a previous client request token returns the same
     * <code>QueryExecutionId</code> even if the requester doesn't have permission on the tables specified in
     * <code>QueryString</code>.
     * </p>
     * <important>
     * <p>
     * This token is listed as not required because Amazon Web Services SDKs (for example the Amazon Web Services SDK
     * for Java) auto-generate the token for users. If you are not using the Amazon Web Services SDK or the Amazon Web
     * Services CLI, you must provide this token or the action will fail.
     * </p>
     * </important>
     * 
     * @return A unique case-sensitive string used to ensure the request to create the query is idempotent (executes
     *         only once). If another <code>StartQueryExecution</code> request is received, the same response is
     *         returned and another query is not created. An error is returned if a parameter, such as
     *         <code>QueryString</code>, has changed. A call to <code>StartQueryExecution</code> that uses a previous
     *         client request token returns the same <code>QueryExecutionId</code> even if the requester doesn't have
     *         permission on the tables specified in <code>QueryString</code>.</p> <important>
     *         <p>
     *         This token is listed as not required because Amazon Web Services SDKs (for example the Amazon Web
     *         Services SDK for Java) auto-generate the token for users. If you are not using the Amazon Web Services
     *         SDK or the Amazon Web Services CLI, you must provide this token or the action will fail.
     *         </p>
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The database within which the query executes.
     * </p>
     * 
     * @return The database within which the query executes.
     */
    public final QueryExecutionContext queryExecutionContext() {
        return queryExecutionContext;
    }

    /**
     * <p>
     * Specifies information about where and how to save the results of the query execution. If the query runs in a
     * workgroup, then workgroup's settings may override query settings. This affects the query results location. The
     * workgroup settings override is specified in EnforceWorkGroupConfiguration (true/false) in the
     * WorkGroupConfiguration. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     * </p>
     * 
     * @return Specifies information about where and how to save the results of the query execution. If the query runs
     *         in a workgroup, then workgroup's settings may override query settings. This affects the query results
     *         location. The workgroup settings override is specified in EnforceWorkGroupConfiguration (true/false) in
     *         the WorkGroupConfiguration. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
     */
    public final ResultConfiguration resultConfiguration() {
        return resultConfiguration;
    }

    /**
     * <p>
     * The name of the workgroup in which the query is being started.
     * </p>
     * 
     * @return The name of the workgroup in which the query is being started.
     */
    public final String workGroup() {
        return workGroup;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExecutionParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExecutionParameters() {
        return executionParameters != null && !(executionParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
     * query in the order in which the parameters occur.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExecutionParameters} method.
     * </p>
     * 
     * @return A list of values for the parameters in a query. The values are applied sequentially to the parameters in
     *         the query in the order in which the parameters occur.
     */
    public final List<String> executionParameters() {
        return executionParameters;
    }

    /**
     * <p>
     * Specifies the query result reuse behavior for the query.
     * </p>
     * 
     * @return Specifies the query result reuse behavior for the query.
     */
    public final ResultReuseConfiguration resultReuseConfiguration() {
        return resultReuseConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(queryString());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(queryExecutionContext());
        hashCode = 31 * hashCode + Objects.hashCode(resultConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(workGroup());
        hashCode = 31 * hashCode + Objects.hashCode(hasExecutionParameters() ? executionParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resultReuseConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartQueryExecutionRequest)) {
            return false;
        }
        StartQueryExecutionRequest other = (StartQueryExecutionRequest) obj;
        return Objects.equals(queryString(), other.queryString())
                && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(queryExecutionContext(), other.queryExecutionContext())
                && Objects.equals(resultConfiguration(), other.resultConfiguration())
                && Objects.equals(workGroup(), other.workGroup()) && hasExecutionParameters() == other.hasExecutionParameters()
                && Objects.equals(executionParameters(), other.executionParameters())
                && Objects.equals(resultReuseConfiguration(), other.resultReuseConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartQueryExecutionRequest").add("QueryString", queryString())
                .add("ClientRequestToken", clientRequestToken()).add("QueryExecutionContext", queryExecutionContext())
                .add("ResultConfiguration", resultConfiguration()).add("WorkGroup", workGroup())
                .add("ExecutionParameters", hasExecutionParameters() ? executionParameters() : null)
                .add("ResultReuseConfiguration", resultReuseConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QueryString":
            return Optional.ofNullable(clazz.cast(queryString()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "QueryExecutionContext":
            return Optional.ofNullable(clazz.cast(queryExecutionContext()));
        case "ResultConfiguration":
            return Optional.ofNullable(clazz.cast(resultConfiguration()));
        case "WorkGroup":
            return Optional.ofNullable(clazz.cast(workGroup()));
        case "ExecutionParameters":
            return Optional.ofNullable(clazz.cast(executionParameters()));
        case "ResultReuseConfiguration":
            return Optional.ofNullable(clazz.cast(resultReuseConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartQueryExecutionRequest, T> g) {
        return obj -> g.apply((StartQueryExecutionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends AthenaRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartQueryExecutionRequest> {
        /**
         * <p>
         * The SQL query statements to be executed.
         * </p>
         * 
         * @param queryString
         *        The SQL query statements to be executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryString(String queryString);

        /**
         * <p>
         * A unique case-sensitive string used to ensure the request to create the query is idempotent (executes only
         * once). If another <code>StartQueryExecution</code> request is received, the same response is returned and
         * another query is not created. An error is returned if a parameter, such as <code>QueryString</code>, has
         * changed. A call to <code>StartQueryExecution</code> that uses a previous client request token returns the
         * same <code>QueryExecutionId</code> even if the requester doesn't have permission on the tables specified in
         * <code>QueryString</code>.
         * </p>
         * <important>
         * <p>
         * This token is listed as not required because Amazon Web Services SDKs (for example the Amazon Web Services
         * SDK for Java) auto-generate the token for users. If you are not using the Amazon Web Services SDK or the
         * Amazon Web Services CLI, you must provide this token or the action will fail.
         * </p>
         * </important>
         * 
         * @param clientRequestToken
         *        A unique case-sensitive string used to ensure the request to create the query is idempotent (executes
         *        only once). If another <code>StartQueryExecution</code> request is received, the same response is
         *        returned and another query is not created. An error is returned if a parameter, such as
         *        <code>QueryString</code>, has changed. A call to <code>StartQueryExecution</code> that uses a previous
         *        client request token returns the same <code>QueryExecutionId</code> even if the requester doesn't have
         *        permission on the tables specified in <code>QueryString</code>.</p> <important>
         *        <p>
         *        This token is listed as not required because Amazon Web Services SDKs (for example the Amazon Web
         *        Services SDK for Java) auto-generate the token for users. If you are not using the Amazon Web Services
         *        SDK or the Amazon Web Services CLI, you must provide this token or the action will fail.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The database within which the query executes.
         * </p>
         * 
         * @param queryExecutionContext
         *        The database within which the query executes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryExecutionContext(QueryExecutionContext queryExecutionContext);

        /**
         * <p>
         * The database within which the query executes.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueryExecutionContext.Builder} avoiding
         * the need to create one manually via {@link QueryExecutionContext#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueryExecutionContext.Builder#build()} is called immediately and
         * its result is passed to {@link #queryExecutionContext(QueryExecutionContext)}.
         * 
         * @param queryExecutionContext
         *        a consumer that will call methods on {@link QueryExecutionContext.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queryExecutionContext(QueryExecutionContext)
         */
        default Builder queryExecutionContext(Consumer<QueryExecutionContext.Builder> queryExecutionContext) {
            return queryExecutionContext(QueryExecutionContext.builder().applyMutation(queryExecutionContext).build());
        }

        /**
         * <p>
         * Specifies information about where and how to save the results of the query execution. If the query runs in a
         * workgroup, then workgroup's settings may override query settings. This affects the query results location.
         * The workgroup settings override is specified in EnforceWorkGroupConfiguration (true/false) in the
         * WorkGroupConfiguration. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * 
         * @param resultConfiguration
         *        Specifies information about where and how to save the results of the query execution. If the query
         *        runs in a workgroup, then workgroup's settings may override query settings. This affects the query
         *        results location. The workgroup settings override is specified in EnforceWorkGroupConfiguration
         *        (true/false) in the WorkGroupConfiguration. See
         *        <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultConfiguration(ResultConfiguration resultConfiguration);

        /**
         * <p>
         * Specifies information about where and how to save the results of the query execution. If the query runs in a
         * workgroup, then workgroup's settings may override query settings. This affects the query results location.
         * The workgroup settings override is specified in EnforceWorkGroupConfiguration (true/false) in the
         * WorkGroupConfiguration. See <a>WorkGroupConfiguration$EnforceWorkGroupConfiguration</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResultConfiguration.Builder} avoiding the
         * need to create one manually via {@link ResultConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResultConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #resultConfiguration(ResultConfiguration)}.
         * 
         * @param resultConfiguration
         *        a consumer that will call methods on {@link ResultConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resultConfiguration(ResultConfiguration)
         */
        default Builder resultConfiguration(Consumer<ResultConfiguration.Builder> resultConfiguration) {
            return resultConfiguration(ResultConfiguration.builder().applyMutation(resultConfiguration).build());
        }

        /**
         * <p>
         * The name of the workgroup in which the query is being started.
         * </p>
         * 
         * @param workGroup
         *        The name of the workgroup in which the query is being started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workGroup(String workGroup);

        /**
         * <p>
         * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
         * query in the order in which the parameters occur.
         * </p>
         * 
         * @param executionParameters
         *        A list of values for the parameters in a query. The values are applied sequentially to the parameters
         *        in the query in the order in which the parameters occur.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionParameters(Collection<String> executionParameters);

        /**
         * <p>
         * A list of values for the parameters in a query. The values are applied sequentially to the parameters in the
         * query in the order in which the parameters occur.
         * </p>
         * 
         * @param executionParameters
         *        A list of values for the parameters in a query. The values are applied sequentially to the parameters
         *        in the query in the order in which the parameters occur.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionParameters(String... executionParameters);

        /**
         * <p>
         * Specifies the query result reuse behavior for the query.
         * </p>
         * 
         * @param resultReuseConfiguration
         *        Specifies the query result reuse behavior for the query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultReuseConfiguration(ResultReuseConfiguration resultReuseConfiguration);

        /**
         * <p>
         * Specifies the query result reuse behavior for the query.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResultReuseConfiguration.Builder}
         * avoiding the need to create one manually via {@link ResultReuseConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResultReuseConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #resultReuseConfiguration(ResultReuseConfiguration)}.
         * 
         * @param resultReuseConfiguration
         *        a consumer that will call methods on {@link ResultReuseConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resultReuseConfiguration(ResultReuseConfiguration)
         */
        default Builder resultReuseConfiguration(Consumer<ResultReuseConfiguration.Builder> resultReuseConfiguration) {
            return resultReuseConfiguration(ResultReuseConfiguration.builder().applyMutation(resultReuseConfiguration).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends AthenaRequest.BuilderImpl implements Builder {
        private String queryString;

        private String clientRequestToken;

        private QueryExecutionContext queryExecutionContext;

        private ResultConfiguration resultConfiguration;

        private String workGroup;

        private List<String> executionParameters = DefaultSdkAutoConstructList.getInstance();

        private ResultReuseConfiguration resultReuseConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(StartQueryExecutionRequest model) {
            super(model);
            queryString(model.queryString);
            clientRequestToken(model.clientRequestToken);
            queryExecutionContext(model.queryExecutionContext);
            resultConfiguration(model.resultConfiguration);
            workGroup(model.workGroup);
            executionParameters(model.executionParameters);
            resultReuseConfiguration(model.resultReuseConfiguration);
        }

        public final String getQueryString() {
            return queryString;
        }

        public final void setQueryString(String queryString) {
            this.queryString = queryString;
        }

        @Override
        public final Builder queryString(String queryString) {
            this.queryString = queryString;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final QueryExecutionContext.Builder getQueryExecutionContext() {
            return queryExecutionContext != null ? queryExecutionContext.toBuilder() : null;
        }

        public final void setQueryExecutionContext(QueryExecutionContext.BuilderImpl queryExecutionContext) {
            this.queryExecutionContext = queryExecutionContext != null ? queryExecutionContext.build() : null;
        }

        @Override
        public final Builder queryExecutionContext(QueryExecutionContext queryExecutionContext) {
            this.queryExecutionContext = queryExecutionContext;
            return this;
        }

        public final ResultConfiguration.Builder getResultConfiguration() {
            return resultConfiguration != null ? resultConfiguration.toBuilder() : null;
        }

        public final void setResultConfiguration(ResultConfiguration.BuilderImpl resultConfiguration) {
            this.resultConfiguration = resultConfiguration != null ? resultConfiguration.build() : null;
        }

        @Override
        public final Builder resultConfiguration(ResultConfiguration resultConfiguration) {
            this.resultConfiguration = resultConfiguration;
            return this;
        }

        public final String getWorkGroup() {
            return workGroup;
        }

        public final void setWorkGroup(String workGroup) {
            this.workGroup = workGroup;
        }

        @Override
        public final Builder workGroup(String workGroup) {
            this.workGroup = workGroup;
            return this;
        }

        public final Collection<String> getExecutionParameters() {
            if (executionParameters instanceof SdkAutoConstructList) {
                return null;
            }
            return executionParameters;
        }

        public final void setExecutionParameters(Collection<String> executionParameters) {
            this.executionParameters = ExecutionParametersCopier.copy(executionParameters);
        }

        @Override
        public final Builder executionParameters(Collection<String> executionParameters) {
            this.executionParameters = ExecutionParametersCopier.copy(executionParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder executionParameters(String... executionParameters) {
            executionParameters(Arrays.asList(executionParameters));
            return this;
        }

        public final ResultReuseConfiguration.Builder getResultReuseConfiguration() {
            return resultReuseConfiguration != null ? resultReuseConfiguration.toBuilder() : null;
        }

        public final void setResultReuseConfiguration(ResultReuseConfiguration.BuilderImpl resultReuseConfiguration) {
            this.resultReuseConfiguration = resultReuseConfiguration != null ? resultReuseConfiguration.build() : null;
        }

        @Override
        public final Builder resultReuseConfiguration(ResultReuseConfiguration resultReuseConfiguration) {
            this.resultReuseConfiguration = resultReuseConfiguration;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartQueryExecutionRequest build() {
            return new StartQueryExecutionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
