/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.health.model.ConcurrentModificationException;
import software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse;
import software.amazon.awssdk.services.health.model.DescribeEntityAggregatesRequest;
import software.amazon.awssdk.services.health.model.DescribeEntityAggregatesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest;
import software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsRequest;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsResponse;
import software.amazon.awssdk.services.health.model.DescribeEventTypesRequest;
import software.amazon.awssdk.services.health.model.DescribeEventTypesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DescribeEventsRequest;
import software.amazon.awssdk.services.health.model.DescribeEventsResponse;
import software.amazon.awssdk.services.health.model.DescribeHealthServiceStatusForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeHealthServiceStatusForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DisableHealthServiceAccessForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DisableHealthServiceAccessForOrganizationResponse;
import software.amazon.awssdk.services.health.model.EnableHealthServiceAccessForOrganizationRequest;
import software.amazon.awssdk.services.health.model.EnableHealthServiceAccessForOrganizationResponse;
import software.amazon.awssdk.services.health.model.HealthException;
import software.amazon.awssdk.services.health.model.HealthRequest;
import software.amazon.awssdk.services.health.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.health.model.UnsupportedLocaleException;
import software.amazon.awssdk.services.health.paginators.DescribeAffectedAccountsForOrganizationIterable;
import software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesForOrganizationIterable;
import software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesIterable;
import software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesIterable;
import software.amazon.awssdk.services.health.paginators.DescribeEventTypesIterable;
import software.amazon.awssdk.services.health.paginators.DescribeEventsForOrganizationIterable;
import software.amazon.awssdk.services.health.paginators.DescribeEventsIterable;
import software.amazon.awssdk.services.health.transform.DescribeAffectedAccountsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeAffectedEntitiesForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeAffectedEntitiesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEntityAggregatesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventAggregatesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventDetailsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventDetailsRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventTypesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventsRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeHealthServiceStatusForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DisableHealthServiceAccessForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.EnableHealthServiceAccessForOrganizationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link HealthClient}.
 *
 * @see HealthClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultHealthClient implements HealthClient {
    private static final Logger log = Logger.loggerFor(DefaultHealthClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final HealthServiceClientConfiguration serviceClientConfiguration;

    protected DefaultHealthClient(HealthServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns a list of accounts in the organization from Organizations that are affected by the provided event. For
     * more information about the different types of Health events, see <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html">Event</a>.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable Health to work with Organizations. To do this, call the
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     * >EnableHealthServiceAccessForOrganization</a> operation from your organization's management account.
     * </p>
     * <note>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </note>
     *
     * @param describeAffectedAccountsForOrganizationRequest
     * @return Result of the DescribeAffectedAccountsForOrganization operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeAffectedAccountsForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedAccountsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAffectedAccountsForOrganizationResponse describeAffectedAccountsForOrganization(
            DescribeAffectedAccountsForOrganizationRequest describeAffectedAccountsForOrganizationRequest)
            throws InvalidPaginationTokenException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAffectedAccountsForOrganizationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeAffectedAccountsForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAffectedAccountsForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAffectedAccountsForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAffectedAccountsForOrganizationRequest, DescribeAffectedAccountsForOrganizationResponse>()
                            .withOperationName("DescribeAffectedAccountsForOrganization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAffectedAccountsForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAffectedAccountsForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of accounts in the organization from Organizations that are affected by the provided event. For
     * more information about the different types of Health events, see <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html">Event</a>.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable Health to work with Organizations. To do this, call the
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     * >EnableHealthServiceAccessForOrganization</a> operation from your organization's management account.
     * </p>
     * <note>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedAccountsForOrganization(software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedAccountsForOrganizationIterable responses = client.describeAffectedAccountsForOrganizationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeAffectedAccountsForOrganizationIterable responses = client
     *             .describeAffectedAccountsForOrganizationPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedAccountsForOrganizationIterable responses = client.describeAffectedAccountsForOrganizationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedAccountsForOrganization(software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAffectedAccountsForOrganizationRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeAffectedAccountsForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedAccountsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAffectedAccountsForOrganizationIterable describeAffectedAccountsForOrganizationPaginator(
            DescribeAffectedAccountsForOrganizationRequest describeAffectedAccountsForOrganizationRequest)
            throws InvalidPaginationTokenException, AwsServiceException, SdkClientException, HealthException {
        return new DescribeAffectedAccountsForOrganizationIterable(this,
                applyPaginatorUserAgent(describeAffectedAccountsForOrganizationRequest));
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the Amazon Web Service. Events that have impact beyond that of the affected entities, or
     * where the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </li>
     * <li>
     * <p>
     * This operation supports resource-level permissions. You can use this operation to allow or deny access to
     * specific Health events. For more information, see <a href=
     * "https://docs.aws.amazon.com/health/latest/ug/security_iam_id-based-policy-examples.html#resource-action-based-conditions"
     * >Resource- and action-based conditions</a> in the <i>Health User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeAffectedEntitiesRequest
     * @return Result of the DescribeAffectedEntities operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeAffectedEntities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAffectedEntitiesResponse describeAffectedEntities(
            DescribeAffectedEntitiesRequest describeAffectedEntitiesRequest) throws InvalidPaginationTokenException,
            UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAffectedEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAffectedEntitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAffectedEntitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAffectedEntities");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAffectedEntitiesRequest, DescribeAffectedEntitiesResponse>()
                            .withOperationName("DescribeAffectedEntities").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAffectedEntitiesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAffectedEntitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the Amazon Web Service. Events that have impact beyond that of the affected entities, or
     * where the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </li>
     * <li>
     * <p>
     * This operation supports resource-level permissions. You can use this operation to allow or deny access to
     * specific Health events. For more information, see <a href=
     * "https://docs.aws.amazon.com/health/latest/ug/security_iam_id-based-policy-examples.html#resource-action-based-conditions"
     * >Resource- and action-based conditions</a> in the <i>Health User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesIterable responses = client.describeAffectedEntitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesIterable responses = client
     *             .describeAffectedEntitiesPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesIterable responses = client.describeAffectedEntitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeAffectedEntities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAffectedEntitiesIterable describeAffectedEntitiesPaginator(
            DescribeAffectedEntitiesRequest describeAffectedEntitiesRequest) throws InvalidPaginationTokenException,
            UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        return new DescribeAffectedEntitiesIterable(this, applyPaginatorUserAgent(describeAffectedEntitiesRequest));
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by one or more events for one or more accounts in your
     * organization in Organizations, based on the filter criteria. Entities can refer to individual customer resources,
     * groups of customer resources, or any other construct, depending on the Amazon Web Service.
     * </p>
     * <p>
     * At least one event Amazon Resource Name (ARN) and account ID are required.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable Health to work with Organizations. To do this, call the
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     * >EnableHealthServiceAccessForOrganization</a> operation from your organization's management account.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </li>
     * <li>
     * <p>
     * This operation doesn't support resource-level permissions. You can't use this operation to allow or deny access
     * to specific Health events. For more information, see <a href=
     * "https://docs.aws.amazon.com/health/latest/ug/security_iam_id-based-policy-examples.html#resource-action-based-conditions"
     * >Resource- and action-based conditions</a> in the <i>Health User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeAffectedEntitiesForOrganizationRequest
     * @return Result of the DescribeAffectedEntitiesForOrganization operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeAffectedEntitiesForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntitiesForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAffectedEntitiesForOrganizationResponse describeAffectedEntitiesForOrganization(
            DescribeAffectedEntitiesForOrganizationRequest describeAffectedEntitiesForOrganizationRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAffectedEntitiesForOrganizationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeAffectedEntitiesForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAffectedEntitiesForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAffectedEntitiesForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAffectedEntitiesForOrganizationRequest, DescribeAffectedEntitiesForOrganizationResponse>()
                            .withOperationName("DescribeAffectedEntitiesForOrganization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAffectedEntitiesForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAffectedEntitiesForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by one or more events for one or more accounts in your
     * organization in Organizations, based on the filter criteria. Entities can refer to individual customer resources,
     * groups of customer resources, or any other construct, depending on the Amazon Web Service.
     * </p>
     * <p>
     * At least one event Amazon Resource Name (ARN) and account ID are required.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable Health to work with Organizations. To do this, call the
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     * >EnableHealthServiceAccessForOrganization</a> operation from your organization's management account.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </li>
     * <li>
     * <p>
     * This operation doesn't support resource-level permissions. You can't use this operation to allow or deny access
     * to specific Health events. For more information, see <a href=
     * "https://docs.aws.amazon.com/health/latest/ug/security_iam_id-based-policy-examples.html#resource-action-based-conditions"
     * >Resource- and action-based conditions</a> in the <i>Health User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedEntitiesForOrganization(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesForOrganizationIterable responses = client.describeAffectedEntitiesForOrganizationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesForOrganizationIterable responses = client
     *             .describeAffectedEntitiesForOrganizationPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesForOrganizationIterable responses = client.describeAffectedEntitiesForOrganizationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedEntitiesForOrganization(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAffectedEntitiesForOrganizationRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeAffectedEntitiesForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntitiesForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAffectedEntitiesForOrganizationIterable describeAffectedEntitiesForOrganizationPaginator(
            DescribeAffectedEntitiesForOrganizationRequest describeAffectedEntitiesForOrganizationRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        return new DescribeAffectedEntitiesForOrganizationIterable(this,
                applyPaginatorUserAgent(describeAffectedEntitiesForOrganizationRequest));
    }

    /**
     * <p>
     * Returns the number of entities that are affected by each of the specified events.
     * </p>
     *
     * @param describeEntityAggregatesRequest
     * @return Result of the DescribeEntityAggregates operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEntityAggregates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEntityAggregates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEntityAggregatesResponse describeEntityAggregates(
            DescribeEntityAggregatesRequest describeEntityAggregatesRequest) throws AwsServiceException, SdkClientException,
            HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEntityAggregatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEntityAggregatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEntityAggregatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEntityAggregates");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEntityAggregatesRequest, DescribeEntityAggregatesResponse>()
                            .withOperationName("DescribeEntityAggregates").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeEntityAggregatesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEntityAggregatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     * <note>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </note>
     *
     * @param describeEventAggregatesRequest
     * @return Result of the DescribeEventAggregates operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventAggregates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEventAggregatesResponse describeEventAggregates(DescribeEventAggregatesRequest describeEventAggregatesRequest)
            throws InvalidPaginationTokenException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventAggregatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventAggregatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventAggregatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventAggregates");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEventAggregatesRequest, DescribeEventAggregatesResponse>()
                            .withOperationName("DescribeEventAggregates").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeEventAggregatesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEventAggregatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     * <note>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesIterable responses = client.describeEventAggregatesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesIterable responses = client
     *             .describeEventAggregatesPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesIterable responses = client.describeEventAggregatesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventAggregatesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventAggregates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEventAggregatesIterable describeEventAggregatesPaginator(
            DescribeEventAggregatesRequest describeEventAggregatesRequest) throws InvalidPaginationTokenException,
            AwsServiceException, SdkClientException, HealthException {
        return new DescribeEventAggregatesIterable(this, applyPaginatorUserAgent(describeEventAggregatesRequest));
    }

    /**
     * <p>
     * Returns detailed information about one or more specified events. Information includes standard event data (Amazon
     * Web Services Region, service, and so on, as returned by <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html">DescribeEvents</a>), a
     * detailed event description, and possible additional metadata that depends upon the nature of the event. Affected
     * entities are not included. To retrieve the entities, use the <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html"
     * >DescribeAffectedEntities</a> operation.
     * </p>
     * <p>
     * If a specified event can't be retrieved, an error message is returned for that event.
     * </p>
     * <note>
     * <p>
     * This operation supports resource-level permissions. You can use this operation to allow or deny access to
     * specific Health events. For more information, see <a href=
     * "https://docs.aws.amazon.com/health/latest/ug/security_iam_id-based-policy-examples.html#resource-action-based-conditions"
     * >Resource- and action-based conditions</a> in the <i>Health User Guide</i>.
     * </p>
     * </note>
     *
     * @param describeEventDetailsRequest
     * @return Result of the DescribeEventDetails operation returned by the service.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventDetails" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeEventDetailsResponse describeEventDetails(DescribeEventDetailsRequest describeEventDetailsRequest)
            throws UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventDetails");

            return clientHandler.execute(new ClientExecutionParams<DescribeEventDetailsRequest, DescribeEventDetailsResponse>()
                    .withOperationName("DescribeEventDetails").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEventDetailsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEventDetailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about one or more specified events for one or more Amazon Web Services accounts in
     * your organization. This information includes standard event data (such as the Amazon Web Services Region and
     * service), an event description, and (depending on the event) possible metadata. This operation doesn't return
     * affected entities, such as the resources related to the event. To return affected entities, use the <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html"
     * >DescribeAffectedEntitiesForOrganization</a> operation.
     * </p>
     * <note>
     * <p>
     * Before you can call this operation, you must first enable Health to work with Organizations. To do this, call the
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     * >EnableHealthServiceAccessForOrganization</a> operation from your organization's management account.
     * </p>
     * </note>
     * <p>
     * When you call the <code>DescribeEventDetailsForOrganization</code> operation, specify the
     * <code>organizationEventDetailFilters</code> object in the request. Depending on the Health event type, note the
     * following differences:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To return event details for a public event, you must specify a null value for the <code>awsAccountId</code>
     * parameter. If you specify an account ID for a public event, Health returns an error message because public events
     * aren't specific to an account.
     * </p>
     * </li>
     * <li>
     * <p>
     * To return event details for an event that is specific to an account in your organization, you must specify the
     * <code>awsAccountId</code> parameter in the request. If you don't specify an account ID, Health returns an error
     * message because the event is specific to an account in your organization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html">Event</a>.
     * </p>
     * <note>
     * <p>
     * This operation doesn't support resource-level permissions. You can't use this operation to allow or deny access
     * to specific Health events. For more information, see <a href=
     * "https://docs.aws.amazon.com/health/latest/ug/security_iam_id-based-policy-examples.html#resource-action-based-conditions"
     * >Resource- and action-based conditions</a> in the <i>Health User Guide</i>.
     * </p>
     * </note>
     *
     * @param describeEventDetailsForOrganizationRequest
     * @return Result of the DescribeEventDetailsForOrganization operation returned by the service.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventDetailsForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventDetailsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEventDetailsForOrganizationResponse describeEventDetailsForOrganization(
            DescribeEventDetailsForOrganizationRequest describeEventDetailsForOrganizationRequest)
            throws UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventDetailsForOrganizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventDetailsForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeEventDetailsForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventDetailsForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEventDetailsForOrganizationRequest, DescribeEventDetailsForOrganizationResponse>()
                            .withOperationName("DescribeEventDetailsForOrganization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeEventDetailsForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEventDetailsForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. You can use this API operation to find
     * information about the Health event, such as the category, Amazon Web Service, and event code. The metadata for
     * each event appears in the <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EventType.html">EventType</a> object.
     * </p>
     * <p>
     * If you don't specify a filter criteria, the API operation returns all event types, in no particular order.
     * </p>
     * <note>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </note>
     *
     * @param describeEventTypesRequest
     * @return Result of the DescribeEventTypes operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventTypesResponse describeEventTypes(DescribeEventTypesRequest describeEventTypesRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventTypes");

            return clientHandler.execute(new ClientExecutionParams<DescribeEventTypesRequest, DescribeEventTypesResponse>()
                    .withOperationName("DescribeEventTypes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEventTypesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEventTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. You can use this API operation to find
     * information about the Health event, such as the category, Amazon Web Service, and event code. The metadata for
     * each event appears in the <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EventType.html">EventType</a> object.
     * </p>
     * <p>
     * If you don't specify a filter criteria, the API operation returns all event types, in no particular order.
     * </p>
     * <note>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesIterable responses = client.describeEventTypesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeEventTypesIterable responses = client
     *             .describeEventTypesPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeEventTypesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesIterable responses = client.describeEventTypesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventTypesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventTypesIterable describeEventTypesPaginator(DescribeEventTypesRequest describeEventTypesRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        return new DescribeEventTypesIterable(this, applyPaginatorUserAgent(describeEventTypesRequest));
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html"
     * >DescribeEventDetails</a> and <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html"
     * >DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent event.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * When you call the <code>DescribeEvents</code> operation and specify an entity for the <code>entityValues</code>
     * parameter, Health might return public events that aren't specific to that resource. For example, if you call
     * <code>DescribeEvents</code> and specify an ID for an Amazon Elastic Compute Cloud (Amazon EC2) instance, Health
     * might return events that aren't specific to that resource or service. To get events that are specific to a
     * service, use the <code>services</code> parameter in the <code>filter</code> object. For more information, see <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html">Event</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param describeEventsRequest
     * @return Result of the DescribeEvents operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventsResponse describeEvents(DescribeEventsRequest describeEventsRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEvents");

            return clientHandler.execute(new ClientExecutionParams<DescribeEventsRequest, DescribeEventsResponse>()
                    .withOperationName("DescribeEvents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html"
     * >DescribeEventDetails</a> and <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html"
     * >DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent event.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * When you call the <code>DescribeEvents</code> operation and specify an entity for the <code>entityValues</code>
     * parameter, Health might return public events that aren't specific to that resource. For example, if you call
     * <code>DescribeEvents</code> and specify an ID for an Amazon Elastic Compute Cloud (Amazon EC2) instance, Health
     * might return events that aren't specific to that resource or service. To get events that are specific to a
     * service, use the <code>services</code> parameter in the <code>filter</code> object. For more information, see <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html">Event</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </li>
     * </ul>
     * </note><br/>
     * <p>
     * This is a variant of {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsIterable responses = client.describeEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeEventsIterable responses = client.describeEventsPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsIterable responses = client.describeEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)} operation.</b>
     * </p>
     *
     * @param describeEventsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventsIterable describeEventsPaginator(DescribeEventsRequest describeEventsRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        return new DescribeEventsIterable(this, applyPaginatorUserAgent(describeEventsRequest));
    }

    /**
     * <p>
     * Returns information about events across your organization in Organizations. You can use the<code>filters</code>
     * parameter to specify the events that you want to return. Events are returned in a summary form and don't include
     * the affected accounts, detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html">
     * DescribeAffectedAccountsForOrganization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html">
     * DescribeEventDetailsForOrganization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html">
     * DescribeAffectedEntitiesForOrganization</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify a <code>filter</code>, the <code>DescribeEventsForOrganizations</code> returns all events
     * across your organization. Results are sorted by <code>lastModifiedTime</code>, starting with the most recent
     * event.
     * </p>
     * <p>
     * For more information about the different types of Health events, see <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html">Event</a>.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable Health to work with Organizations. To do this, call the
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     * >EnableHealthServiceAccessForOrganization</a> operation from your organization's management account.
     * </p>
     * <note>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </note>
     *
     * @param describeEventsForOrganizationRequest
     * @return Result of the DescribeEventsForOrganization operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventsForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEventsForOrganizationResponse describeEventsForOrganization(
            DescribeEventsForOrganizationRequest describeEventsForOrganizationRequest) throws InvalidPaginationTokenException,
            UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventsForOrganizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventsForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeEventsForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventsForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEventsForOrganizationRequest, DescribeEventsForOrganizationResponse>()
                            .withOperationName("DescribeEventsForOrganization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeEventsForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEventsForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about events across your organization in Organizations. You can use the<code>filters</code>
     * parameter to specify the events that you want to return. Events are returned in a summary form and don't include
     * the affected accounts, detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html">
     * DescribeAffectedAccountsForOrganization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html">
     * DescribeEventDetailsForOrganization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html">
     * DescribeAffectedEntitiesForOrganization</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify a <code>filter</code>, the <code>DescribeEventsForOrganizations</code> returns all events
     * across your organization. Results are sorted by <code>lastModifiedTime</code>, starting with the most recent
     * event.
     * </p>
     * <p>
     * For more information about the different types of Health events, see <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html">Event</a>.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable Health to work with Organizations. To do this, call the
     * <a
     * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     * >EnableHealthServiceAccessForOrganization</a> operation from your organization's management account.
     * </p>
     * <note>
     * <p>
     * This API operation uses pagination. Specify the <code>nextToken</code> parameter in the next request to return
     * more results.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeEventsForOrganization(software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsForOrganizationIterable responses = client.describeEventsForOrganizationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeEventsForOrganizationIterable responses = client
     *             .describeEventsForOrganizationPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsForOrganizationIterable responses = client.describeEventsForOrganizationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventsForOrganization(software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventsForOrganizationRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventsForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEventsForOrganizationIterable describeEventsForOrganizationPaginator(
            DescribeEventsForOrganizationRequest describeEventsForOrganizationRequest) throws InvalidPaginationTokenException,
            UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        return new DescribeEventsForOrganizationIterable(this, applyPaginatorUserAgent(describeEventsForOrganizationRequest));
    }

    /**
     * <p>
     * This operation provides status information on enabling or disabling Health to work with your organization. To
     * call this operation, you must use the organization's management account.
     * </p>
     *
     * @param describeHealthServiceStatusForOrganizationRequest
     * @return Result of the DescribeHealthServiceStatusForOrganization operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeHealthServiceStatusForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeHealthServiceStatusForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeHealthServiceStatusForOrganizationResponse describeHealthServiceStatusForOrganization(
            DescribeHealthServiceStatusForOrganizationRequest describeHealthServiceStatusForOrganizationRequest)
            throws AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeHealthServiceStatusForOrganizationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeHealthServiceStatusForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeHealthServiceStatusForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHealthServiceStatusForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeHealthServiceStatusForOrganizationRequest, DescribeHealthServiceStatusForOrganizationResponse>()
                            .withOperationName("DescribeHealthServiceStatusForOrganization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeHealthServiceStatusForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeHealthServiceStatusForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables Health from working with Organizations. To call this operation, you must sign in to the organization's
     * management account. For more information, see <a
     * href="https://docs.aws.amazon.com/health/latest/ug/aggregate-events.html">Aggregating Health events</a> in the
     * <i>Health User Guide</i>.
     * </p>
     * <p>
     * This operation doesn't remove the service-linked role from the management account in your organization. You must
     * use the IAM console, API, or Command Line Interface (CLI) to remove the service-linked role. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/using-service-linked-roles.html#delete-service-linked-role"
     * >Deleting a Service-Linked Role</a> in the <i>IAM User Guide</i>.
     * </p>
     * <note>
     * <p>
     * You can also disable the organizational feature by using the Organizations <a
     * href="https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisableAWSServiceAccess.html"
     * >DisableAWSServiceAccess</a> API operation. After you call this operation, Health stops aggregating events for
     * all other Amazon Web Services accounts in your organization. If you call the Health API operations for
     * organizational view, Health returns an error. Health continues to aggregate health events for your Amazon Web
     * Services account.
     * </p>
     * </note>
     *
     * @param disableHealthServiceAccessForOrganizationRequest
     * @return Result of the DisableHealthServiceAccessForOrganization operation returned by the service.
     * @throws ConcurrentModificationException
     *         <a href=
     *         "https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     *         >EnableHealthServiceAccessForOrganization</a> is already in progress. Wait for the action to complete
     *         before trying again. To get the current status, use the <a href=
     *         "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html"
     *         >DescribeHealthServiceStatusForOrganization</a> operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DisableHealthServiceAccessForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DisableHealthServiceAccessForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableHealthServiceAccessForOrganizationResponse disableHealthServiceAccessForOrganization(
            DisableHealthServiceAccessForOrganizationRequest disableHealthServiceAccessForOrganizationRequest)
            throws ConcurrentModificationException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableHealthServiceAccessForOrganizationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisableHealthServiceAccessForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableHealthServiceAccessForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableHealthServiceAccessForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<DisableHealthServiceAccessForOrganizationRequest, DisableHealthServiceAccessForOrganizationResponse>()
                            .withOperationName("DisableHealthServiceAccessForOrganization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(disableHealthServiceAccessForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisableHealthServiceAccessForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables Health to work with Organizations. You can use the organizational view feature to aggregate events from
     * all Amazon Web Services accounts in your organization in a centralized location.
     * </p>
     * <p>
     * This operation also creates a service-linked role for the management account in the organization.
     * </p>
     * <note>
     * <p>
     * To call this operation, you must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must have a Business, Enterprise On-Ramp, or Enterprise Support plan from <a
     * href="http://aws.amazon.com/premiumsupport/">Amazon Web Services Support</a> to use the Health API. If you call
     * the Health API from an Amazon Web Services account that doesn't have a Business, Enterprise On-Ramp, or
     * Enterprise Support plan, you receive a <code>SubscriptionRequiredException</code> error.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must have permission to call this operation from the organization's management account. For example IAM
     * policies, see <a
     * href="https://docs.aws.amazon.com/health/latest/ug/security_iam_id-based-policy-examples.html">Health
     * identity-based policy examples</a>.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * If you don't have the required support plan, you can instead use the Health console to enable the organizational
     * view feature. For more information, see <a
     * href="https://docs.aws.amazon.com/health/latest/ug/aggregate-events.html">Aggregating Health events</a> in the
     * <i>Health User Guide</i>.
     * </p>
     *
     * @param enableHealthServiceAccessForOrganizationRequest
     * @return Result of the EnableHealthServiceAccessForOrganization operation returned by the service.
     * @throws ConcurrentModificationException
     *         <a href=
     *         "https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html"
     *         >EnableHealthServiceAccessForOrganization</a> is already in progress. Wait for the action to complete
     *         before trying again. To get the current status, use the <a href=
     *         "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html"
     *         >DescribeHealthServiceStatusForOrganization</a> operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.EnableHealthServiceAccessForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/EnableHealthServiceAccessForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableHealthServiceAccessForOrganizationResponse enableHealthServiceAccessForOrganization(
            EnableHealthServiceAccessForOrganizationRequest enableHealthServiceAccessForOrganizationRequest)
            throws ConcurrentModificationException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableHealthServiceAccessForOrganizationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, EnableHealthServiceAccessForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableHealthServiceAccessForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Health");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableHealthServiceAccessForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableHealthServiceAccessForOrganizationRequest, EnableHealthServiceAccessForOrganizationResponse>()
                            .withOperationName("EnableHealthServiceAccessForOrganization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(enableHealthServiceAccessForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableHealthServiceAccessForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends HealthRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(HealthException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedLocale")
                                .exceptionBuilderSupplier(UnsupportedLocaleException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationToken")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).httpStatusCode(400).build());
    }

    @Override
    public final HealthServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
